/*
 * Copyright (c) 2009-2020, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Intel Corporation nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __KASUMI_TEST_VECTORS_H__
#define __KASUMI_TEST_VECTORS_H__

#define MAX_DATA_LEN (2500)
#define MAX_KEY_LEN (32)
#define MAX_IV_LEN (16)
#define NUM_SUPPORTED_BUFFERS (16)
#define MAX_SIZE_IN_BYTES_1024 (1024)
#define MAX_BIT_BUFFERS (6)

typedef struct cipher_test_vector_s {
        uint32_t dataLenInBytes;
        uint32_t keyLenInBytes;
        uint32_t ivLenInBytes;
        uint8_t plaintext[MAX_DATA_LEN];
        uint8_t ciphertext[MAX_DATA_LEN];
        uint8_t key[MAX_KEY_LEN];
        uint8_t iv[MAX_IV_LEN];
} cipher_test_vector_t;
typedef struct cipherbit_test_vector_s {
        uint32_t LenInBits;
        uint32_t keyLenInBytes;
        uint32_t ivLenInBytes;
        uint8_t plaintext[MAX_DATA_LEN];
        uint8_t ciphertext[MAX_DATA_LEN];
        uint8_t key[MAX_KEY_LEN];
        uint8_t iv[MAX_IV_LEN];
} cipherbit_test_vector_t;
typedef struct cipherbit_test_linear_vector_s {
        uint32_t LenInBits[MAX_BIT_BUFFERS];
        uint32_t keyLenInBytes;
        uint32_t ivLenInBytes;
        uint8_t plaintext[MAX_DATA_LEN];
        uint8_t ciphertext[MAX_DATA_LEN];
        uint8_t key[MAX_BIT_BUFFERS][MAX_KEY_LEN];
        uint8_t iv[MAX_BIT_BUFFERS][MAX_IV_LEN];
} cipherbit_test_linear_vector_t;

typedef struct hash_test_vector_s {
        uint8_t input[MAX_DATA_LEN];
        uint32_t lengthInBytes;
        uint8_t key[MAX_DATA_LEN];
        uint32_t keyLenInBytes;
        uint8_t exp_out[KASUMI_BLOCK_SIZE];
} hash_test_vector_t;

typedef struct hash_iv_test_vector_s {
        uint8_t input[MAX_DATA_LEN];
        uint32_t lengthInBits;
        uint8_t key[MAX_DATA_LEN];
        uint32_t keyLenInBytes;
        uint8_t exp_out[KASUMI_BLOCK_SIZE];
        uint8_t iv[MAX_DATA_LEN];
        uint32_t ivLenInBytes;
        uint32_t direction;
} hash_iv_test_vector_t;

static const cipherbit_test_linear_vector_t kasumi_f8_linear_bitvectors = {

    {798, 510, 253, 120, 837},
    16,
    8,
    {0x7e, 0xc6, 0x12, 0x72, 0x74, 0x3b, 0xf1, 0x61, 0x47,
     0x26, 0x44, 0x6a, 0x6c, 0x38, 0xce, 0xd1, 0x66, 0xf6,
     0xca, 0x76, 0xeb, 0x54, 0x30, 0x4,  0x42, 0x86, 0x34,
     0x6c, 0xef, 0x13, 0xf,  0x92, 0x92, 0x2b, 0x3,  0x45,
     0xd,  0x3a, 0x99, 0x75, 0xe5, 0xbd, 0x2e, 0xa0, 0xeb,
     0x55, 0xad, 0x8e, 0x1b, 0x19, 0x9e, 0x3e, 0xc4, 0x31,
     0x60, 0x20, 0xe9, 0xa1, 0xb2, 0x85, 0xe7, 0x62, 0x79,
     0x53, 0x59, 0xb7, 0xbd, 0xfd, 0x39, 0xbe, 0xf4, 0xb2,
     0x48, 0x45, 0x83, 0xd5, 0xaf, 0xe0, 0x82, 0xae, 0xe6,
     0x38, 0xbf, 0x5f, 0xd5, 0xa6, 0x6,  0x19, 0x39, 0x1,
     0xa0, 0x8f, 0x4a, 0xb4, 0x1a, 0xab, 0x9b, 0x13, 0x48,
     0x80, 0x40, 0x44, 0x48, 0xc7, 0x81, 0x80, 0x94, 0xe9,
     0xf,  0xf4, 0xfd, 0x5f, 0x8d, 0xd8, 0x1e, 0xac, 0xa0,
     0x9e, 0xd6, 0x66, 0xda, 0xc6, 0xef, 0x68, 0xde, 0xa2,
     0xaf, 0x31, 0x6a, 0x31, 0x54, 0x34, 0x6f, 0xec, 0xbd,
     0x25, 0x18, 0x93, 0xed, 0x40, 0xd9, 0xfe, 0x8d, 0xb3,
     0x8e, 0xf1, 0xa3, 0xc4, 0x73, 0xe4, 0xec, 0x54, 0x40,
     0xdd, 0xac, 0x8,  0x4c, 0x3e, 0x4,  0xaa, 0x7e, 0x85,
     0xa7, 0x69, 0x81, 0xba, 0x68, 0x24, 0xc1, 0xbf, 0xb1,
     0xab, 0x48, 0x54, 0x72, 0x2,  0x9b, 0x71, 0xd8, 0x8,
     0xce, 0x33, 0xe2, 0xcc, 0x3c, 0xb,  0x5f, 0xc1, 0xf3,
     0xde, 0x8a, 0x6d, 0xc6, 0x6b, 0x1f, 0x56, 0xce, 0x22,
     0xf,  0xc4, 0x85, 0x9c, 0x62, 0x2b, 0xd2, 0x4e, 0xa1,
     0xa,  0x3,  0xf4, 0x20, 0x4c, 0xd,  0xd3, 0x41, 0x26,
     0xd,  0xfd, 0xa1, 0x43, 0x59, 0x4c, 0xbc, 0x1e, 0xd7,
     0xa2, 0x16, 0x4,  0xcf, 0xbd, 0x58, 0x7a, 0xc6, 0xae,
     0x47, 0x23, 0x58, 0x82, 0x78, 0x47, 0x80, 0xda, 0xd,
     0x5a, 0x42, 0xa3, 0x90, 0x14, 0xdb, 0x8e, 0x9b, 0x5e,
     0x8d, 0x1e, 0xc8, 0x6e, 0x1d, 0x20, 0xda, 0x36, 0xa8,
     0xb3, 0x95, 0x62, 0x64, 0xb3, 0x1d, 0x15, 0xf0, 0x31,
     0xed, 0x25, 0xe4, 0x69, 0x40, 0x46, 0x71, 0x9f, 0x16,
     0x66, 0x5f, 0xe3, 0x1a, 0x70, 0xd9, 0x2c, 0x83, 0x4,
     0x3b, 0x50, 0x7d, 0xda, 0xd2, 0x1b, 0xf5, 0xe6, 0x46,
     0x98, 0xe0, 0xcf, 0x22, 0x2a, 0x18, 0xc3, 0xa2, 0xf1,
     0xcc, 0x3b, 0x22, 0xcc, 0x3d, 0x4c, 0x37, 0x96, 0x58,
     0x0},
    {0xd1, 0xe2, 0xde, 0x70, 0xee, 0xf8, 0x6c, 0x69, 0x64, 0xfb, 0x54, 0x2b,
     0xc2, 0xd4, 0x60, 0xaa, 0xbf, 0xaa, 0x10, 0xa4, 0xa0, 0x93, 0x26, 0x2b,
     0x7d, 0x19, 0x9e, 0x70, 0x6f, 0xc2, 0xd4, 0x89, 0x15, 0x53, 0x29, 0x69,
     0x10, 0xf3, 0xa9, 0x73, 0x1,  0x26, 0x82, 0xe4, 0x1c, 0x4e, 0x2b, 0x2,
     0xbe, 0x20, 0x17, 0xb7, 0x25, 0x3b, 0xbf, 0x93, 0x9,  0xde, 0x58, 0x19,
     0xcb, 0x42, 0xe8, 0x19, 0x56, 0xf4, 0xc9, 0x9b, 0xc9, 0x76, 0x5c, 0xaf,
     0x53, 0xb1, 0xd0, 0xbb, 0x82, 0x79, 0x82, 0x6a, 0xdb, 0xbc, 0x55, 0x22,
     0xe9, 0x15, 0xc1, 0x20, 0xa6, 0x18, 0xa5, 0xa7, 0xf5, 0xe8, 0x97, 0x8,
     0x93, 0x39, 0x65, 0xc,  0xf7, 0xab, 0x31, 0xf0, 0x56, 0x8,  0x72, 0xaa,
     0x27, 0xbb, 0x2b, 0x7a, 0x6d, 0x6f, 0x4d, 0x85, 0x2f, 0x43, 0x21, 0x6,
     0x75, 0xc4, 0xe,  0x17, 0x76, 0xf9, 0x61, 0x27, 0xbc, 0x6e, 0xb1, 0x6b,
     0xa2, 0xc5, 0x29, 0x6c, 0x29, 0x9d, 0x5,  0x48, 0x7a, 0xd3, 0x80, 0x2e,
     0xe7, 0xb3, 0xcf, 0xa7, 0xdf, 0x32, 0xe7, 0x2b, 0x9d, 0x5,  0x4b, 0x5f,
     0xd3, 0x8a, 0x80, 0xd2, 0xdb, 0xa8, 0x3,  0xb5, 0xbb, 0x94, 0x31, 0xbb,
     0x1e, 0x98, 0xbd, 0x11, 0xb9, 0x3d, 0xb7, 0xc3, 0xd4, 0x51, 0x36, 0x55,
     0x9b, 0xb8, 0x6a, 0x29, 0x5a, 0xa2, 0x4,  0xec, 0xbe, 0xbf, 0x6f, 0x7a,
     0x51, 0x1,  0x51, 0x4d, 0xe4, 0x96, 0x54, 0x1,  0xe3, 0x3d, 0x94, 0x50,
     0x8d, 0x25, 0xf7, 0x2d, 0x6,  0x12, 0xee, 0xd9, 0xb2, 0x6e, 0x95, 0x57,
     0x61, 0x26, 0xff, 0x94, 0x8c, 0xab, 0xdb, 0xc5, 0xd6, 0x83, 0x1d, 0x62,
     0xbc, 0xe6, 0xc8, 0x20, 0xdd, 0x5f, 0x44, 0xfe, 0x8c, 0xae, 0x2,  0xbc,
     0x65, 0xcf, 0xef, 0x21, 0xb,  0xab, 0x30, 0xbc, 0x69, 0x1,  0x20, 0x10,
     0x36, 0x83, 0xe7, 0xd3, 0xc,  0xf6, 0x2,  0xcf, 0xb1, 0xa8, 0xa2, 0x2c,
     0xfe, 0x8,  0x6a, 0x16, 0xe4, 0xc9, 0xa7, 0x2b, 0x75, 0xe0, 0x65, 0xe3,
     0x6,  0xa6, 0x96, 0xf8, 0xba, 0x3b, 0xa6, 0x5e, 0xe6, 0xae, 0xd2, 0x51,
     0xa8, 0x18, 0xbd, 0x3f, 0x89, 0x70, 0xca, 0x4a, 0x38, 0xfc, 0x51, 0x4a,
     0xf9, 0x39, 0x73, 0x47, 0xe0, 0x38, 0xac, 0xd8, 0x3e, 0xc7, 0x16, 0x93,
     0x72, 0x2c, 0xcc, 0x0},
    {
        {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
         0x48, 0x81, 0xFF, 0x48},
        {0xEF, 0xA8, 0xB2, 0x22, 0x9E, 0x72, 0x0C, 0x2A, 0x7C, 0x36, 0xEA, 0x55,
         0xE9, 0x60, 0x56, 0x95},
        {0xD3, 0xC5, 0xD5, 0x92, 0x32, 0x7F, 0xB1, 0x1C, 0x40, 0x35, 0xC6, 0x68,
         0x0A, 0xF8, 0xC6, 0xD1},
        {0x5A, 0xCB, 0x1D, 0x64, 0x4C, 0x0D, 0x51, 0x20, 0x4E, 0xA5, 0xF1, 0x45,
         0x10, 0x10, 0xD8, 0x52},
        {0x60, 0x90, 0xEA, 0xE0, 0x4C, 0x83, 0x70, 0x6E, 0xEC, 0xBF, 0x65, 0x2B,
         0xE8, 0xE3, 0x65, 0x66},
    },
    {
        {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00},
        {0xE2, 0x8B, 0xCF, 0x7B, 0xC0, 0x00, 0x00, 0x00},
        {0x39, 0x8A, 0x59, 0xB4, 0x2C, 0x00, 0x00, 0x00},
        {0xFA, 0x55, 0x6B, 0x26, 0x1C, 0x00, 0x00, 0x00},
        {0x72, 0xA4, 0xF2, 0x0F, 0x48, 0x00, 0x00, 0x00},
    }

};

static const cipherbit_test_vector_t kasumi_f8_bitvectors[] = {
    {798,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x83},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer0*/
    {510,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xDB},

     {0x3D, 0xEA, 0xCC, 0x7C, 0x15, 0x82, 0x1C, 0xAA, 0x89, 0xEE, 0xCA,
      0xDE, 0x9B, 0x5B, 0xD3, 0x61, 0x4B, 0xD0, 0xC8, 0x41, 0x9D, 0x71,
      0x03, 0x85, 0xDD, 0xBE, 0x58, 0x49, 0xEF, 0x1B, 0xAC, 0x5A, 0xE8,
      0xB1, 0x4A, 0x5B, 0x0A, 0x67, 0x41, 0x52, 0x1E, 0xB4, 0xE0, 0x0B,
      0xB9, 0xEC, 0xF3, 0xE9, 0xF7, 0xCC, 0xB9, 0xCA, 0xE7, 0x41, 0x52,
      0xD7, 0xF4, 0xE2, 0xA0, 0x34, 0xB6, 0xEA, 0x00, 0xEF},

     {0xEF, 0xA8, 0xB2, 0x22, 0x9E, 0x72, 0x0C, 0x2A, 0x7C, 0x36, 0xEA, 0x55,
      0xE9, 0x60, 0x56, 0x95},
     {0xE2, 0x8B, 0xCF, 0x7B, 0xC0, 0x00, 0x00, 0x00} }, /*buffer1*/
    {253,
     16,
     8,
     {0x98, 0x1B, 0xA6, 0x82, 0x4C, 0x1B, 0xFB, 0x1A, 0xB4, 0x85, 0x47,
      0x20, 0x29, 0xB7, 0x1D, 0x80, 0x8C, 0xE3, 0x3E, 0x2C, 0xC3, 0xC0,
      0xB5, 0xFC, 0x1F, 0x3D, 0xE8, 0xA6, 0xDC, 0x66, 0xB1, 0xF7},
     {0x5B, 0xB9, 0x43, 0x1B, 0xB1, 0xE9, 0x8B, 0xD1, 0x1B, 0x93, 0xDB,
      0x7C, 0x3D, 0x45, 0x13, 0x65, 0x59, 0xBB, 0x86, 0xA2, 0x95, 0xAA,
      0x20, 0x4E, 0xCB, 0xEB, 0xF6, 0xF7, 0xA5, 0x10, 0x15, 0x17},
     {0xD3, 0xC5, 0xD5, 0x92, 0x32, 0x7F, 0xB1, 0x1C, 0x40, 0x35, 0xC6, 0x68,
      0x0A, 0xF8, 0xC6, 0xD1},
     {0x39, 0x8A, 0x59, 0xB4, 0x2C, 0x00, 0x00, 0x00} }, /*buffer2*/
    {120,
     16,
     8,
     {0xAD, 0x9C, 0x44, 0x1F, 0x89, 0x0B, 0x38, 0xC4, 0x57, 0xA4, 0x9D, 0x42,
      0x14, 0x07, 0xE8},
     {0x9B, 0xC9, 0x2C, 0xA8, 0x03, 0xC6, 0x7B, 0x28, 0xA1, 0x1A, 0x4B, 0xEE,
      0x5A, 0x0C, 0x25},
     {0x5A, 0xCB, 0x1D, 0x64, 0x4C, 0x0D, 0x51, 0x20, 0x4E, 0xA5, 0xF1, 0x45,
      0x10, 0x10, 0xD8, 0x52},
     {0xFA, 0x55, 0x6B, 0x26, 0x1C, 0x00, 0x00, 0x00} }, /*buffer3*/
    {837,
     16,
     8,
     {0x40, 0x98, 0x1B, 0xA6, 0x82, 0x4C, 0x1B, 0xFB, 0x42, 0x86, 0xB2, 0x99,
      0x78, 0x3D, 0xAF, 0x44, 0x2C, 0x09, 0x9F, 0x7A, 0xB0, 0xF5, 0x8D, 0x5C,
      0x8E, 0x46, 0xB1, 0x04, 0xF0, 0x8F, 0x01, 0xB4, 0x1A, 0xB4, 0x85, 0x47,
      0x20, 0x29, 0xB7, 0x1D, 0x36, 0xBD, 0x1A, 0x3D, 0x90, 0xDC, 0x3A, 0x41,
      0xB4, 0x6D, 0x51, 0x67, 0x2A, 0xC4, 0xC9, 0x66, 0x3A, 0x2B, 0xE0, 0x63,
      0xDA, 0x4B, 0xC8, 0xD2, 0x80, 0x8C, 0xE3, 0x3E, 0x2C, 0xCC, 0xBF, 0xC6,
      0x34, 0xE1, 0xB2, 0x59, 0x06, 0x08, 0x76, 0xA0, 0xFB, 0xB5, 0xA4, 0x37,
      0xEB, 0xCC, 0x8D, 0x31, 0xC1, 0x9E, 0x44, 0x54, 0x31, 0x87, 0x45, 0xE3,
      0x98, 0x76, 0x45, 0x98, 0x7A, 0x98, 0x6F, 0x2C, 0xB7},
     {0xDD, 0xB3, 0x64, 0xDD, 0x2A, 0xAE, 0xC2, 0x4D, 0xFF, 0x29, 0x19,
      0x57, 0xB7, 0x8B, 0xAD, 0x06, 0x3A, 0xC5, 0x79, 0xCD, 0x90, 0x41,
      0xBA, 0xBE, 0x89, 0xFD, 0x19, 0x5C, 0x05, 0x78, 0xCB, 0x9F, 0xDE,
      0x42, 0x17, 0x56, 0x61, 0x78, 0xD2, 0x02, 0x40, 0x20, 0x6D, 0x07,
      0xCF, 0xA6, 0x19, 0xEC, 0x05, 0x9F, 0x63, 0x51, 0x44, 0x59, 0xFC,
      0x10, 0xD4, 0x2D, 0xC9, 0x93, 0x4E, 0x56, 0xEB, 0xC0, 0xCB, 0xC6,
      0x0D, 0x4D, 0x2D, 0xF1, 0x74, 0x77, 0x4C, 0xBD, 0xCD, 0x5D, 0xA4,
      0xA3, 0x50, 0x31, 0x7A, 0x7F, 0x12, 0xE1, 0x94, 0x94, 0x71, 0xF8,
      0xA2, 0x95, 0xF2, 0x72, 0xE6, 0x8F, 0xC0, 0x71, 0x59, 0xB0, 0x7D,
      0x8E, 0x2D, 0x26, 0xE4, 0x59,
      0x9F},
     {0x60, 0x90, 0xEA, 0xE0, 0x4C, 0x83, 0x70, 0x6E, 0xEC, 0xBF, 0x65, 0x2B,
      0xE8, 0xE3, 0x65, 0x66},
     {0x72, 0xA4, 0xF2, 0x0F, 0x48, 0x00, 0x00, 0x00} }, /*buffer3*/
};

static cipher_test_vector_t kasumi_f8_vectors[] = {
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer0*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer1*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer2*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer3*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer4*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer5*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer6*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer7*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer8*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer9*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer10*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer11*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer12*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer13*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer14*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer15*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer16*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer17*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer18*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer19*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer20*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer21*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer22*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer23*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer24*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer25*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer26*/
    {64,
     16,
     8,
     {0x10, 0x11, 0x12, 0x31, 0xE0, 0x60, 0x25, 0x3A, 0x43, 0xFD, 0x3F,
      0x57, 0xE3, 0x76, 0x07, 0xAB, 0x28, 0x27, 0xB5, 0x99, 0xB6, 0xB1,
      0xBB, 0xDA, 0x37, 0xA8, 0xAB, 0xCC, 0x5A, 0x8C, 0x55, 0x0D, 0x1B,
      0xFB, 0x2F, 0x49, 0x46, 0x24, 0xFB, 0x50, 0x36, 0x7F, 0xA3, 0x6C,
      0xE3, 0xBC, 0x68, 0xF1, 0x1C, 0xF9, 0x3B, 0x15, 0x10, 0x37, 0x6B,
      0x02, 0x13, 0x0F, 0x81, 0x2A, 0x9F, 0xA1, 0x69, 0xD8},
     {0xBF, 0x35, 0xDE, 0x33, 0x7A, 0xA3, 0xB8, 0x32, 0x60, 0x20, 0x2F,
      0x16, 0x4D, 0x9A, 0xA9, 0xD0, 0xF1, 0x7B, 0x6F, 0x4B, 0xFD, 0x76,
      0xAD, 0xF5, 0x08, 0x37, 0x01, 0xD0, 0xDA, 0x5D, 0x8E, 0x16, 0x9C,
      0x83, 0x05, 0x65, 0x5B, 0xED, 0xCB, 0x56, 0xD2, 0xE4, 0x0F, 0x28,
      0x14, 0xA7, 0xEE, 0x7D, 0xB9, 0xC0, 0xB2, 0x9C, 0xF1, 0x3D, 0xB4,
      0xB1, 0xF3, 0x70, 0x6B, 0xB6, 0xB3, 0x81, 0xF8, 0x92},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} }, /*buffer27*/
    {100,
     16,
     8,
     {0x7E, 0xC6, 0x12, 0x72, 0x74, 0x3B, 0xF1, 0x61, 0x47, 0x26, 0x44, 0x6A,
      0x6C, 0x38, 0xCE, 0xD1, 0x66, 0xF6, 0xCA, 0x76, 0xEB, 0x54, 0x30, 0x04,
      0x42, 0x86, 0x34, 0x6C, 0xEF, 0x13, 0x0F, 0x92, 0x92, 0x2B, 0x03, 0x45,
      0x0D, 0x3A, 0x99, 0x75, 0xE5, 0xBD, 0x2E, 0xA0, 0xEB, 0x55, 0xAD, 0x8E,
      0x1B, 0x19, 0x9E, 0x3E, 0xC4, 0x31, 0x60, 0x20, 0xE9, 0xA1, 0xB2, 0x85,
      0xE7, 0x62, 0x79, 0x53, 0x59, 0xB7, 0xBD, 0xFD, 0x39, 0xBE, 0xF4, 0xB2,
      0x48, 0x45, 0x83, 0xD5, 0xAF, 0xE0, 0x82, 0xAE, 0xE6, 0x38, 0xBF, 0x5F,
      0xD5, 0xA6, 0x06, 0x19, 0x39, 0x01, 0xA0, 0x8F, 0x4A, 0xB4, 0x1A, 0xAB,
      0x9B, 0x13, 0x48, 0x80},
     {0xD1, 0xE2, 0xDE, 0x70, 0xEE, 0xF8, 0x6C, 0x69, 0x64, 0xFB, 0x54, 0x2B,
      0xC2, 0xD4, 0x60, 0xAA, 0xBF, 0xAA, 0x10, 0xA4, 0xA0, 0x93, 0x26, 0x2B,
      0x7D, 0x19, 0x9E, 0x70, 0x6F, 0xC2, 0xD4, 0x89, 0x15, 0x53, 0x29, 0x69,
      0x10, 0xF3, 0xA9, 0x73, 0x01, 0x26, 0x82, 0xE4, 0x1C, 0x4E, 0x2B, 0x02,
      0xBE, 0x20, 0x17, 0xB7, 0x25, 0x3B, 0xBF, 0x93, 0x09, 0xDE, 0x58, 0x19,
      0xCB, 0x42, 0xE8, 0x19, 0x56, 0xF4, 0xC9, 0x9B, 0xC9, 0x76, 0x5C, 0xAF,
      0x53, 0xB1, 0xD0, 0xBB, 0x82, 0x79, 0x82, 0x6A, 0xDB, 0xBC, 0x55, 0x22,
      0xE9, 0x15, 0xC1, 0x20, 0xA6, 0x18, 0xA5, 0xA7, 0xF5, 0xE8, 0x97, 0x08,
      0x93, 0x39, 0x65, 0x0F},
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     {0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00} } /*buffer28*/
};

static hash_test_vector_t kasumi_f9_vectors[] = {
    {/*input*/
     {0x38, 0xA6, 0xF0, 0x56, 0x05, 0xD2, 0xEC, 0x49, 0x6B, 0x22, 0x77,
      0x37, 0x29, 0x6F, 0x39, 0x3C, 0x80, 0x79, 0x35, 0x3E, 0xDC, 0x87,
      0xE2, 0xE8, 0x05, 0xD2, 0xEC, 0x49, 0xA4, 0xF2, 0xD8, 0xE2},
     /*inputlength*/
     32,
     /*key*/
     {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
      0x48, 0x81, 0xFF, 0x48},
     /*keylength*/
     16,
     /*expectedout*/
     {0xF6, 0x3B, 0xD7, 0x2C} },
    {{0x3E, 0xDC, 0x87, 0xE2, 0xA4, 0xF2, 0xD8, 0xE2, 0xB5, 0x92,
      0x43, 0x84, 0x32, 0x8A, 0x4A, 0xE0, 0x0B, 0x73, 0x71, 0x09,
      0xF8, 0xB6, 0xC8, 0xDD, 0x2B, 0x4D, 0xB6, 0x3D, 0xD5, 0x33,
      0x98, 0x1C, 0xEB, 0x19, 0xAA, 0xD5, 0x2A, 0x5B, 0x2B, 0xC3},
     40,
     {0xD4, 0x2F, 0x68, 0x24, 0x28, 0x20, 0x1C, 0xAF, 0xCD, 0x9F, 0x97, 0x94,
      0x5E, 0x6D, 0xE7, 0xB7},
     16,
     {0xA9, 0xDA, 0xF1, 0xFF} },
    {{0x14, 0x79, 0x3E, 0x41, 0x03, 0x97, 0xE8, 0xFD, 0xD0, 0xA7, 0xD4, 0x63,
      0xDF, 0x9F, 0xB2, 0xB2, 0x78, 0x83, 0x3F, 0xA0, 0x2E, 0x23, 0x5A, 0xA1,
      0x72, 0xBD, 0x97, 0x0C, 0x14, 0x73, 0xE1, 0x29, 0x07, 0xFB, 0x64, 0x8B,
      0x65, 0x99, 0xAA, 0xA0, 0xB2, 0x4A, 0x03, 0x86, 0x65, 0x42, 0x2B, 0x20,
      0xA4, 0x99, 0x27, 0x6A, 0x50, 0x42, 0x70, 0x09, 0xC0},
     57,
     {0xC7, 0x36, 0xC6, 0xAA, 0xB2, 0x2B, 0xFF, 0xF9, 0x1E, 0x26, 0x98, 0xD2,
      0xE2, 0x2A, 0xD5, 0x7E},
     16,
     {0xDD, 0x7D, 0xFA, 0xDD} },
    {{0x36, 0xAF, 0x61, 0x44, 0x98, 0x38, 0xF0, 0x3A, 0xB3, 0xD3,
      0xC9, 0x17, 0x0A, 0x4E, 0x16, 0x32, 0xF6, 0x0F, 0x86, 0x10,
      0x13, 0xD2, 0x2D, 0x84, 0xB7, 0x26, 0xB6, 0xA2, 0x78, 0xD8,
      0x02, 0xD1, 0xEE, 0xAF, 0x13, 0x21, 0xBA, 0x59, 0x29, 0xDF},
     40,
     {0x7E, 0x5E, 0x94, 0x43, 0x1E, 0x11, 0xD7, 0x38, 0x28, 0xD7, 0x39, 0xCC,
      0x6C, 0xED, 0x45, 0x73},
     16,
     {0x2B, 0xEE, 0xF3, 0xAC}

    },
    {{0x36, 0xAF, 0x61, 0x44, 0x98, 0x38, 0xF0, 0x3A, 0x59, 0x32,
      0xBC, 0x0A, 0xCE, 0x2B, 0x0A, 0xBA, 0x33, 0xD8, 0xAC, 0x18,
      0x8A, 0xC5, 0x4F, 0x34, 0x6F, 0xAD, 0x10, 0xBF, 0x9D, 0xEE,
      0x29, 0x20, 0xB4, 0x3B, 0xD0, 0xC5, 0x3A, 0x91, 0x5C, 0xB7,
      0xDF, 0x6C, 0xAA, 0x72, 0x05, 0x3A, 0xBF, 0xF3, 0x80},
     49,
     {0xFD, 0xB9, 0xCF, 0xDF, 0x28, 0x93, 0x6C, 0xC4, 0x83, 0xA3, 0x18, 0x69,
      0xD8, 0x1B, 0x8F, 0xAB},
     16,
     {0x15, 0x37, 0xD3, 0x16} },
    {{0x36, 0xAF, 0x61, 0x44, 0x98, 0x38, 0xF0, 0x3A, 0xD3, 0xC5, 0x38, 0x39,
      0x62, 0x68, 0x20, 0x71, 0x77, 0x65, 0x66, 0x76, 0x20, 0x32, 0x38, 0x37,
      0x63, 0x62, 0x40, 0x98, 0x1B, 0xA6, 0x82, 0x4C, 0x1B, 0xFB, 0x1A, 0xB4,
      0x85, 0x47, 0x20, 0x29, 0xB7, 0x1D, 0x80, 0x8C, 0xE3, 0x3E, 0x2C, 0xC3,
      0xC0, 0xB5, 0xFC, 0x1F, 0x3D, 0xE8, 0xA6, 0xDC, 0x80},
     57,
     {0x68, 0x32, 0xA6, 0x5C, 0xFF, 0x44, 0x73, 0x62, 0x1E, 0xBD, 0xD4, 0xBA,
      0x26, 0xA9, 0x21, 0xFE},
     16,
     {0x8B, 0x2D, 0x57, 0x0F} },
    {{0xC7, 0x59, 0x0E, 0xA9, 0x57, 0xD5, 0xDF, 0x7D, 0xBB, 0xB0, 0x57,
      0x03, 0x88, 0x09, 0x49, 0x6B, 0xCF, 0xF8, 0x6D, 0x6F, 0xBC, 0x8C,
      0xE5, 0xB1, 0x35, 0xA0, 0x6B, 0x16, 0x60, 0x54, 0xF2, 0xD5, 0x65,
      0xBE, 0x8A, 0xCE, 0x75, 0xDC, 0x85, 0x1E, 0x0B, 0xCD, 0xD8, 0xF0,
      0x71, 0x41, 0xC4, 0x95, 0x87, 0x2F, 0xB5, 0xD8, 0xC0, 0xC6, 0x6A,
      0x8B, 0x6D, 0xA5, 0x56, 0x66, 0x3E, 0x4E, 0x46, 0x12, 0x05, 0xD8,
      0x45, 0x80, 0xBE, 0xE5, 0xBC, 0x7E, 0x80},
     73,
     {0xD3, 0x41, 0x9B, 0xE8, 0x21, 0x08, 0x7A, 0xCD, 0x02, 0x12, 0x3A, 0x92,
      0x48, 0x03, 0x33, 0x59},
     16,
     {0x02, 0x15, 0x81, 0x70} },
    {{0x36, 0xAF, 0x61, 0x44, 0x4F, 0x30, 0x2A, 0xD2, 0x35, 0xC6, 0x87, 0x16,
      0x63, 0x3C, 0x66, 0xFB, 0x75, 0x0C, 0x26, 0x68, 0x65, 0xD5, 0x3C, 0x11,
      0xEA, 0x05, 0xB1, 0xE9, 0xFA, 0x49, 0xC8, 0x39, 0x8D, 0x48, 0xE1, 0xEF,
      0xA5, 0x90, 0x9D, 0x39, 0x47, 0x90, 0x28, 0x37, 0xF5, 0xAE, 0x96, 0xD5,
      0xA0, 0x5B, 0xC8, 0xD6, 0x1C, 0xA8, 0xDB, 0xEF, 0x1B, 0x13, 0xA4, 0xB4,
      0xAB, 0xFE, 0x4F, 0xB1, 0x00, 0x60, 0x45, 0xB6, 0x74, 0xBB, 0x54, 0x72,
      0x93, 0x04, 0xC3, 0x82, 0xBE, 0x53, 0xA5, 0xAF, 0x05, 0x55, 0x61, 0x76,
      0xF6, 0xEA, 0xA2, 0xEF, 0x1D, 0x05, 0xE4, 0xB0, 0x83, 0x18, 0x1E, 0xE6,
      0x74, 0xCD, 0xA5, 0xA4, 0x85, 0xF7, 0x4D, 0x7A, 0xC0},
     105,
     {0x83, 0xFD, 0x23, 0xA2, 0x44, 0xA7, 0x4C, 0xF3, 0x58, 0xDA, 0x30, 0x19,
      0xF1, 0x72, 0x26, 0x35},
     16,
     {0x95, 0xAE, 0x41, 0xBA} },
    {{0x29, 0x6F, 0x39, 0x3C, 0x6B, 0x22, 0x77, 0x37, 0x10, 0xBF, 0xFF, 0x83,
      0x9E, 0x0C, 0x71, 0x65, 0x8D, 0xBB, 0x2D, 0x17, 0x07, 0xE1, 0x45, 0x72,
      0x4F, 0x41, 0xC1, 0x6F, 0x48, 0xBF, 0x40, 0x3C, 0x3B, 0x18, 0xE3, 0x8F,
      0xD5, 0xD1, 0x66, 0x3B, 0x6F, 0x6D, 0x90, 0x01, 0x93, 0xE3, 0xCE, 0xA8,
      0xBB, 0x4F, 0x1B, 0x4F, 0x5B, 0xE8, 0x22, 0x03, 0x22, 0x32, 0xA7, 0x8D,
      0x7D, 0x75, 0x23, 0x8D, 0x5E, 0x6D, 0xAE, 0xCD, 0x3B, 0x43, 0x22, 0xCF,
      0x59, 0xBC, 0x7E, 0xA8, 0x4A, 0xB1, 0x88, 0x11, 0xB5, 0xBF, 0xB7, 0xBC,
      0x55, 0x3F, 0x4F, 0xE4, 0x44, 0x78, 0xCE, 0x28, 0x7A, 0x14, 0x87, 0x99,
      0x90, 0xD1, 0x8D, 0x12, 0xCA, 0x79, 0xD2, 0xC8, 0x55, 0x14, 0x90, 0x21,
      0xCD, 0x5C, 0xE8, 0xCA, 0x03, 0x71, 0xCA, 0x04, 0xFC, 0xCE, 0x14, 0x3E,
      0x3D, 0x7C, 0xFE, 0xE9, 0x45, 0x85, 0xB5, 0x88, 0x5C, 0xAC, 0x46, 0x06,
      0x8B, 0xC0},
     134,
     {0xF4, 0xEB, 0xEC, 0x69, 0xE7, 0x3E, 0xAF, 0x2E, 0xB2, 0xCF, 0x6A, 0xF4,
      0xB3, 0x12, 0x0F, 0xFD},
     16,
     {0xC3, 0x83, 0x83, 0x9D} },
    {{0x78, 0x27, 0xFA, 0xB2, 0xA5, 0x6C, 0x6C, 0xA2, 0x70, 0xDE, 0xDF, 0x2D,
      0xC4, 0x2C, 0x5C, 0xBD, 0x3A, 0x96, 0xF8, 0xA0, 0xB1, 0x14, 0x18, 0xB3,
      0x60, 0x8D, 0x57, 0x33, 0x60, 0x4A, 0x2C, 0xD3, 0x6A, 0xAB, 0xC7, 0x0C,
      0xE3, 0x19, 0x3B, 0xB5, 0x15, 0x3B, 0xE2, 0xD3, 0xC0, 0x6D, 0xFD, 0xB2,
      0xD1, 0x6E, 0x9C, 0x35, 0x71, 0x58, 0xBE, 0x6A, 0x41, 0xD6, 0xB8, 0x61,
      0xE4, 0x91, 0xDB, 0x3F, 0xBF, 0xEB, 0x51, 0x8E, 0xFC, 0xF0, 0x48, 0xD7,
      0xD5, 0x89, 0x53, 0x73, 0x0F, 0xF3, 0x0C, 0x9E, 0xC4, 0x70, 0xFF, 0xCD,
      0x66, 0x3D, 0xC3, 0x42, 0x01, 0xC3, 0x6A, 0xDD, 0xC0, 0x11, 0x1C, 0x35,
      0xB3, 0x8A, 0xFE, 0xE7, 0xCF, 0xDB, 0x58, 0x2E, 0x37, 0x31, 0xF8, 0xB4,
      0xBA, 0xA8, 0xD1, 0xA8, 0x9C, 0x06, 0xE8, 0x11, 0x99, 0xA9, 0x71, 0x62,
      0x27, 0xBE, 0x34, 0x4E, 0xFC, 0xB4, 0x36, 0xDD, 0xD0, 0xF0, 0x96, 0xC0,
      0x64, 0xC3, 0xB5, 0xE2, 0xC3, 0x99, 0x99, 0x3F, 0xC7, 0x73, 0x94, 0xF9,
      0xE0, 0x97, 0x20, 0xA8, 0x11, 0x85, 0x0E, 0xF2, 0x3B, 0x2E, 0xE0, 0x5D,
      0x9E, 0x61, 0x73, 0x60, 0x9D, 0x86, 0xE1, 0xC0, 0xC1, 0x8E, 0xA5, 0x1A,
      0x01, 0x2A, 0x00, 0xBB, 0x41, 0x3B, 0x9C, 0xB8, 0x18, 0x8A, 0x70, 0x3C,
      0xD6, 0xBA, 0xE3, 0x1C, 0xC6, 0x7B, 0x34, 0xB1, 0xB0, 0x00, 0x19, 0xE6,
      0xA2, 0xB2, 0xA6, 0x90, 0xF0, 0x26, 0x71, 0xFE, 0x7C, 0x9E, 0xF8, 0xDE,
      0xC0, 0x09, 0x4E, 0x53, 0x37, 0x63, 0x47, 0x8D, 0x58, 0xD2, 0xC5, 0xF5,
      0xB8, 0x27, 0xA0, 0x14, 0x8C, 0x59, 0x48, 0xA9, 0x69, 0x31, 0xAC, 0xF8,
      0x4F, 0x46, 0x5A, 0x64, 0xE6, 0x2C, 0xE7, 0x40, 0x07, 0xE9, 0x91, 0xE3,
      0x7E, 0xA8, 0x23, 0xFA, 0x0F, 0xB2, 0x19, 0x23, 0xB7, 0x99, 0x05, 0xB7,
      0x33, 0xB6, 0x31, 0xE6, 0xC7, 0xD6, 0x86, 0x0A, 0x38, 0x31, 0xAC, 0x35,
      0x1A, 0x9C, 0x73, 0x0C, 0x52, 0xFF, 0x72, 0xD9, 0xD3, 0x08, 0xEE, 0xDB,
      0xAB, 0x21, 0xFD, 0xE1, 0x43, 0xA0, 0xEA, 0x17, 0xE2, 0x3E, 0xDC, 0x1F,
      0x74, 0xCB, 0xB3, 0x63, 0x8A, 0x20, 0x33, 0xAA, 0xA1, 0x54, 0x64, 0xEA,
      0xA7, 0x33, 0x38, 0x5D, 0xBB, 0xEB, 0x6F, 0xD7, 0x35, 0x09, 0xB8, 0x57,
      0xE6, 0xA4, 0x19, 0xDC, 0xA1, 0xD8, 0x90, 0x7A, 0xF9, 0x77, 0xFB, 0xAC,
      0x4D, 0xFA, 0x35, 0xEF},
     328,
     {0x5D, 0x0A, 0x80, 0xD8, 0x13, 0x4A, 0xE1, 0x96, 0x77, 0x82, 0x4B, 0x67,
      0x1E, 0x83, 0x8A, 0xF4},
     16,
     {0x3A, 0xE4, 0xBF, 0xF3} } };

static hash_iv_test_vector_t kasumi_f9_IV_vectors[] = {
    {
        /* 3GPP specs Test Set 1 */
        /*input*/
        {0x6B, 0x22, 0x77, 0x37, 0x29, 0x6F, 0x39, 0x3C,
         0x80, 0x79, 0x35, 0x3E, 0xDC, 0x87, 0xE2, 0xE8,
         0x05, 0xD2, 0xEC, 0x49, 0xA4, 0xF2, 0xD8, 0xE0},
        /*length*/
        189,
        /*key*/
        {0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00, 0x95, 0x2C, 0x49, 0x10,
         0x48, 0x81, 0xFF, 0x48},
        /*KeyLeninBytes*/
        16,
        /*exp out*/
        {0xF6, 0x3B, 0xD7, 0x2C},
        /*iv*/
        {0x38, 0xA6, 0xF0, 0x56, 0x05, 0xD2, 0xEC, 0x49},
        /*ivLeninbytes*/
        8,
        /*direction*/
        0,
    },
    {
        /* 3GPP specs Test Set 2 */
        /*input*/
        {0xB5, 0x92, 0x43, 0x84, 0x32, 0x8A, 0x4A, 0xE0, 0x0B, 0x73, 0x71,
         0x09, 0xF8, 0xB6, 0xC8, 0xDD, 0x2B, 0x4D, 0xB6, 0x3D, 0xD5, 0x33,
         0x98, 0x1C, 0xEB, 0x19, 0xAA, 0xD5, 0x2A, 0x5B, 0x2B, 0xC0},
        /*length*/
        254,
        /*key*/
        {0xD4, 0x2F, 0x68, 0x24, 0x28, 0x20, 0x1C, 0xAF, 0xCD, 0x9F, 0x97, 0x94,
         0x5E, 0x6D, 0xE7, 0xB7},
        /*KeyLeninBytes*/
        16,
        /*exp out*/
        {0xA9, 0xDA, 0xF1, 0xFF},
        /*iv*/
        {0x3E, 0xDC, 0x87, 0xE2, 0xA4, 0xF2, 0xD8, 0xE2},
        /*ivLeninbytes*/
        8,
        /*direction*/
        1,
    },
    {
        /* 3GPP specs Test Set 3 */
        /*input*/
        {0x59, 0x32, 0xBC, 0x0A, 0xCE, 0x2B, 0x0A, 0xBA, 0x33, 0xD8,
         0xAC, 0x18, 0x8A, 0xC5, 0x4F, 0x34, 0x6F, 0xAD, 0x10, 0xBF,
         0x9D, 0xEE, 0x29, 0x20, 0xB4, 0x3B, 0xD0, 0xC5, 0x3A, 0x91,
         0x5C, 0xB7, 0xDF, 0x6C, 0xAA, 0x72, 0x05, 0x3A, 0xBF, 0xF2},
        /*length*/
        319,
        /*key*/
        {0xFD, 0xB9, 0xCF, 0xDF, 0x28, 0x93, 0x6C, 0xC4, 0x83, 0xA3, 0x18, 0x69,
         0xD8, 0x1B, 0x8F, 0xAB},
        /*KeyLeninBytes*/
        16,
        /*exp out*/
        {0x15, 0x37, 0xD3, 0x16},
        /*iv*/
        {0x36, 0xAF, 0x61, 0x44, 0x98, 0x38, 0xF0, 0x3A},
        /*ivLeninbytes*/
        8,
        /*direction*/
        1,
    },
    {
        /* 3GPP specs Test Set 4 */
        /*input*/
        {0xD0, 0xA7, 0xD4, 0x63, 0xDF, 0x9F, 0xB2, 0xB2, 0x78, 0x83,
         0x3F, 0xA0, 0x2E, 0x23, 0x5A, 0xA1, 0x72, 0xBD, 0x97, 0x0C,
         0x14, 0x73, 0xE1, 0x29, 0x07, 0xFB, 0x64, 0x8B, 0x65, 0x99,
         0xAA, 0xA0, 0xB2, 0x4A, 0x03, 0x86, 0x65, 0x42, 0x2B, 0x20,
         0xA4, 0x99, 0x27, 0x6A, 0x50, 0x42, 0x70, 0x09},
        /*length*/
        384,
        /*key*/
        {0xC7, 0x36, 0xC6, 0xAA, 0xB2, 0x2B, 0xFF, 0xF9, 0x1E, 0x26, 0x98, 0xD2,
         0xE2, 0x2A, 0xD5, 0x7E},
        /*KeyLeninBytes*/
        16,
        /*exp out*/
        {0xDD, 0x7D, 0xFA, 0xDD},
        /*iv*/
        {0x14, 0x79, 0x3E, 0x41, 0x03, 0x97, 0xE8, 0xFD},
        /*ivLeninbytes*/
        8,
        /*direction*/
        1,
    },
    {
        /* 3GPP specs Test Set 5 */
        /*input*/
        {0x10, 0xBF, 0xFF, 0x83, 0x9E, 0x0C, 0x71, 0x65, 0x8D, 0xBB, 0x2D, 0x17,
         0x07, 0xE1, 0x45, 0x72, 0x4F, 0x41, 0xC1, 0x6F, 0x48, 0xBF, 0x40, 0x3C,
         0x3B, 0x18, 0xE3, 0x8F, 0xD5, 0xD1, 0x66, 0x3B, 0x6F, 0x6D, 0x90, 0x01,
         0x93, 0xE3, 0xCE, 0xA8, 0xBB, 0x4F, 0x1B, 0x4F, 0x5B, 0xE8, 0x22, 0x03,
         0x22, 0x32, 0xA7, 0x8D, 0x7D, 0x75, 0x23, 0x8D, 0x5E, 0x6D, 0xAE, 0xCD,
         0x3B, 0x43, 0x22, 0xCF, 0x59, 0xBC, 0x7E, 0xA8, 0x4A, 0xB1, 0x88, 0x11,
         0xB5, 0xBF, 0xB7, 0xBC, 0x55, 0x3F, 0x4F, 0xE4, 0x44, 0x78, 0xCE, 0x28,
         0x7A, 0x14, 0x87, 0x99, 0x90, 0xD1, 0x8D, 0x12, 0xCA, 0x79, 0xD2, 0xC8,
         0x55, 0x14, 0x90, 0x21, 0xCD, 0x5C, 0xE8, 0xCA, 0x03, 0x71, 0xCA, 0x04,
         0xFC, 0xCE, 0x14, 0x3E, 0x3D, 0x7C, 0xFE, 0xE9, 0x45, 0x85, 0xB5, 0x88,
         0x5C, 0xAC, 0x46, 0x06, 0x8B},
        /*length*/
        1000,
        /*key*/
        {0xF4, 0xEB, 0xEC, 0x69, 0xE7, 0x3E, 0xAF, 0x2E, 0xB2, 0xCF, 0x6A, 0xF4,
         0xB3, 0x12, 0x0F, 0xFD},
        /*KeyLeninBytes*/
        16,
        /*exp out*/
        {0xC3, 0x83, 0x83, 0x9D},
        /*iv*/
        {0x29, 0x6F, 0x39, 0x3C, 0x6B, 0x22, 0x77, 0x37},
        /*ivLeninbytes*/
        8,
        /*direction*/
        1,
    } };

const cipher_test_vector_t *cipher_test_vectors[] = {kasumi_f8_vectors};
const uint32_t numCipherTestVectors[] = {
        sizeof(kasumi_f8_vectors) / sizeof(cipher_test_vector_t),
        sizeof(kasumi_f8_bitvectors) / sizeof(cipherbit_test_vector_t)};

const uint32_t numHashTestVectors[] = {
        sizeof(kasumi_f9_vectors) / sizeof(hash_test_vector_t),
        sizeof(kasumi_f9_IV_vectors) / sizeof(hash_iv_test_vector_t)};

#endif /*__KASUMI_TEST_VECTORS_H__*/
