/*--------------------------------------------------------------------------+
$Id: ConstraintValidatorTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.constraint;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import junit.framework.TestCase;
import edu.tum.cs.commons.error.CollectingExceptionHandler;
import edu.tum.cs.commons.string.StringUtils;
import edu.tum.cs.commons.visitor.VisitorUtilsTest;

/**
 * Tests for the {@link ConstraintValidator} class.
 * 
 * @author hummelb
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 4C6270202BE7838A8790725969B11E16
 */
public class ConstraintValidatorTest extends TestCase {

	/** The validator under test. */
	private ConstraintValidator validator;

	/** Fill the validator. */
	@SuppressWarnings("unchecked")
	@Override
	protected void setUp() {
		validator = new ConstraintValidator();

		// strings should not be empty
		validator.addConstraint(String.class, new ILocalConstraint<String>() {
			public void checkLocalConstraint(String s)
					throws ConstraintViolationException {
				if (StringUtils.isEmpty(s)) {
					throw new ConstraintViolationException(
							"String must not be empty", s);
				}
			}
		});

		// collections must not be empty
		validator.addConstraint(Collection.class,
				new ILocalConstraint<Collection>() {
					public void checkLocalConstraint(Collection c)
							throws ConstraintViolationException {
						if (c.isEmpty()) {
							throw new ConstraintViolationException(
									"Collection must not be empty", c);
						}
					}
				});

		// but lists must even contain more than one element
		validator.addConstraint(List.class, new ILocalConstraint<List>() {
			public void checkLocalConstraint(List l)
					throws ConstraintViolationException {
				if (l.size() <= 1) {
					throw new ConstraintViolationException(
							"Lists must have more than 1 element", l);
				}
			}
		});

		// and LinkedLists are disallowed
		validator.addConstraint(LinkedList.class,
				new ILocalConstraint<LinkedList>() {
					public void checkLocalConstraint(LinkedList l)
							throws ConstraintViolationException {
						throw new ConstraintViolationException(
								"Linked lists are disallowed", l);
					}
				});
	}

	/** Test very simple case with strings. */
	public void testSimple() throws Exception {
		validator.checkConstaints("bla");
		try {
			validator.checkConstaints("");
			fail();
		} catch (ConstraintViolationException e) {
			// expected
		}
	}

	/** Test the case where inheritance matters. */
	public void testWithInheritance() throws Exception {

		HashSet<Object> set = new HashSet<Object>();
		try {
			validator.checkConstaints(set);
			fail();
		} catch (ConstraintViolationException e) {
			// expected
		}

		set.add("r");
		validator.checkConstaints(set);

		List<String> l = new ArrayList<String>();
		l.add("foo");
		try {
			validator.checkConstaints(l);
			fail();
		} catch (ConstraintViolationException e) {
			// expected
		}

		l.add("bar");
		validator.checkConstaints(l);

		List<String> l2 = new LinkedList<String>(l);
		try {
			validator.checkConstaints(l2);
			fail();
		} catch (ConstraintViolationException e) {
			// expected
		}
	}

	/** Test validating an entire mesh. */
	public void testMeshed() {

		List<Object> root = new ArrayList<Object>();
		List<Object> ch1 = new ArrayList<Object>();
		List<Object> ch2 = new ArrayList<Object>();
		List<Object> ch3 = new ArrayList<Object>();
		root.add(ch1);
		root.add(ch2);
		ch2.add(ch3);

		ch2.add("a");
		ch2.add("b");
		ch2.add("c");
		ch3.add(root);
		ch3.add("");
		ch3.add("e");

		CollectingExceptionHandler<ConstraintViolationException> handler = new CollectingExceptionHandler<ConstraintViolationException>();
		validator
				.validateMesh(root, new VisitorUtilsTest.ListWalker(), handler);

		// two from the empty list (empty collection _and_ list <= 1),
		// one empty string
		assertEquals(2 + 1, handler.getExceptions().size());
	}

}