/*
 * @(#)MeasureMark.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import net.sourceforge.groboutils.codecoverage.v2.logger.ICoverageLoggerConst;

import org.apache.bcel.generic.INVOKESTATIC;
import org.apache.bcel.generic.InstructionList;
import org.apache.bcel.generic.LDC;
import org.apache.bcel.generic.SIPUSH;

/**
 * Exactly one mark for a specific measure on an instruction.  This class is
 * tightly coupled with the ParseCoverageLogger class, in that the signatures
 * must match between the two.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:25 $
 * @since     December 17, 2002
 */
class MeasureMark
{
    private static final Class COVERAGE_SIGNATURE[] =
        ICoverageLoggerConst.COVERAGE_SIGNATURE;
    
    
    /**
     * constant pool index for the name of the class's signature.
     */
    private final int classSigPoolIndex;
    
    /**
     * constant pool index for the method-ref for invoking the logger.
     */
    private final int staticMethodPoolIndex;
    
    /**
     * Reference to the owning method's index
     */
    private final short methodIndex;
    
    
    private final short measureIndex;
    private final short markIndex;
    
    /*
    Java language code for logger:
    
        net.sourceforge.groboutils.codecoverage.v2.logger.CoverageLogger.
            cover( classSig, methodIndex, channel, markIndex );
    
    Java bytecode calls for the logger:
    (this is why we use shorts, not ints!!!!)
      
        LDC     [class sig constant pool index]
        SIPUSH  [methodIndex]
        SIPUSH  [channel]
        SIPUSH  [markIndex]
        INVOKESTATIC ["net.sourceforge.groboutils.codecoverage.v2.logger.
            CoverageLogger.cover (Ljava.lang.String;SSS)V
    */
    
    /**
     * @throws IllegalStateException if the class file has already been
     *         modified (identified by a class name field).
     */
    MeasureMark( int classSigPoolIndex, int staticMethodPoolIndex,
            short methodIndex, short measureIndex, short markIndex )
    {
        this.classSigPoolIndex = classSigPoolIndex;
        this.staticMethodPoolIndex = staticMethodPoolIndex;
        this.methodIndex = methodIndex;
        this.measureIndex = measureIndex;
        this.markIndex = markIndex;
    }
    
    
    public void addToInstructionList( InstructionList list )
    {
        list.append( new LDC( this.classSigPoolIndex ) );
        list.append( new SIPUSH( this.methodIndex ) );
        list.append( new SIPUSH( this.measureIndex ) );
        list.append( new SIPUSH( this.markIndex ) );
        list.append( new INVOKESTATIC( this.staticMethodPoolIndex ) );
    }
}

