/*
 * @(#)ClassRecordIO.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;

/**
 * Knows how to read and write a ClassRecord object.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 15, 2002
 */
class ClassRecordIO
{
    public ClassRecordIO()
    {
        // do nothing
    }
    
    
    /**
     * Write only the class data for a class record.
     */
    public void writeClass( ClassRecord cr, Writer out )
            throws IOException
    {
        StringBuffer sb = new StringBuffer();
        
        sb.append( cr.getClassName().length() ).
            append( ':' ).
            append( cr.getClassName() ).
            append( ',' ).
            append( cr.getClassCRC() ).
            append( ',' ).
            append( cr.getSourceFileName().length() ).
            append( ':' ).
            append( cr.getSourceFileName() ).
            append( ',' );
        
        int count = cr.getMethodCount();
        sb.append( count ).append( '[' );
        for (short i = 0; i < count; ++i)
        {
            sb.append( '(' );
            String m = cr.getMethodAt( i );
            sb.append( m.length() ).
                append( ':' ).
                append( m );
            sb.append( ')' );
        }
        sb.append( ']' );
        
        out.write( sb.toString() );
    }
    
    
    /**
     * Read only the class data for a class record.
     */
    public ClassRecord readClass( AnalysisModuleSet ams, Reader in )
            throws IOException
    {
        int count = ReadUtil.toInt( ReadUtil.readTo( in, ':' ) );
        String className = ReadUtil.readCount( in, count );
        ReadUtil.readTo( in, ',' );
        long classCRC = ReadUtil.toLong( ReadUtil.readTo( in, ',' ) );
        count = ReadUtil.toInt( ReadUtil.readTo( in, ':' ) );
        String sourceFileName = ReadUtil.readCount( in, count );
        ReadUtil.readTo( in, ',' );
        
        int methodCount = ReadUtil.toInt( ReadUtil.readTo( in, '[' ) );
        String[] methodSigs = new String[ methodCount ];
        for (int i = 0; i < methodCount; ++i)
        {
            ReadUtil.readTo( in, '(' );
            count = ReadUtil.toInt( ReadUtil.readTo( in, ':' ) );
            methodSigs[i] = ReadUtil.readCount( in, count );
            ReadUtil.readTo( in, ')' );
        }
        ReadUtil.readTo( in, ']' );
        
        return new ClassRecord( className, classCRC, sourceFileName,
            methodSigs, ams );
    }
    
    
    //---------------------------------------------------------------------
    
    
    /**
     * Writes the set of marks for the given analysis module.
     */
    public void writeMarksForAnalysisModule( ClassRecord cr,
            IAnalysisModule am, Writer out )
            throws IOException
    {
        writeMarksForMeasure( cr, am.getMeasureName(), out );
    }
    
    
    /**
     * Writes the set of marks for the given analysis module.
     */
    public void writeMarksForMeasure( ClassRecord cr,
            String measureName, Writer out )
            throws IOException
    {
        MarkRecord mr[] = cr.getMarksForAnalysisModule( measureName );
        StringBuffer sb = new StringBuffer();
        sb.append( measureName.length() ).
            append( ':' ).
            append( measureName ).
            append( ',' ).
            append( mr.length ).
            append( '[' );
        for (int i = 0; i < mr.length; ++i)
        {
            StringWriter sw = new StringWriter();
            writeMark( mr[i], sw );
            
            String text = sw.toString();
            sb.append( text.length() ).
                append( '(' ).
                append( text ).
                append( ')' );
        }
        sb.append( ']' );
        
        out.write( sb.toString() );
    }
    
    
    public void readMarks( ClassRecord cr, Reader in )
            throws IOException
    {
        AnalysisModuleSet ams = cr.getAnalysisModuleSet();
        
        int count = ReadUtil.toInt( ReadUtil.readTo( in, ':' ) );
        String measureName = ReadUtil.readCount( in, count );
        ReadUtil.readTo( in, ',' );
        int markCount = ReadUtil.toInt( ReadUtil.readTo( in, '[' ) );
        for (int i = 0; i < markCount; ++i)
        {
            count = ReadUtil.toInt( ReadUtil.readTo( in, '(' ) );
            StringReader sr = new StringReader(
                ReadUtil.readCount( in, count ) );
            MarkRecord mr = readMark( cr, ams, sr );
            cr.addMark( mr );
            ReadUtil.readTo( in, ')' );
        }
        ReadUtil.readTo( in, ']' );
    }
    
    
    
    //-----------------------------------------------------------------------
    
    
    public void writeMark( MarkRecord mr, Writer out )
            throws IOException
    {
        short amIndex = mr.getAnalysisModuleIndex();
        short methodIndex = mr.getMethodIndex();
        short markIndex = mr.getMarkIndex();
        int lineNumber = mr.getLineNumber();
        IAnalysisMetaData amd = mr.getAnalysisMetaData();
        
        StringBuffer sb = new StringBuffer();
        sb.append( amIndex ).
            append( ',' ).
            append( methodIndex ).
            append( ',' ).
            append( markIndex ).
            append( ',' ).
            append( lineNumber ).
            append( ',' );
        
        StringWriter sw = new StringWriter();
        AnalysisMetaDataIO amdw = new AnalysisMetaDataIO();
        amdw.writeAnalysisMetaData( amd, sw );
        String text = sw.toString();
        sb.append( text.length() ).
            append( ':' ).
            append( text ).
            append( ',' );
        
        out.write( sb.toString() );
    }
    
    
    public MarkRecord readMark( ClassRecord cr, AnalysisModuleSet ams,
            Reader in )
            throws IOException
    {
        short amIndex = (short)ReadUtil.toInt( ReadUtil.readTo( in, ',' ) );
        short methodIndex = (short)ReadUtil.toInt( ReadUtil.readTo( in, ',' ) );
        short markIndex = (short)ReadUtil.toInt( ReadUtil.readTo( in, ',' ) );
        int lineNumber = ReadUtil.toInt( ReadUtil.readTo( in, ',' ) );
        int count = ReadUtil.toInt( ReadUtil.readTo( in, ':' ) );
        StringReader sr = new StringReader( ReadUtil.readCount( in, count ) );
        AnalysisMetaDataIO amdr = new AnalysisMetaDataIO();
        IAnalysisMetaData amd = amdr.readAnalysisMetaData( sr );
        ReadUtil.readTo( in, ',' );
        
        return new MarkRecord( amd,
            ams.getAnalysisModuleAt( amIndex ).getMeasureName(),
            cr.getMethodAt( methodIndex ),
            markIndex, lineNumber );
    }
}

