/*
 * @(#)DefaultMethodCodeUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import junit.framework.Test;
import junit.framework.TestCase;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BCELCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCodeUTestI;
import net.sourceforge.groboutils.codecoverage.v2.datastore.AnalysisModuleSet;
import net.sourceforge.groboutils.codecoverage.v2.datastore.ClassRecord;
import net.sourceforge.groboutils.junit.v1.iftc.CxFactory;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestSuite;

import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.MethodGen;


/**
 * Tests the DefaultMethodCode class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 11, 2003
 */
public class DefaultMethodCodeUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = DefaultMethodCodeUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public DefaultMethodCodeUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        try
        {
            new DefaultMethodCode( (short)0, null, null );
            fail( "Did not throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor2()
            throws Exception
    {
        try
        {
            new DefaultMethodCode( (short)0, createModifiedMethod(), null );
            fail( "Did not throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor3()
            throws Exception
    {
        try
        {
            new DefaultMethodCode( (short)0, null, createClassRecord(
                createModifiedMethod() ) );
            fail( "Did not throw IllegalArgumentException" );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor4()
            throws Exception
    {
        ModifiedMethod mm = createModifiedMethod();
        new DefaultMethodCode( (short)0, mm, createClassRecord( mm ) );
    }
    
    
    public void testConstructor5() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( GCI1.class, "m" );
        try
        {
            new DefaultMethodCode( (short)0, mm,
                createClassRecord( mm ) );
            fail( "Did not throw an IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            assertTrue(
                "Did not mention the abstract method.",
                ise.getMessage().indexOf( "abstract method" ) >= 0 );
        }
    }
    
    
    public void testGetOriginalMethod1()
            throws Exception
    {
        ModifiedMethod mm = createModifiedMethod();
        DefaultMethodCode dmc = new DefaultMethodCode( (short)0, mm,
            createClassRecord( mm ) );
        assertNotNull(
            "Returned null original method.",
            dmc.getOriginalMethod() );
        assertSame(
            "Not the same original method.",
            mm.getOriginalMethod(),
            dmc.getOriginalMethod() );
    }
    
    
    public void testGetMethodName1()
            throws Exception
    {
        ModifiedMethod mm = createModifiedMethod();
        Method m = mm.getOriginalMethod();
        DefaultMethodCode dmc = new DefaultMethodCode( (short)0, mm,
            createClassRecord( mm ) );
        assertNotNull(
            "Returned null method name.",
            dmc.getMethodName() );
        assertEquals(
            "Not the same original method.",
            m.getName()+m.getSignature(),
            dmc.getMethodName() );
    }
    
    
    public void testGetClassName1()
            throws Exception
    {
        ModifiedMethod mm = createModifiedMethod();
        DefaultMethodCode dmc = new DefaultMethodCode( (short)0, mm,
            createClassRecord( mm ) );
        assertNotNull(
            "Returned null class name.",
            dmc.getClassName() );
        assertSame(
            "Not the same original method.",
            mm.getOriginalClass().getClassName(),
            dmc.getClassName() );
    }
    
    
    // the other methods are sufficiently tested by IMethodCodeUTestI
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected static ModifiedMethod createModifiedMethod()
            throws Exception
    {
        return CCCreatorUtil.createModifiedMethod( THIS_CLASS, 0 );
    }
    
    
    protected static ModifiedMethod getModifiedMethod( Class c,
            String methodName ) throws Exception
    {
        // return the first method with the given name (not sig)
        
        ModifiedClass mc = CCCreatorUtil.createModifiedClass( c );
        ModifiedMethod mmL[] = mc.getMethods();
        
        for (int i = 0; i < mmL.length; ++i)
        {
            if (methodName.equals( mmL[i].getOriginalMethod().getName() ))
            {
                return mmL[i];
            }
        }
        fail( "Could not find method named '"+methodName+"' in class "+c+"." );
        // for the fans...
        throw new Exception();
    }
    
    
    protected static ModifiedMethod createModifiedMethod( Class c,
            String methodName ) throws Exception
    {
        // This is really difficult now, because the non-modifiable
        // methods are not returned by ModifierClass.
        JavaClass jc = BCELCreatorUtil.createJavaClass( c );
        Method mL[] = jc.getMethods();
        for (int i = 0; i < mL.length; ++i)
        {
            if (methodName.equals( mL[i].getName()))
            {
                Method m = mL[i];
                MethodGen mg = BCELCreatorUtil.createMethodGen( jc, m );
                ModifiedMethod mm = CCCreatorUtil.createModifiedMethod( jc,
                    i, m, mg );
                return mm;
            }
        }
        fail( "Could not find method named '"+methodName+"' in class "+c+"." );
        // for the fans...
        throw new Exception();
    }
    
    
    protected static AnalysisModuleSet createAnalysisModuleSet()
    {
        IAnalysisModule amL[] = new IAnalysisModule[] {
                CCCreatorUtil.createIAnalysisModule( 0 ),
                CCCreatorUtil.createIAnalysisModule( 1 ),
                CCCreatorUtil.createIAnalysisModule( 2 ),
            };
        return CCCreatorUtil.createAnalysisModuleSet( amL );
    }
    
    
    protected static ClassRecord createClassRecord( ModifiedMethod mm )
    {
        return CCCreatorUtil.createClassRecord( THIS_CLASS, mm,
            createAnalysisModuleSet() );
    }

    private static abstract class GCI1
    {
        public abstract void m();
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    public static Test suite()
    {
        InterfaceTestSuite suite = IMethodCodeUTestI.suite();
        suite.addTestSuite( THIS_CLASS );
        suite.addFactory( new CxFactory( "A" ) {
            public Object createImplObject() throws Exception {
                ModifiedMethod mm = createModifiedMethod();
                return new DefaultMethodCode( (short)0, mm,
                    createClassRecord( mm ) );
            }
        } );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

