/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C++ code and copyright notice are from the [Boost library]{@link http://www.boost.org/doc/libs/1_65_0/boost/math/special_functions/trigamma.hpp}. The implementation follows the original but has been reformatted and modified for JavaScript.
*
* ```text
* (C) Copyright John Maddock 2006.
*
* Use, modification and distribution are subject to the
* Boost Software License, Version 1.0. (See accompanying file
* LICENSE or copy at http://www.boost.org/LICENSE_1_0.txt)
* ```
*/

'use strict';

// MODULES //

var floor = require( './../../../../base/special/floor' );
var sinpi = require( './../../../../base/special/sinpi' );
var PI_SQUARED = require( '@stdlib/constants/float64/pi-squared' );
var rateval12 = require( './rational_p12q12.js' );
var rateval24 = require( './rational_p24q24.js' );
var rateval48 = require( './rational_p48q48.js' );
var rateval816 = require( './rational_p816q816.js' );
var rateval16INF = require( './rational_p16infq16inf.js' );


// VARIABLES //

var YOFFSET24 = 3.558437347412109375;


// MAIN //

/**
* Evaluates the trigamma function.
*
* @param {number} x - input value
* @returns {number} function value
*
* @example
* var v = trigamma( -2.5 );
* // returns ~9.539
*
* @example
* var v = trigamma( 1.0 );
* // returns ~1.645
*
* @example
* var v = trigamma( 10.0 );
* // returns ~0.105
*
* @example
* var v = trigamma( NaN );
* // returns NaN
*
* @example
* var v = trigamma( -1.0 );
* // returns NaN
*/
function trigamma( x ) {
	var result;
	var s;
	var y;
	var z;

	result = 0.0;

	// Check for negative arguments and use reflection:
	if ( x <= 0 ) {
		if ( floor( x ) === x ) {
			return NaN;
		}
		s = sinpi( x );
		z = 1.0 - x;
		return -trigamma( z ) + ( PI_SQUARED / ( s*s ) );
	}
	if ( x < 1.0 ) {
		result = 1.0 / ( x*x );
		x += 1.0;
	}
	if ( x <= 2.0 ) {
		result += ( 2.0+rateval12( x ) ) / ( x*x );
	}
	else if ( x <= 4.0 ) {
		result += ( YOFFSET24+rateval24( x ) ) / ( x*x );
	}
	else if ( x <= 8.0 ) {
		y = 1.0 / x;
		result += ( 1.0+rateval48( y ) ) / x;
	}
	else if ( x <= 16.0 ) {
		y = 1.0 / x;
		result += ( 1.0+rateval816( y ) ) / x;
	}
	else {
		y = 1.0 / x;
		result += ( 1.0+rateval16INF( y ) ) / x;
	}
	return result;
}


// EXPORTS //

module.exports = trigamma;
