#ifndef OSMIUM_OSM_DIFF_OBJECT_HPP
#define OSMIUM_OSM_DIFF_OBJECT_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <osmium/osm/item_type.hpp>
#include <osmium/osm/object.hpp>
#include <osmium/osm/timestamp.hpp>
#include <osmium/osm/types.hpp>

namespace osmium {

    class Node;
    class Way;
    class Relation;

    class DiffObject {

    protected:

        osmium::OSMObject* m_prev;
        osmium::OSMObject* m_curr;
        osmium::OSMObject* m_next;

    public:

        DiffObject() noexcept :
            m_prev(nullptr),
            m_curr(nullptr),
            m_next(nullptr) {
        }

        explicit DiffObject(osmium::OSMObject& prev, osmium::OSMObject& curr, osmium::OSMObject& next) noexcept :
            m_prev(&prev),
            m_curr(&curr),
            m_next(&next) {
        }

        DiffObject(const DiffObject&) = default;
        DiffObject& operator=(const DiffObject&) = default;

        DiffObject(DiffObject&&) = default;
        DiffObject& operator=(DiffObject&&) = default;

        const osmium::OSMObject& prev() const noexcept {
            return *m_prev;
        }

        const osmium::OSMObject& curr() const noexcept {
            return *m_curr;
        }

        const osmium::OSMObject& next() const noexcept {
            return *m_next;
        }

        bool first() const noexcept {
            return m_prev == m_curr;
        }

        bool last() const noexcept {
            return m_curr == m_next;
        }

        osmium::item_type type() const noexcept {
            return m_curr->type();
        }

        osmium::object_id_type id() const noexcept {
            return m_curr->id();
        }

        osmium::object_version_type version() const noexcept {
            return m_curr->version();
        }

        osmium::changeset_id_type changeset() const noexcept {
            return m_curr->changeset();
        }

        const osmium::Timestamp start_time() const noexcept {
            return m_curr->timestamp();
        }

        const osmium::Timestamp end_time() const noexcept {
            return last() ? osmium::Timestamp() : m_next->timestamp();
        }

    }; // class DiffObject

    template <class T>
    class DiffObjectDerived : public DiffObject {

    public:

        DiffObjectDerived(T& prev, T& curr, T& next) noexcept :
            DiffObject(prev, curr, next) {
        }

        DiffObjectDerived(const DiffObjectDerived&) = default;
        DiffObjectDerived& operator=(const DiffObjectDerived&) = default;

        DiffObjectDerived(DiffObjectDerived&&) = default;
        DiffObjectDerived& operator=(DiffObjectDerived&&) = default;

        const T& prev() const noexcept {
            return *static_cast<const T*>(m_prev);
        }

        const T& curr() const noexcept {
            return *static_cast<const T*>(m_curr);
        }

        const T& next() const noexcept {
            return *static_cast<const T*>(m_next);
        }

    }; // class DiffObjectDerived

    typedef DiffObjectDerived<osmium::Node>     DiffNode;
    typedef DiffObjectDerived<osmium::Way>      DiffWay;
    typedef DiffObjectDerived<osmium::Relation> DiffRelation;

} // namespace osmium

#endif // OSMIUM_OSM_DIFF_OBJECT_HPP
