/**
 * @brief Input Manager - handles internal and external data exchange
 *
 * This module contains the input manager. The input manager supports
 * the modules with communication functions to the outer world or 
 * between eachother. It is the router of information in pbbuttonsd.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/input_manager.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 *
 * @page inputmanager Input Manager
 *
 * The input manager is the spider in the information web. It collects
 * data from various sources and transmit it to receivers who are
 * interested in them.
 *
 * The basic mechanism to distribute information are message queues.
 * Following queues are available:
 *
 * <table class="pbb">
 * <tr><th>Queue-ID</th><th>Type of information/event</th></tr>
 * <tr><td>KBDQUEUE</td>
 *     <td>This queue distributes keyboard events. Each module which is
 *     interested in receiving keystrokes should register for this queue.
 *     Also laptop-specific hot keys would be sent here.</td></tr>
 * <tr><td>MOUSEQUEUE</td>
 *     <td>This queue distributes mouse events.</td></tr>
 * <tr><td>T100QUEUE</td>
 *     <td>All modules registered for this queue would be called every 100ms.</td></tr>
 * <tr><td>T1000QUEUE</td>
 *     <td>All modules registered for this queue would be called every second.</td></tr>
 * <tr><td>QUERYQUEUE</td>
 *     <td>This queue is used to collect data from modules.</td></tr>
 * <tr><td>CONFIGQUEUE</td>
 *     <td>This queue is used to configure modules or to change certain runtime
 *     variables.</td></tr>
 * <tr><td>SECUREQUEUE</td>
 *     <td>A module subscribed in this queue would only be called, if the user
 *     was idle. That means no key on the keyboard is currently pressed. This
 *     queue should only be used for tasks that would not work correctly if a
 *     key is still pressed on execution. On PowerMacs for example this queue
 *     is used for triggering sleep mode and configuring input devices like
 *     the keyboard itself. Pressed keys would interfere with the reconfiguration
 *     process so it would only be done if no key was pressed.</td></tr>
 * </table>
 * 
 * A module could request information from this queue by registering a callback
 * function which will be called as soon as a message is queued:
 *
 *   @li register_function()
 *
 * Each time the queue <i>queueid</i> was processed the callback will be called.
 * Arguments will be handed over in a taglist. The callback could now interpret
 * the taglist but shouldn't modify any tag, except it is especially asked for,
 * because the same taglist is subsequently passed to all registered callback
 * functions.
 *
 * At the moment there is no way to unregister a function from a queue.
 *
 * <h2>Module <-> Module data transfer</h2>
 *
 * If a module wanted to send data to other modules, it can use the following
 * functions:
 *   @li process_queue(), if multiple properties should be transfered
 *   @li process_queue_single(), if only a single property should be transfered
 *
 * To send properties to other modules the CONFIGQUEUE msut be used. If a
 * module needs information it asks other modules through the QUERYQUEUE.
 *
 * <h2>Other Input Sources</h2>
 *
 * Traditionally the input manager handles all input devices like keyboards,
 * mice, trackpads, etc. Events from these devices were read, preprocessed
 * and passed over to the interested receivers via queue described above.
 *
 * Since glib is used as basic library the input manager offers a highly
 * flexible system to handle any input stream comming from a file handle.
 *
 * For example if a module needs input from a socket, it could open and
 * initialize the socket and register it as InputSource. The input handler
 * will take care about the socket and inform the module when there is
 * something to read.
 *
 * File handles will be registered as input source with
 *   @li addInputSource();
 *
 * <h2>Input Handler</h2>
 *
 * The input manager has four predefined input handler to process
 * incomming events:
 *
 *   @li input_event_handler(), process input events from kernel
 *   @li ipc_handler(), process input from pbbuttonsd clients
 *   @li cb_timer100(), process timer event every 100ms
 *   @li cb_timer1000(), process timer event  every second
 *
 * Please see the handler documentation for detailed information about
 * the function.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "debug.h"

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "pbbinput.h"

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "systems.h"
#include "init.h"
#include "input_manager.h"

extern volatile sig_atomic_t prgexit;

/**
 * @brief Private data of module input manager
 */
struct moddata_inputmanager {
	GList	*inputsources;  /* list of input sources */
	struct {
		unsigned int shift:1;
		unsigned int ctrl:1;
		unsigned int alt:1;
		unsigned int :0;
	} flags;
	unsigned short lastkey;
	inputqueue_t *iqueues[QUEUECOUNT];
	inputqueue_t kbdqueue[MODULECOUNT];
	inputqueue_t mousequeue[MODULECOUNT];
	inputqueue_t t100queue[MODULECOUNT];
	inputqueue_t t1000queue[MODULECOUNT];
	inputqueue_t queryqueue[MODULECOUNT];
	inputqueue_t configqueue[MODULECOUNT];
	inputqueue_t securequeue[MODULECOUNT];
} modbase_inputmanager;

/**
 * @name Private Functions
 * @{
 */

/**
 * @brief Initalization of the Input manager
 *
 * This function sets up the internal strutures of the input manager.
 *
 * @return  error code, but this function can't fail and always returns
 *          zero.
 */
int
inputmanager_init ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	int x, y;

	base->inputsources = NULL;
	scanEventDevices();

#if 0    /* see mice_handler for comment */
	if ((fd = open("/dev/input/mice", O_RDONLY)) >= 0)  /* open mouse device */
		addInputSource (fd, mice_handler, NULL, TRUE);
#endif

	base->flags.shift = 0;
	base->flags.ctrl = 0;
	base->flags.alt = 0;
	base->lastkey = 0;

	base->iqueues[KBDQUEUE] = base->kbdqueue;
	base->iqueues[MOUSEQUEUE] = base->mousequeue;
	base->iqueues[T100QUEUE] = base->t100queue;
	base->iqueues[T1000QUEUE] = base->t1000queue;
	base->iqueues[QUERYQUEUE] = base->queryqueue;
	base->iqueues[CONFIGQUEUE] = base->configqueue;
	base->iqueues[SECUREQUEUE] = base->securequeue;

	for (y=0; y < QUEUECOUNT; y++)
		for (x=0; x < MODULECOUNT; x++)
			base->iqueues[y][x] = NULL;

	return 0;
}

/**
 * @brief  Callback to free an Glib input source
 *
 * This function removed a input source from the main context.
 * That means the attached file handle is no longer monitored.
 *
 * The function addInputSource() has already reduced the reference
 * counter of the IOChannel so that it will automatically removed
 * too after this action. At the end the complete InputSource was
 * freed.
 *
 * @see addInputSource()
 *
 * @param  data       Pointer to a InputSource structure
 * @param  user_data  Standard callback parameter. Not used in 
 *                    this function.
 */
void
cbFreeInputSource (gpointer data, gpointer user_data)
{
	InputSource *src = data;
	g_source_remove (src->watch);
}

/**
 * @brief  Frees all ressources allocated by the input manager
 */
void
inputmanager_exit ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;

	g_list_foreach (base->inputsources, cbFreeInputSource, NULL);
}

/**
 * @brief  Track the status of the modifier keys
 *
 * Pbbuttonsd reads keyboard events directly from the kernel
 * input layer. Each key will trigger an event containing the
 * keycode and the key status (pressed, released or repeat).
 * Two or more keys pressed together will result in two or more
 * events in sequence, even one of the keys is a modifier key.
 *
 * Because 'A' is not equal shift-'A', pbbuttonsd has to remember
 * which modifier keys were pressed when an 'A'-key event is
 * received. This function does this. It filters modifier keys out
 * of the event stream and stores their current status.
 *
 * @param  code   keycode from the event
 * @param  value  key status,
 *                @li 0 = released,
 *                @li 1 = pressed,
 *                @li 2 = repeat
 * @return modifier mask
 *         @li Bit 0 = shift is pressed
 *         @li Bit 1 = alt is pressed
 *         @li Bit 2 = ctrl is pressed
 *         One or more bits could be set.
 */
int
track_modifier(unsigned short code, unsigned int value)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;

	if (value > 1) value = 1;   /* key repeated = pressed */
	switch (code)
	{
		case KEY_RIGHTSHIFT:
		case KEY_LEFTSHIFT:
			base->flags.shift = value;
			break;
		case KEY_RIGHTCTRL:
		case KEY_LEFTCTRL:
			base->flags.ctrl = value;
			break;
		case KEY_RIGHTALT:
		case KEY_LEFTALT:
			base->flags.alt = value;
			break;
	}
	return (base->flags.ctrl << 2) | (base->flags.alt << 1) | (base->flags.shift << 0);
}

/**
 * @brief  Returns the current modifier mask
 *
 * This function returns the currently saved modifier mask.
 *
 * @return modifier mask
 */
int
get_modifier()
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  return (base->flags.ctrl << 2) | (base->flags.alt << 1) | (base->flags.shift << 0);
}

/* --- event handler management --- */

gint
cbEventDevices (gconstpointer a, gconstpointer venprod)
{
	const InputSource *src = a;
	
	if (src->user_data == venprod)
		return 0;
	return 1;
}
/*
 * The Input Manager will keep the list of input devices up-to-date and
 * connect any input device which is able to send the supported events
 * with it.
 */
void
scanEventDevices ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	int n, fd, fcnt = 0;
	char filename[20];
	gpointer evdevid;

	for (n = 0; n < EVDEVCOUNT; n++) {
		sprintf(filename, "/dev/input/event%d", n);
		if ((fd = open(filename, O_RDONLY)) >= 0) {
			evdevid = (gpointer) (0x45460000 | n);
			if ((g_list_find_custom (base->inputsources, evdevid, cbEventDevices)) == NULL)
				addInputSource (fd, input_event_handler, evdevid, TRUE);
			else
				close(fd);
		} else
			fcnt++;  /* count number of event devices that could not be opened */
	}

	if (fcnt == EVDEVCOUNT)  /* none of the 32 devices can be opened */
		print_msg (PBB_WARN, _("No event devices available. Please check your configuration.\n"));
}

/* --- input source --- */

void
destroyInputSource (gpointer data)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	InputSource *src = data;
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: InputSource removed: %08x\n", src->user_data);
#endif
	base->inputsources = g_list_remove (base->inputsources, src);
	g_free (src);
}

gboolean
handleInputSource (GIOChannel *io, GIOCondition condition, gpointer data)
{
	InputSource *src = data;
	gboolean rc = TRUE;
	
	if (condition & G_IO_IN) {
		/* call the appropriate handler. We assume that io = src->io.
		 * In case of an error the InputSource will be removed to
		 * prevent further problems reading invalid devices. */
		rc = src->handler (g_io_channel_unix_get_fd (io), src->user_data);
	} else
		/* GIOChannel reported an error */
		rc = FALSE;

#ifdef DEBUG
	if (rc == FALSE) {
		print_msg (PBB_INFO, "DBG: InputSource received %s %s %s\n",
				(condition & G_IO_IN)  ? "G_IO_IN"  : "",
		        (condition & G_IO_ERR) ? "G_IO_ERR" : "",
		        (condition & G_IO_HUP) ? "G_IO_HUP" : "");
	}
#endif
	return rc;
}
/* @} */

/**
 * @name Public API
 * @{
 */
InputSource*
addInputSource (int fd, int (*handler)(), gpointer user_data, gboolean close_on_exit)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	InputSource *src;

	src = g_new0 (InputSource, 1);
	src->io = g_io_channel_unix_new (fd);
	
	/* set channel encoding to binary */
	g_io_channel_set_encoding (src->io, NULL, NULL);

	/* Bound the file handle to the IOChannel. The file handle will be
	 * closed as soon as this IOChannel got obsolete. If this shouldn't
	 * happen, give 'close_on_exit' = FALSE. */
	if (close_on_exit == TRUE)
		g_io_channel_set_close_on_unref (src->io, TRUE);
	
	src->handler   = handler;
	src->user_data = user_data;
	src->watch   = g_io_add_watch_full (src->io, G_PRIORITY_DEFAULT,
			G_IO_IN | G_IO_ERR | G_IO_HUP, handleInputSource, src,
			destroyInputSource);

	/* g_io_channel_unix_new() and g_io_add_watch both increment the
	 * ref counter but we want to get rid of the IOchannel too, when
	 * the source was removed. */
	g_io_channel_unref (src->io);

	/* Attach the newly created InputSource to the global list */
	base->inputsources = g_list_append (base->inputsources, src);
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: InputSource added:   %08x\n", src->user_data);
#endif
	return src;
}

/* --- queue managers --- */

int
register_function (int queueid, void (*func)())
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  inputqueue_t *queue = base->iqueues[queueid];
  int n;

  for (n=0; n < MODULECOUNT; n++)
    if (queue[n] == NULL) {
      queue[n] = (inputqueue_t) func;
      return 0;
    }
  return -1;
}

long
process_queue_single (int queueid, long tag, long data)
{
  struct tagitem taglist[2];

  taglist_init (taglist);
  taglist_add (taglist, tag, data);
  process_queue (queueid, taglist);
  return taglist->tag & FLG_ERROR ? data : taglist->data;
}

int
process_queue (int queueid, struct tagitem *taglist)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	inputqueue_t *queue = base->iqueues[queueid];
	int n=0;

	if (queueid == KBDQUEUE) {
		if ((tagfind (taglist, TAG_KEYREPEAT, 0))) {
			base->lastkey = tagfind (taglist, TAG_KEYCODE, 0);
		} else
			base->lastkey = 0;
	}
	while ((queue[n] != NULL) && (n < MODULECOUNT)) {
		queue[n++] (taglist);
	}
	return n;
}
/* @} */

/**
 * @name Input Handlers
 * @{
 */

/**
 * @brief Process IPC messaged from pbbuttonsd clients
 *
 * This input handler is called every 100ms (from timer handler) and checks
 * if there are messages from clients pending. If so, the message would be
 * polled and according the action code in the message decides the message
 * splitter to route it either through the query queue or through the config
 * queue.
 *
 * In case of a query message it collects all data and sends it back to the
 * clients. If a config message was sent, nothing would be returned. The
 * client has to query the tags again to be sure that all configurations
 * were successful.
 */ 
void
ipc_handler ()
{
	struct pbbmessage *msg;
	struct tagitem *source, *dest;
	char msgbuffer[8192];

	if ((ipc_receive (msgbuffer, sizeof(msgbuffer))) == 0) {
		msg = (struct pbbmessage *) msgbuffer;
		switch (msg->action) {
		case READVALUE:
			process_queue (QUERYQUEUE, msg->taglist);
			ipc_send (msg->returnport, CHANGEVALUE, msg->taglist);
			break;
		case CHANGEVALUE:
			process_queue (CONFIGQUEUE, msg->taglist);
			source = dest = msg->taglist;
			while (source->tag != TAG_END) {
				if ((source->tag & FLG_ERROR)) {
					dest->tag = source->tag;
					dest->data = source->data;
					dest++;
				}
				source++;
			}
			dest->tag = TAG_END;
			dest->data = 0;
			ipc_send (msg->returnport, CHANGEERROR, msg->taglist);
			break;
		}
	}
}

/**
 * @brief  Process input events from kernel input system
 *
 * The input_event_handler() process events comming from /dev/input/event%.
 * Currently supported events are:
 *  @li keyboard events
 *  @li relative mouse events
 *  @li absolute mouse events
 * 
 * The input event handler will be called as soon as new events are available.
 * 
 * From <b>key events</b> the key code, the repeat indicator and the current
 * modifier mask is evaluated and sended them through the keyboard queue to
 * the modules. Furthermore the modifier keys <i>shift</i>, <i>alt</i> and
 * <i>ctrl</i> will be tracked.
 *
 * From <b>relative mouse events</b> the relative X and Y coordinates are
 * sent through the mouse queue to the modules. Mouse buttons are treated
 * as key events and handled by the keyboard section of the event handler.
 *
 * <b>Absolute mouse events</b> are only used to send a trigger signal to
 * the modules. No information from the event is used.
 *
 * @param fd         file handle to read the event from. It must be a open
 *                   filehandle to a /dev/input/event% device file.
 * @param user_data  not used in this function
 *
 * @return TRUE, if the event handler should be called again or FALSE if an
 *         error ocoured and the event file can't be read. In this case the
 *         event file will be removed from the input list and not used anymore.
 */
gboolean
input_event_handler (int fd, gpointer user_data)
{
	struct input_event inp;
	struct tagitem taglist[10];
	gboolean rc = FALSE;
	
	taglist_init(taglist);
	
	if (read(fd, &inp, sizeof(inp)) == sizeof(inp)) {
		switch (inp.type) {
		case EV_KEY:
			taglist_add (taglist, TAG_KEYCODE,  (long) inp.code);
			taglist_add (taglist, TAG_KEYREPEAT,(long) inp.value);
			taglist_add (taglist, TAG_MODIFIER, (long) track_modifier (inp.code, inp.value));
			process_queue (KBDQUEUE, taglist);
			break;
		case EV_REL:
			/* TAG_MOUSERELx are currently not used. Only the occourence
			 * of this events will be used */
			if (inp.code)
				taglist_add (taglist, TAG_MOUSERELY, (long) inp.value);
			else
				taglist_add (taglist, TAG_MOUSERELX, (long) inp.value);
    		process_queue (MOUSEQUEUE, taglist);
			break;
		case EV_ABS:
			/* Mouse events are not used. Only the occurence of this events
			 * will currently be used */
    		process_queue (MOUSEQUEUE, taglist);
			break;
		default:
			break; /* catch unknown event types */
		}
		rc = TRUE;
	}
	return rc;
}

/* This function is a tribute to the buggy Xorg synaptics driver.
 * This driver blocks the event device of the trackpad for exclusive
 * use so pbbuttonsd won't get any events from the trackpad. This
 * function uses /dev/input/mice in parallel to the event device
 * so we hopefully tick the synaptics driver.
 *
 * Unfortunately the synaptics driver pushed the mouse events read
 * from the event device directly to X and doesn't mirrot them to
 * /dev/event/mice. This makes it impossible to work around this
 * problem and make the following handler useless. (2006-09-23 MG)
 */
#if 0
gboolean
mice_handler (int fd, gpointer user_data)
{
	signed char buffer[] = {0, 0, 0 ,0};
	struct tagitem taglist[3];
	int count;

	taglist_init(taglist);
	if ((count = read(fd, buffer, sizeof(buffer))) >= 3) {
		taglist_add (taglist, TAG_MOUSERELX, (long) buffer[1]);
		taglist_add (taglist, TAG_MOUSERELY, (long) buffer[2]);
    	process_queue (MOUSEQUEUE, taglist);
	}
	return TRUE;
}
#endif

/**
 * @brief  Process timer event every 100ms
 *
 * There are two regulary timer functions:
 *  @li cb_timer100(), called every 100ms
 *  @li cb_timer1000(), called every second
 *
 * If a module need a faster timer, it is up to the module to install one
 * with the glib g_timeout_add() function. The module is responsible to
 * free any allocated ressources after the timer is no longer needed.
 *
 * The regulary timer functions send trigger signals through the timer
 * queues to the modules. 
 *
 * @see cb_timer1000()
 *
 * @param  data  Standard callback parameter, not used in this function
 * @return Always TRUE. This guarantee that glib continues to call this
 *         handler.
 */
gboolean
cb_timer100 (gpointer data)
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  struct tagitem taglist[] = { { TAG_END, 0 } };
  int mod;
  
  process_queue (T100QUEUE, taglist);
 
  mod = get_modifier();
  if ((mod == 0) && (base->lastkey == 0))
    process_queue (SECUREQUEUE, taglist);
  
  return TRUE;
}

/**
 * @brief  Process timer event every second
 *
 * @see cb_timer100()
 *
 * @param  data  Standard callback parameter, not used in this function
 * @return Always TRUE. This guarantee that glib continues to call this
 *         handler.
 */
gboolean
cb_timer1000 (gpointer data)
{
  struct tagitem taglist[] = { { TAG_END, 0 } };
  process_queue (T1000QUEUE, taglist);
  return TRUE;
}
/* @} */

