% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/removeAmbience.R
\name{removeAmbience}
\alias{removeAmbience}
\title{Remove the ambient profile}
\usage{
removeAmbience(
  y,
  ambient,
  groups,
  features = NULL,
  ...,
  size.factors = librarySizeFactors(y),
  dispersion = 0.1,
  sink = NULL,
  BPPARAM = SerialParam()
)
}
\arguments{
\item{y}{A numeric matrix-like object containing counts for each gene (row) and cell/library (column).}

\item{ambient}{A numeric vector of length equal to \code{nrow(y)},
containing the proportions of transcripts for each gene in the ambient solution.}

\item{groups}{A vector of length equal to \code{ncol(y)}, specifying the assigned group for each cell.
This can also be a \linkS4class{DataFrame}, see \code{?\link{sumCountsAcrossCells}}.}

\item{features}{A vector of control features or a list of mutually exclusive feature sets, 
see \code{?\link{controlAmbience}} for more details.}

\item{...}{Further arguments to pass to \code{\link{maximumAmbience}}.}

\item{size.factors}{Numeric scalar specifying the size factors for each column of \code{y},
defaults to library size-derived size factors.}

\item{dispersion}{Numeric scalar specifying the dispersion to use in the quantile-quantile mapping.}

\item{sink}{An optional \linkS4class{RealizationSink} object of the same dimensions as \code{y}.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization should be performed.}
}
\value{
A numeric matrix-like object of the same dimensions as \code{y},
containing the counts after removing the ambient contamination.
The exact representation of the output will depend on the class of \code{y} and whether \code{sink} was used.
}
\description{
Estimate and remove the ambient profile from a count matrix, given pre-existing groupings of similar cells.
This function is largely intended for plot beautification rather than real analysis.
}
\details{
This function will aggregate counts from each group of related cells into an average profile.
For each group, we estimate the contribution of the ambient profile and subtract it from the average.
By default, this is done with \code{\link{maximumAmbience}}, but if enough is known about the biological system, users can specify \code{feaures} to use \code{\link{controlAmbience}} instead.

We then perform quantile-quantile mapping of counts in \code{y} from the old to new averages.
This approach preserves the mean-variance relationship and improves the precision of estimate of the ambient contribution, but relies on a sensible grouping of similar cells, e.g., unsupervised clusters or cell type annotations.
As such, this function is best used at the end of the analysis to clean up expression matrices prior to visualization.
}
\examples{
# Making up some data.
ngenes <- 1000
ambient <- runif(ngenes, 0, 0.1)
cells <- c(runif(100) * 10, integer(900))
y <- matrix(rpois(ngenes * 100, cells + ambient), nrow=ngenes)

# Pretending that all cells are in one group, in this example.
removed <- removeAmbience(y, ambient, groups=rep(1, ncol(y)))
summary(rowMeans(removed[1:100,]))
summary(rowMeans(removed[101:1000,]))

}
\seealso{
\code{\link{maximumAmbience}} and \code{\link{controlAmbience}}, to estimate the ambient contribution.

\code{\link{estimateAmbience}}, to estimate the ambient profile.

The \pkg{SoupX} package, which provides another implementation of the same general approach.
}
\author{
Aaron Lun
}
