/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test handle-mapper managers.
 *
 * ti-handle-mapper.h:
 * Maps handle numbers to their respective data (like their names)
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Copyright (C) 2007-2008 Collabora Ltd <http://www.collabora.co.uk/>
 * Copyright (C) 2007-2008 Nokia Corporation
 *
 * Original author:
 *  Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef __TI_HANDLE_MAPPER_H__
#define __TI_HANDLE_MAPPER_H__

#include <glib-object.h>
#include <gtk/gtk.h>
#include <telepathy-glib/connection.h>
#include <telepathy-glib/enums.h>

typedef struct _TIHandleMapper TIHandleMapper;
typedef struct _TIHandleMapperClass TIHandleMapperClass;
typedef struct _TIHandleMapperPrivate TIHandleMapperPrivate;

/*
 * Type macros.
 */
#define TI_TYPE_HANDLE_MAPPER \
    (ti_handle_mapper_get_type ())
#define TI_HANDLE_MAPPER(object) \
    (G_TYPE_CHECK_INSTANCE_CAST ((object), TI_TYPE_HANDLE_MAPPER, \
        TIHandleMapper))
#define TI_HANDLE_MAPPER_CLASS(klass) \
    (G_TYPE_CHECK_CLASS_CAST ((klass), TI_TYPE_HANDLE_MAPPER, \
        TIHandleMapperClass))
#define TT_IS_HANDLE_MAPPER(object) \
    (G_TYPE_CHECK_INSTANCE_TYPE ((object), TI_TYPE_HANDLE_MAPPER))
#define TT_IS_HANDLE_MAPPER_CLASS(klass) \
    (G_TYPE_CHECK_CLASS_TYPE ((klass), TI_TYPE_HANDLE_MAPPER))
#define TI_HANDLE_MAPPER_GET_CLASS(object) \
    (G_TYPE_INSTANCE_GET_CLASS ((object), TI_TYPE_HANDLE_MAPPER, \
        TIHandleMapperClass))

struct _TIHandleMapper {
    GObject parent;
    TIHandleMapperPrivate *priv;
};

GType ti_handle_mapper_get_type (void);

TIHandleMapper *ti_handle_mapper_new (TpConnection *connection);

/**
 * Set Handle
 * Sets an unnamed handle
 */
void ti_handle_mapper_set_handle (TIHandleMapper *self,
    guint handle_type, guint handle_number);

void ti_handle_mapper_set_handle_name (TIHandleMapper *self,
    guint handle_type, guint handle_number, const gchar *name);

static inline void
ti_handle_mapper_set_contact_handle_name (TIHandleMapper *self,
                                          guint handle_number,
                                          const gchar *name)
{
  ti_handle_mapper_set_handle_name (self, TP_HANDLE_TYPE_CONTACT,
      handle_number, name);
}

static inline void
ti_handle_mapper_set_list_handle_name (TIHandleMapper *self,
                                       guint handle_number,
                                       const gchar *name)
{
  ti_handle_mapper_set_handle_name (self, TP_HANDLE_TYPE_LIST,
      handle_number, name);
}

static inline void
ti_handle_mapper_set_room_handle_name (TIHandleMapper *self,
                                       guint handle_number,
                                       const gchar *name)
{
  ti_handle_mapper_set_handle_name (self, TP_HANDLE_TYPE_ROOM,
      handle_number, name);
}

/**
 * Get Handle Name
 *
 * @return Handle's name or NULL, if not found. Must be free'd after use.
 */
gchar *ti_handle_mapper_get_handle_name (TIHandleMapper *self,
    guint handle_type, guint handle_number);

static inline gchar *
ti_handle_mapper_get_contact_handle_name (TIHandleMapper *self,
                                          guint handle_number)
{
  return ti_handle_mapper_get_handle_name (self, TP_HANDLE_TYPE_CONTACT,
      handle_number);
}

static inline gchar *
ti_handle_mapper_get_list_handle_name (TIHandleMapper *self,
                                       guint handle_number)
{
  return ti_handle_mapper_get_handle_name (self, TP_HANDLE_TYPE_LIST,
      handle_number);
}

static inline gchar *
ti_handle_mapper_get_room_handle_name (TIHandleMapper *self,
                                       guint handle_number)
{
  return ti_handle_mapper_get_handle_name (self, TP_HANDLE_TYPE_ROOM,
      handle_number);
}

/**
 * Invalidate Handle
 *
 * Destroy any info related to that handle
 */
void ti_handle_mapper_invalidate_handle (TIHandleMapper *self,
    guint handle_type, guint handle_number);

/**
 * Invalidate Contact Handles
 *
 * Destroy any info related to that handles
 */
void ti_handle_mapper_invalidate_contact_handles (TIHandleMapper *self,
    GArray *handle_numbers);

/**
 * Invalidate List Handles
 *
 * Destroy any info related to that handles
 */
void ti_handle_mapper_invalidate_list_handles (TIHandleMapper *self,
    GArray *handle_numbers);

/**
 * Get a GtkListStore containing all handles. This list is automatically updated and
 * shouldn't be changed.
 *
 * @return A GtkListStore. Must be unref'ed after use.
 */
GtkListStore *ti_handle_mapper_get_handles_list_store (TIHandleMapper *self);

/**
 * Get a GtkListStore containing all contact handles. This list is automatically updated and
 * shouldn't be changed.
 *
 * @return A GtkListStore. Must be unref'ed after use.
 */
GtkListStore *ti_handle_mapper_get_contact_handle_list_store
    (TIHandleMapper *self);

#endif
