/**************************************************************************
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 *
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this driver.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
/*
 */

#include "drmP.h"
#include "drm.h"
#include "psb_drm.h"
#include "psb_drv.h"
#include "psb_reg.h"
#include "i915_reg.h"
#include "drm_pciids.h"

int drm_psb_debug = 0;
EXPORT_SYMBOL(drm_psb_debug);
static int drm_psb_disable_cg = 1;
int drm_psb_no_fb = 0;

/*
 *
 */

static struct mutex drm_psb_plugin_mutex;
static spinlock_t psb_plugin_lock = SPIN_LOCK_UNLOCKED;
static int psb_plugin_usage = 0;
static struct drm_psb_plugin *psb_plug = NULL;
static LIST_HEAD(drm_psb_dev_list);
MODULE_PARM_DESC(debug, "Enable debug output");
MODULE_PARM_DESC(disable_clockgating, "Disable clock gating");
MODULE_PARM_DESC(no_fb, "Disable FBdev");
module_param_named(debug, drm_psb_debug, int, 0600);
module_param_named(disable_clockgating, drm_psb_disable_cg, int, 0600);
module_param_named(no_fb, drm_psb_no_fb, int, 0600);


struct drm_psb_plugin *psb_lock_plugin(int from_irq)
{
	struct drm_psb_plugin *tmp = NULL;
	unsigned long flags = 0;

	if (from_irq)
		spin_lock(&psb_plugin_lock);
	else
		spin_lock_irqsave(&psb_plugin_lock, flags);

	if (psb_plug) {
		psb_plugin_usage++;
		tmp = psb_plug;
	}

	if (from_irq)
		spin_unlock(&psb_plugin_lock);
	else
		spin_unlock_irqrestore(&psb_plugin_lock, flags);

	return tmp;
}

void psb_unlock_plugin(int from_irq)
{
	unsigned long flags;

	if (from_irq)
		spin_lock(&psb_plugin_lock);
	else
		spin_lock_irqsave(&psb_plugin_lock, flags);

	--psb_plugin_usage;

	if (from_irq)
		spin_unlock(&psb_plugin_lock);
	else
		spin_unlock_irqrestore(&psb_plugin_lock, flags);

}

static struct drm_psb_plugin *psb_set_plugin(struct drm_psb_plugin *plug)
{
	struct drm_psb_plugin *old_plug;
	unsigned long flags;

	spin_lock_irqsave(&psb_plugin_lock, flags);
	while(psb_plugin_usage > 0) {
		spin_unlock_irqrestore(&psb_plugin_lock, flags);
		msleep(1);
		spin_lock_irqsave(&psb_plugin_lock, flags);
	}
	old_plug = psb_plug;
	psb_plug = plug;
	spin_unlock_irqrestore(&psb_plugin_lock, flags);
	return old_plug;
}

void  psb_load_plugin(struct drm_psb_plugin *plug)
{
	struct dev_list_entry *entry;
	struct drm_psb_private * dev_priv;

	mutex_lock(&drm_psb_plugin_mutex);
	psb_set_plugin(NULL);
	list_for_each_entry(entry, &drm_psb_dev_list, head) {
		if (!entry->plugin_initialised) {
			dev_priv = (struct drm_psb_private *)entry->dev->dev_private;
			dev_priv->plug_priv = plug->init(entry->dev);
			entry->plugin_initialised = 1;
		}	
	}
	psb_set_plugin(plug);
	mutex_unlock(&drm_psb_plugin_mutex);
}
EXPORT_SYMBOL(psb_load_plugin);

void psb_unload_plugin(void)
{
	struct dev_list_entry *entry;
	struct drm_psb_private * dev_priv;
	struct drm_psb_plugin *plug;

	mutex_lock(&drm_psb_plugin_mutex);
	plug = psb_set_plugin(NULL);

	list_for_each_entry(entry, &drm_psb_dev_list, head) {
		if (entry->plugin_initialised) {
			dev_priv = (struct drm_psb_private *)entry->dev->dev_private;
			if (dev_priv->plug_priv) {
				plug->takedown(dev_priv->plug_priv);
				dev_priv->plug_priv = NULL;
			}				
			entry->plugin_initialised = 0;
		}	
	}
	mutex_unlock(&drm_psb_plugin_mutex);
}	
EXPORT_SYMBOL(psb_unload_plugin);

static struct pci_device_id pciidlist[] = {
	psb_PCI_IDS
};

static drm_ioctl_desc_t psb_ioctls[] = {
	[DRM_IOCTL_NR(DRM_PSB_CMDBUF)] = {psb_cmdbuf_ioctl, DRM_AUTH}
};
static int psb_max_ioctl = DRM_ARRAY_SIZE(psb_ioctls);

/*
 * Wrap the DRM ioctl dispatcher in order to efficiently support 
 * plugin ioctls.
 */

static int psb_ioctl(struct inode *inode, struct file *filp,
		     unsigned int cmd, unsigned long arg)
{
	drm_file_t *priv;
	drm_device_t *dev;
	drm_ioctl_t *func;
	int nr = DRM_IOCTL_NR(cmd);
	int retcode = -EINVAL;
	struct drm_psb_plugin *plug;

	if (NULL == (plug = psb_lock_plugin(0)))
		goto out_default;

	nr -= DRM_COMMAND_BASE + PSB_PLUG_IOCTL_START;
	if (nr < 0 || nr >= plug->num_ioctls) {
		psb_unlock_plugin(0);
		goto out_default;
	}
	
	priv = filp->private_data;
	dev = priv->head->dev;
	atomic_inc(&dev->ioctl_count);
	atomic_inc(&dev->counts[_DRM_STAT_IOCTLS]);
	++priv->ioctl_count;
	func = plug->ioctls[nr];

	if (!func) {
		DRM_DEBUG("No function.\n");
		retcode = -EINVAL;
	} else if (!priv->authenticated) {
		DRM_DEBUG("Not authenticated.\n");
		retcode = -EACCES;
	} else {
		retcode = func(inode, filp, cmd, arg);
	}
	psb_unlock_plugin(0);
	return retcode;
out_default:
	return drm_ioctl(inode, filp, cmd, arg);
}


static int probe(struct pci_dev *pdev, const struct pci_device_id *ent);

static int dri_library_name(struct drm_device *dev, char *buf)
{
	return snprintf(buf, PAGE_SIZE, "psb\n");
}

static void psb_set_uopt(struct drm_psb_uopt * uopt)
{
	uopt->disable_clock_gating = drm_psb_disable_cg;
}

static void psb_sgx_save(drm_psb_private_t * dev_priv)
{
	unsigned i;
	u32 *save = dev_priv->sgx_save;

	for (i = 0; i < PSB_SGX_SAVE_SIZE; i += 4) {
		*save++ = PSB_RSGX32(i);
	}
}

static void psb_sgx_restore(drm_psb_private_t * dev_priv)
{
	unsigned i;
	u32 *restore = dev_priv->sgx_save;

	for (i = 0; i < PSB_SGX_SAVE_SIZE; i += 4) {
		PSB_WSGX32(*restore++, i);
	}
}

void psb_clockgating(struct drm_psb_private *dev_priv)
{
	if (dev_priv->uopt.disable_clock_gating) {
		PSB_DEBUG_INIT("Disabling clock gating.\n");

		PSB_WSGX32((_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_2D_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_ISP_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_TSP_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_TA_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_DPM_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_DISABLED <<
			    _PSB_C_CLKGATECTL_USE_CLKG_SHIFT),
			   PSB_CR_CLKGATECTL);
	} else {
		PSB_DEBUG_INIT("Enabling clock gating.\n");

		PSB_WSGX32((_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_2D_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_ISP_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_TSP_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_TA_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_DPM_CLKG_SHIFT) |
			   (_PSB_C_CLKGATECTL_CLKG_AUTO <<
			    _PSB_C_CLKGATECTL_USE_CLKG_SHIFT),
			   PSB_CR_CLKGATECTL);

	}
	(void)PSB_RSGX32(PSB_CR_CLKGATECTL);
}

static int psb_driver_unload(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	struct drm_psb_plugin *plug;

	intel_modeset_cleanup(dev);

	mutex_lock(&dev->bm.init_mutex);
	mutex_lock(&dev->struct_mutex);
	if (dev->bm.initialized) {
		if (dev_priv->have_mem_mmu) {
			drm_bo_clean_mm(dev, DRM_PSB_MEM_MMU);
			dev_priv->have_mem_mmu = 0;
		}
		if (dev_priv->have_mem_aper) {
			drm_bo_clean_mm(dev, DRM_PSB_MEM_APER);
			dev_priv->have_mem_aper = 0;
		}
		if (dev_priv->have_tt) {
			drm_bo_clean_mm(dev, DRM_BO_MEM_TT);
			dev_priv->have_tt = 0;
		}
		if (dev_priv->have_vram) {
			drm_bo_clean_mm(dev, DRM_BO_MEM_VRAM);
			dev_priv->have_vram = 0;
		}
	}
	mutex_unlock(&dev->struct_mutex);
	mutex_unlock(&dev->bm.init_mutex);
	if (dev_priv) {
		if (dev_priv->comm) {
			kunmap(dev_priv->comm_page);
			dev_priv->comm = NULL;
		}
		if (dev_priv->comm_page) {
			__free_page(dev_priv->comm_page);
			dev_priv->comm_page = NULL;
		}
		mutex_lock(&drm_psb_plugin_mutex);
		list_del_init(&dev_priv->dev_list.head);
		if (NULL != (plug = psb_lock_plugin(0))) {
			if (dev_priv->plug_priv) {
				plug->takedown(dev_priv->plug_priv);
				dev_priv->plug_priv = NULL;
			}
			psb_unlock_plugin(0);
		}
		mutex_unlock(&drm_psb_plugin_mutex);
		if (dev_priv->have_mem_kernel) {
			drm_bo_clean_mm(dev, DRM_PSB_MEM_KERNEL);
			dev_priv->have_mem_kernel = 0;
		}
		if (dev_priv->pf_pd) {
			psb_mmu_free_pagedir(dev_priv->pf_pd);
			dev_priv->pf_pd = NULL;
		}
		if (dev_priv->mmu) {
			psb_mmu_driver_takedown(dev_priv->mmu);
			dev_priv->mmu = NULL;
		}
		psb_gtt_takedown(dev_priv->pg, 1);
		if (dev_priv->scratch_page) {
			__free_page(dev_priv->scratch_page);
			dev_priv->scratch_page = NULL;
		}
		if (dev_priv->sgx_save) {
			psb_sgx_restore(dev_priv);
			drm_free(dev_priv->sgx_save, PSB_SGX_SIZE, DRM_MEM_DRIVER);
			dev_priv->sgx_save = NULL;
		}
		if (dev_priv->vdc_reg) {
			iounmap(dev_priv->vdc_reg);
			dev_priv->vdc_reg = NULL;
		}
		if (dev_priv->sgx_reg) {
			iounmap(dev_priv->sgx_reg);
			dev_priv->sgx_reg = NULL;
		}
		drm_free(dev_priv, sizeof(*dev_priv), DRM_MEM_DRIVER);
		dev->dev_private = NULL;
	}
	return 0;
}

static int psb_driver_load(drm_device_t * dev, unsigned long chipset)
{
	drm_psb_private_t *dev_priv;
	const struct drm_psb_plugin *plug;
	unsigned long resource_start;
	struct psb_gtt *pg;
	u32 stolen_gtt;
	u32 tt_start;
	int tt_pages;
	int ret = -ENOMEM;

	dev_priv = drm_calloc(1, sizeof(*dev_priv), DRM_MEM_DRIVER);
	if (dev_priv == NULL)
		return -ENOMEM;

	mutex_init(&dev_priv->temp_mem);
	dev->dev_private = (void *)dev_priv;
	dev_priv->chipset = chipset;
	psb_set_uopt(&dev_priv->uopt);

	resource_start = pci_resource_start(dev->pdev, PSB_MMIO_RESOURCE);

	dev_priv->vdc_reg =
	    ioremap(resource_start + PSB_VDC_OFFSET, PSB_VDC_SIZE);
	if (!dev_priv->vdc_reg)
		goto out_err;

	dev_priv->sgx_reg =
	    ioremap(resource_start + PSB_SGX_OFFSET, PSB_SGX_SIZE);
	if (!dev_priv->sgx_reg)
		goto out_err;

	dev_priv->sgx_save = drm_calloc(1, PSB_SGX_SIZE, DRM_MEM_DRIVER);
	if (!dev_priv->sgx_save)
		goto out_err;

	psb_sgx_save(dev_priv);	
	psb_clockgating(dev_priv);

	dev_priv->scratch_page = alloc_page(GFP_DMA32 | __GFP_ZERO);
	if (!dev_priv->scratch_page)
		goto out_err;

	dev_priv->pg = psb_gtt_alloc(dev);
	if (!dev_priv->pg)
		goto out_err;

	ret = psb_gtt_init(dev_priv->pg, 0);
	if (ret)
		goto out_err;

	dev_priv->mmu = psb_mmu_driver_init(dev_priv->sgx_reg);
	if (!dev_priv->mmu)
		goto out_err;

	pg = dev_priv->pg;

	down_read(&pg->sem);
	psb_mmu_mirror_gtt(psb_mmu_get_default_pd(dev_priv->mmu),
			   pg->gatt_start, pg->gtt_phys_start, pg->gtt_pages);
	up_read(&pg->sem);

	dev_priv->pf_pd = psb_mmu_alloc_pd(dev_priv->mmu);
	if (!dev_priv->pf_pd)
		goto out_err;

	/*
	 * Make all presumably unused requestors page-fault by making them
	 * use context 1 which does not have any valid mappings.
	 */

	PSB_WSGX32(0x00000000, PSB_CR_BIF_BANK0);
	PSB_WSGX32(0x00000000, PSB_CR_BIF_BANK1);
	PSB_RSGX32(PSB_CR_BIF_BANK1);

	psb_mmu_set_pd_context(psb_mmu_get_default_pd(dev_priv->mmu), 0);
	psb_mmu_set_pd_context(dev_priv->pf_pd, 1);
	psb_mmu_enable_requestor(dev_priv->mmu, _PSB_MMU_ER_MASK);

	psb_init_2d(dev_priv);

	ret = drm_bo_driver_init(dev);
	if (ret)
		goto out_err;

	ret = drm_bo_init_mm(dev, DRM_PSB_MEM_KERNEL, 0x00000000,
			     (PSB_MEM_MMU_START - PSB_MEM_KERNEL_START) 
			     >> PAGE_SHIFT);
	if (ret) 
		goto out_err;
	
	
	dev_priv->have_mem_kernel = 1;
	dev_priv->dev_list.dev = dev;
	dev_priv->dev_list.plugin_initialised = 0;
	mutex_lock(&drm_psb_plugin_mutex);
	if ((plug = psb_lock_plugin(0))) {
		dev_priv->plug_priv = plug->init(dev);
		dev_priv->dev_list.plugin_initialised = 1;
		psb_unlock_plugin(0);
	}
	list_add_tail(&dev_priv->dev_list.head, &drm_psb_dev_list);
	mutex_unlock(&drm_psb_plugin_mutex);

	dev_priv->comm_page = alloc_page(GFP_KERNEL);
	if (!dev_priv->comm_page)
		goto out_err;

	dev_priv->comm = kmap(dev_priv->comm_page);
	memset((void *)dev_priv->comm, 0, PAGE_SIZE);

	if (pg->gatt_start & 0x0FFFFFFF) {
		DRM_ERROR("Gatt must be 256M aligned. This is a bug.\n");
		ret = -EINVAL;
		goto out_err;
	}
			
	stolen_gtt = (pg->stolen_size >> PAGE_SHIFT) * 4;
	stolen_gtt = (stolen_gtt + PAGE_SIZE - 1) >> PAGE_SHIFT;
	stolen_gtt = (stolen_gtt < pg->gtt_pages) ? stolen_gtt : pg->gtt_pages;

	dev_priv->gatt_free_offset = pg->gatt_start +
	    (stolen_gtt << PAGE_SHIFT) * 1024;

	/*
	 * Insert a cache-coherent communications page in mmu space
	 * just after the stolen area. Will be used for fencing etc.
	 */

	dev_priv->comm_mmu_offset = dev_priv->gatt_free_offset;
	dev_priv->gatt_free_offset += PAGE_SIZE;

	ret = psb_mmu_insert_pages(psb_mmu_get_default_pd(dev_priv->mmu),
				   &dev_priv->comm_page,
				   dev_priv->comm_mmu_offset, 1, 0, 0,
				   PSB_MMU_CACHED_MEMORY);

	if (ret)
		goto out_err;

	if (1||drm_debug) {
		u32 core_id = PSB_RSGX32(PSB_CR_CORE_ID);
		u32 core_rev = PSB_RSGX32(PSB_CR_CORE_REVISION);
		DRM_INFO("SGX core id = 0x%08x\n", core_id);
		DRM_INFO("SGX core rev major = 0x%02x, minor = 0x%02x\n",
			(core_rev & _PSB_CC_REVISION_MAJOR_MASK) >>
			_PSB_CC_REVISION_MAJOR_SHIFT,
			(core_rev & _PSB_CC_REVISION_MINOR_MASK) >>
			_PSB_CC_REVISION_MINOR_SHIFT);
		DRM_INFO("SGX core rev maintenance = 0x%02x, designer = 0x%02x\n",
			(core_rev & _PSB_CC_REVISION_MAINTENANCE_MASK) >>
			_PSB_CC_REVISION_MAINTENANCE_SHIFT,
			(core_rev & _PSB_CC_REVISION_DESIGNER_MASK) >>
			_PSB_CC_REVISION_DESIGNER_SHIFT);
	}


	dev_priv->irqmask_lock = SPIN_LOCK_UNLOCKED;
	dev_priv->fence0_irq_on = 0;

	/*
	 * We must align tt mem start to a GTT page start to avoid
	 * MMU page table conflicts.
	 */

	tt_pages = (pg->gatt_pages > PSB_TT_PRIV0_PLIMIT) ?
	    pg->gatt_pages : PSB_TT_PRIV0_PLIMIT;
	tt_start = PSB_ALIGN_TO(dev_priv->gatt_free_offset, PAGE_SIZE * 1024);
	tt_pages -= (tt_start - pg->gatt_start) >> PAGE_SHIFT;


	tt_pages = (pg->gatt_pages > PSB_TT_PRIV0_PLIMIT) ?
	    pg->gatt_pages : PSB_TT_PRIV0_PLIMIT;
	tt_start = PSB_ALIGN_TO(dev_priv->gatt_free_offset, PAGE_SIZE * 1024) -
	    pg->gatt_start;
	tt_pages -= tt_start >> PAGE_SHIFT;

	mutex_lock(&dev->bm.init_mutex);
	mutex_lock(&dev->struct_mutex);

	if (!drm_bo_init_mm(dev, DRM_BO_MEM_VRAM, 0,
			    pg->stolen_size >> PAGE_SHIFT)) {
		dev_priv->have_vram = 1;
	}

	if (!drm_bo_init_mm(dev, DRM_BO_MEM_TT, tt_start >> PAGE_SHIFT,
			    tt_pages)) {
		dev_priv->have_tt = 1;
	}

	if (!drm_bo_init_mm(dev, DRM_PSB_MEM_MMU, 0x00000000,
			    (pg->gatt_start - PSB_MEM_MMU_START) >> PAGE_SHIFT)) {
		dev_priv->have_mem_mmu = 1;
	}

	if (pg->gatt_pages > PSB_TT_PRIV0_PLIMIT) {
		if (!drm_bo_init_mm(dev, DRM_PSB_MEM_APER, PSB_TT_PRIV0_PLIMIT,
				    pg->gatt_pages - PSB_TT_PRIV0_PLIMIT)) {
			dev_priv->have_mem_aper = 1;
		}
	}

	mutex_unlock(&dev->struct_mutex);
	mutex_unlock(&dev->bm.init_mutex);

	intel_modeset_init(dev);
	drm_initial_config(dev, false);

	return 0;
out_err:
	psb_driver_unload(dev);	
	return ret;
}


int psb_driver_device_is_agp(drm_device_t * dev)
{
	return 0;
}

static int psb_suspend(struct pci_dev *pdev, pm_message_t state)
{
        drm_device_t *dev = pci_get_drvdata(pdev);
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	struct drm_output *output;
	int i;

	psb_gtt_takedown(dev_priv->pg, 0);

	pci_save_state(pdev);

	/* Save video mode information for native mode-setting. */
	dev_priv->saveDSPACNTR = I915_READ(DSPACNTR);
	dev_priv->savePIPEACONF = I915_READ(PIPEACONF);
	dev_priv->savePIPEASRC = I915_READ(PIPEASRC);
	dev_priv->saveFPA0 = I915_READ(FPA0);
	dev_priv->saveFPA1 = I915_READ(FPA1);
	dev_priv->saveDPLL_A = I915_READ(DPLL_A);
	dev_priv->saveHTOTAL_A = I915_READ(HTOTAL_A);
	dev_priv->saveHBLANK_A = I915_READ(HBLANK_A);
	dev_priv->saveHSYNC_A = I915_READ(HSYNC_A);
	dev_priv->saveVTOTAL_A = I915_READ(VTOTAL_A);
	dev_priv->saveVBLANK_A = I915_READ(VBLANK_A);
	dev_priv->saveVSYNC_A = I915_READ(VSYNC_A);
	dev_priv->saveDSPASTRIDE = I915_READ(DSPASTRIDE);
	dev_priv->saveDSPASIZE = I915_READ(DSPASIZE);
	dev_priv->saveDSPAPOS = I915_READ(DSPAPOS);
	dev_priv->saveDSPABASE = I915_READ(DSPABASE);

	for(i= 0; i < 256; i++)
		dev_priv->savePaletteA[i] = I915_READ(PALETTE_A + (i << 2));

	if(dev->mode_config.num_crtc == 2) {
		dev_priv->savePIPEBCONF = I915_READ(PIPEBCONF);
		dev_priv->savePIPEBSRC = I915_READ(PIPEBSRC);
		dev_priv->saveDSPBCNTR = I915_READ(DSPBCNTR);
		dev_priv->saveFPB0 = I915_READ(FPB0);
		dev_priv->saveFPB1 = I915_READ(FPB1);
		dev_priv->saveDPLL_B = I915_READ(DPLL_B);
		dev_priv->saveHTOTAL_B = I915_READ(HTOTAL_B);
		dev_priv->saveHBLANK_B = I915_READ(HBLANK_B);
		dev_priv->saveHSYNC_B = I915_READ(HSYNC_B);
		dev_priv->saveVTOTAL_B = I915_READ(VTOTAL_B);
		dev_priv->saveVBLANK_B = I915_READ(VBLANK_B);
		dev_priv->saveVSYNC_B = I915_READ(VSYNC_B);
		dev_priv->saveDSPBSTRIDE = I915_READ(DSPBSTRIDE);
		dev_priv->saveDSPBSIZE = I915_READ(DSPBSIZE);
		dev_priv->saveDSPBPOS = I915_READ(DSPBPOS);
		dev_priv->saveDSPBBASE = I915_READ(DSPBBASE);
		for(i= 0; i < 256; i++)
			dev_priv->savePaletteB[i] =
				I915_READ(PALETTE_B + (i << 2));
	}

	dev_priv->saveVCLK_DIVISOR_VGA0 = I915_READ(VCLK_DIVISOR_VGA0);
	dev_priv->saveVCLK_DIVISOR_VGA1 = I915_READ(VCLK_DIVISOR_VGA1);
	dev_priv->saveVCLK_POST_DIV = I915_READ(VCLK_POST_DIV);
	dev_priv->saveVGACNTRL = I915_READ(VGACNTRL);

	dev_priv->saveLVDS = I915_READ(LVDS);
	dev_priv->savePFIT_CONTROL = I915_READ(PFIT_CONTROL);

	list_for_each_entry(output, &dev->mode_config.output_list, head)
		if (output->funcs->save)
			(*output->funcs->save) (output);

#if 0 /* FIXME: save VGA bits */
	vgaHWUnlock(hwp);
	vgaHWSave(pScrn, vgaReg, VGA_SR_FONTS);
#endif
	pci_disable_device(pdev);
	pci_set_power_state(pdev, PCI_D3hot);

	return 0;
}

static int psb_resume(struct pci_dev *pdev)
{
        drm_device_t *dev = pci_get_drvdata(pdev);
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	struct drm_output *output;
	struct drm_crtc *crtc;
	int i;

	/* reset the rest */
	pci_set_power_state(pdev, PCI_D0);
	pci_restore_state(pdev);
	if (pci_enable_device(pdev))
		return -1;

	/* reset, may need more tweaking yet. */

	psb_gtt_init(dev_priv->pg, 1);

	PSB_WSGX32(0x00000000, PSB_CR_BIF_BANK0);
	PSB_WSGX32(0x00000000, PSB_CR_BIF_BANK1);
	PSB_RSGX32(PSB_CR_BIF_BANK1);

	psb_mmu_set_pd_context(psb_mmu_get_default_pd(dev_priv->mmu), 0);
	psb_mmu_set_pd_context(dev_priv->pf_pd, 1);

	psb_mmu_enable_requestor(dev_priv->mmu, _PSB_MMU_ER_MASK);

	psb_init_2d(dev_priv);

	/* Disable outputs */
	list_for_each_entry(output, &dev->mode_config.output_list, head)
		output->funcs->dpms(output, DPMSModeOff);

	msleep(60);
   
	/* Disable pipes */
	list_for_each_entry(crtc, &dev->mode_config.crtc_list, head)
		crtc->funcs->dpms(crtc, DPMSModeOff);

	/* FIXME: wait for vblank on each pipe? */
	msleep(60);

	I915_WRITE(LVDS, dev_priv->saveLVDS);
	I915_WRITE(PFIT_CONTROL, dev_priv->savePFIT_CONTROL);

	if (dev_priv->saveDPLL_A & DPLL_VCO_ENABLE) {
		I915_WRITE(DPLL_A, dev_priv->saveDPLL_A & ~DPLL_VCO_ENABLE);
		udelay(150);
	}
	I915_WRITE(FPA0, dev_priv->saveFPA0);
	I915_WRITE(FPA1, dev_priv->saveFPA1);
	I915_WRITE(DPLL_A, dev_priv->saveDPLL_A);
	udelay(150);
	I915_WRITE(DPLL_A, dev_priv->saveDPLL_A);
	udelay(150);

	I915_WRITE(HTOTAL_A, dev_priv->saveHTOTAL_A);
	I915_WRITE(HBLANK_A, dev_priv->saveHBLANK_A);
	I915_WRITE(HSYNC_A, dev_priv->saveHSYNC_A);
	I915_WRITE(VTOTAL_A, dev_priv->saveVTOTAL_A);
	I915_WRITE(VBLANK_A, dev_priv->saveVBLANK_A);
	I915_WRITE(VSYNC_A, dev_priv->saveVSYNC_A);
   
	I915_WRITE(DSPASTRIDE, dev_priv->saveDSPASTRIDE);
	I915_WRITE(DSPASIZE, dev_priv->saveDSPASIZE);
	I915_WRITE(DSPAPOS, dev_priv->saveDSPAPOS);
	I915_WRITE(PIPEASRC, dev_priv->savePIPEASRC);
	I915_WRITE(DSPABASE, dev_priv->saveDSPABASE);
	I915_WRITE(PIPEACONF, dev_priv->savePIPEACONF);
	msleep(60);
	I915_WRITE(DSPACNTR, dev_priv->saveDSPACNTR);
	I915_WRITE(DSPABASE, I915_READ(DSPABASE));
	msleep(60);
   
	if(dev->mode_config.num_crtc == 2) {
		if (dev_priv->saveDPLL_B & DPLL_VCO_ENABLE) {
			I915_WRITE(DPLL_B, dev_priv->saveDPLL_B & ~DPLL_VCO_ENABLE);
			udelay(150);
		}
		I915_WRITE(FPB0, dev_priv->saveFPB0);
		I915_WRITE(FPB1, dev_priv->saveFPB1);
		I915_WRITE(DPLL_B, dev_priv->saveDPLL_B);
		udelay(150);
		I915_WRITE(DPLL_B, dev_priv->saveDPLL_B);
		udelay(150);
   
		I915_WRITE(HTOTAL_B, dev_priv->saveHTOTAL_B);
		I915_WRITE(HBLANK_B, dev_priv->saveHBLANK_B);
		I915_WRITE(HSYNC_B, dev_priv->saveHSYNC_B);
		I915_WRITE(VTOTAL_B, dev_priv->saveVTOTAL_B);
		I915_WRITE(VBLANK_B, dev_priv->saveVBLANK_B);
		I915_WRITE(VSYNC_B, dev_priv->saveVSYNC_B);
		I915_WRITE(DSPBSTRIDE, dev_priv->saveDSPBSTRIDE);
		I915_WRITE(DSPBSIZE, dev_priv->saveDSPBSIZE);
		I915_WRITE(DSPBPOS, dev_priv->saveDSPBPOS);
		I915_WRITE(PIPEBSRC, dev_priv->savePIPEBSRC);
		I915_WRITE(DSPBBASE, dev_priv->saveDSPBBASE);
		I915_WRITE(PIPEBCONF, dev_priv->savePIPEBCONF);
		msleep(60);
		I915_WRITE(DSPBCNTR, dev_priv->saveDSPBCNTR);
		I915_WRITE(DSPBBASE, I915_READ(DSPBBASE));
		msleep(60);
	}

	/* Restore outputs */
	list_for_each_entry(output, &dev->mode_config.output_list, head)
		if (output->funcs->restore)
			output->funcs->restore(output);
    
	I915_WRITE(VGACNTRL, dev_priv->saveVGACNTRL);

	I915_WRITE(VCLK_DIVISOR_VGA0, dev_priv->saveVCLK_DIVISOR_VGA0);
	I915_WRITE(VCLK_DIVISOR_VGA1, dev_priv->saveVCLK_DIVISOR_VGA1);
	I915_WRITE(VCLK_POST_DIV, dev_priv->saveVCLK_POST_DIV);

	for(i = 0; i < 256; i++)
		I915_WRITE(PALETTE_A + (i << 2), dev_priv->savePaletteA[i]);
   
	if(dev->mode_config.num_crtc == 2)
		for(i= 0; i < 256; i++)
			I915_WRITE(PALETTE_B + (i << 2), dev_priv->savePaletteB[i]);

#if 0 /* FIXME: restore VGA bits */
	vgaHWRestore(pScrn, vgaReg, VGA_SR_FONTS);
	vgaHWLock(hwp);
#endif

	return 0;
}


static drm_fence_driver_t psb_fence_driver = {
	.num_classes = 1,
	.wrap_diff = (1 << 30),
	.flush_diff = (1 << 29),
	.sequence_mask = 0xFFFFFFFFU,
	.lazy_capable = 1,
	.emit = psb_fence_emit_sequence,
	.poke_flush = psb_poke_flush,
	.has_irq = psb_fence_has_irq,
};

/*
 * Use this memory type priority if no eviction is needed.
 */
static uint32_t psb_mem_prios[] = { DRM_BO_MEM_VRAM,
	DRM_BO_MEM_TT,
	DRM_BO_MEM_PRIV0,
	DRM_BO_MEM_PRIV1,
	DRM_BO_MEM_LOCAL
};

/*
 * Use this memory type priority if need to evict.
 */
static uint32_t psb_busy_prios[] = { DRM_BO_MEM_TT,
	DRM_BO_MEM_VRAM,
	DRM_BO_MEM_PRIV0,
	DRM_BO_MEM_PRIV1,
	DRM_BO_MEM_LOCAL
};

static drm_bo_driver_t psb_bo_driver = {
	.mem_type_prio = psb_mem_prios,
	.mem_busy_prio = psb_busy_prios,
	.num_mem_type_prio = ARRAY_SIZE(psb_mem_prios),
	.num_mem_busy_prio = ARRAY_SIZE(psb_busy_prios),
	.create_ttm_backend_entry = drm_psb_tbe_init,
	.fence_type = psb_fence_types,
	.invalidate_caches = psb_invalidate_caches,
	.init_mem_type = psb_init_mem_type,
	.evict_mask = psb_evict_mask,
	.move = psb_move
};

static struct drm_driver driver = {
	.driver_features = DRIVER_HAVE_IRQ | DRIVER_IRQ_SHARED | 
	DRIVER_IRQ_VBL | DRIVER_IRQ_VBL2,
	.load = psb_driver_load,
	.unload = psb_driver_unload,
	.dri_library_name = dri_library_name,
	.get_reg_ofs = drm_core_get_reg_ofs,
	.ioctls = psb_ioctls,
	.device_is_agp = psb_driver_device_is_agp,
	.irq_preinstall = psb_irq_preinstall,
	.irq_postinstall = psb_irq_postinstall,
	.irq_uninstall = psb_irq_uninstall,
	.irq_handler = psb_irq_handler,
	.fb_probe = psbfb_probe,
	.fb_remove = psbfb_remove,
	.fops = {
		 .owner = THIS_MODULE,
		 .open = drm_open,
		 .release = drm_release,
		 .ioctl = psb_ioctl,
		 .mmap = drm_mmap,
		 .poll = drm_poll,
		 .fasync = drm_fasync,
		 },
	.pci_driver = {
		       .name = DRIVER_NAME,
		       .id_table = pciidlist,
		       .probe = probe,
		       .remove = __devexit_p(drm_cleanup_pci),
		       .resume = psb_resume,
		       .suspend = psb_suspend,
		       },
	.fence_driver = &psb_fence_driver,
	.bo_driver = &psb_bo_driver,
	.name = DRIVER_NAME,
	.desc = DRIVER_DESC,
	.date = PSB_DRM_DRIVER_DATE,
	.major = PSB_DRM_DRIVER_MAJOR,
	.minor = PSB_DRM_DRIVER_MINOR,
	.patchlevel = PSB_DRM_DRIVER_PATCHLEVEL
};


static int probe(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	return drm_get_dev(pdev, ent, &driver);
}

static int __init psb_init(void)
{
	mutex_init(&drm_psb_plugin_mutex);
	driver.num_ioctls = psb_max_ioctl;
	
	return drm_init(&driver, pciidlist);
}

static void __exit psb_exit(void)
{
	drm_exit(&driver);
}

module_init(psb_init);
module_exit(psb_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("MIT/BSD style and exposing kernel to non-GPL plugins.");
