/**************************************************************************
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 *
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this driver.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
/*
 */
#ifndef _PSB_DRV_H_
#define _PSB_DRV_H_

#include "psb_drm.h"

enum {
	CHIP_PSB_8108 = 0
};

#define DRIVER_NAME "psb"
#define DRIVER_DESC "drm driver for the Intel Poulsbo chipset"
#define DRIVER_AUTHOR "Tungsten Graphics Inc."

#define PSB_DRM_DRIVER_DATE "20060307"
#define PSB_DRM_DRIVER_MAJOR 0
#define PSB_DRM_DRIVER_MINOR 1
#define PSB_DRM_DRIVER_PATCHLEVEL 0
#define PSB_VDC_OFFSET           0x00000000
#define PSB_VDC_SIZE             0x000080000
#define PSB_SGX_SAVE_SIZE        0x1000
#define PSB_SGX_SIZE             0x8000
#define PSB_SGX_OFFSET           0x00040000
#define PSB_MMIO_RESOURCE        0
#define PSB_GATT_RESOURCE        2
#define PSB_GTT_RESOURCE         3
#define PSB_GMCH_CTRL            0x52
#define PSB_BSM                  0x5C
#define _PSB_GMCH_ENABLED        0x4
#define PSB_PGETBL_CTL           0x2020
#define _PSB_PGETBL_ENABLED      0x00000001
#define PSB_SGX_2D_SLAVE_PORT    0x4000
#define PSB_TT_PRIV0_LIMIT       (256*1024*1024)
#define PSB_TT_PRIV0_PLIMIT      (PSB_TT_PRIV0_LIMIT >> PAGE_SHIFT)
#define PSB_NUM_VALIDATE_BUFFERS 256
#define PSB_MEM_MMU_START        0x20000000
#define PSB_MEM_KERNEL_START     0x10000000

#define DRM_PSB_MEM_KERNEL       DRM_BO_MEM_PRIV0
#define DRM_PSB_FLAG_MEM_KERNEL  DRM_BO_FLAG_MEM_PRIV0

/*
 * Flags for external memory type field.
 */

#define PSB_MMU_CACHED_MEMORY     0x0001	/* Bind to MMU only */
#define PSB_MMU_RO_MEMORY         0x0002	/* MMU RO memory */
#define PSB_MMU_WO_MEMORY         0x0004	/* MMU WO memory */

/*
 * PTE's and PDE's
 */

#define PSB_PDE_MASK              0x003FFFFF
#define PSB_PDE_SHIFT             22
#define PSB_PTE_SHIFT             12

#define PSB_PTE_VALID             0x0001	/* PTE / PDE valid */
#define PSB_PTE_WO                0x0002	/* Write only */
#define PSB_PTE_RO                0x0004	/* Read only */
#define PSB_PTE_CACHED            0x0008	/* CPU cache coherent */

/*
 * VDC registers and bits
 */
#define PSB_HWSTAM                0x2098
#define PSB_INSTPM                0x20C0
#define PSB_INT_IDENTITY_R        0x20A4
#define _PSB_VSYNC_PIPEB_FLAG     (1<<5)
#define _PSB_VSYNC_PIPEA_FLAG     (1<<7)
#define _PSB_IRQ_SGX_FLAG         (1<<18)
#define _PSG_IRQ_MSVDX_FLAG       (1<<19)
#define PSB_INT_MASK_R            0x20A8
#define PSB_INT_ENABLE_R          0x20A0
#define PSB_PIPEASTAT             0x70024
#define _PSB_VBLANK_INTERRUPT_ENABLE (1 << 17)
#define _PSB_VBLANK_CLEAR         (1 << 1)
#define PSB_PIPEBSTAT             0x71024

#define _PSB_MMU_ER_MASK      0x0001FF00
#define _PSB_MMU_ER_HOST      (1 << 16)
#define GPIOA			0x5010
#define GPIOB			0x5014
#define GPIOC			0x5018
#define GPIOD			0x501c
#define GPIOE			0x5020
#define GPIOF			0x5024
#define GPIOG			0x5028
#define GPIOH			0x502c
#define GPIO_CLOCK_DIR_MASK		(1 << 0)
#define GPIO_CLOCK_DIR_IN		(0 << 1)
#define GPIO_CLOCK_DIR_OUT		(1 << 1)
#define GPIO_CLOCK_VAL_MASK		(1 << 2)
#define GPIO_CLOCK_VAL_OUT		(1 << 3)
#define GPIO_CLOCK_VAL_IN		(1 << 4)
#define GPIO_CLOCK_PULLUP_DISABLE	(1 << 5)
#define GPIO_DATA_DIR_MASK		(1 << 8)
#define GPIO_DATA_DIR_IN		(0 << 9)
#define GPIO_DATA_DIR_OUT		(1 << 9)
#define GPIO_DATA_VAL_MASK		(1 << 10)
#define GPIO_DATA_VAL_OUT		(1 << 11)
#define GPIO_DATA_VAL_IN		(1 << 12)
#define GPIO_DATA_PULLUP_DISABLE	(1 << 13)

#define VCLK_DIVISOR_VGA0   0x6000
#define VCLK_DIVISOR_VGA1   0x6004
#define VCLK_POST_DIV       0x6010

#define DRM_DRIVER_PRIVATE_T drm_psb_private_t
#define I915_WRITE(_offs, _val) \
  iowrite32(_val, dev_priv->vdc_reg + (_offs))
#define I915_READ(_offs) \
  ioread32(dev_priv->vdc_reg + (_offs))

/*
 * User options.
 */

struct drm_psb_uopt{
	int disable_clock_gating;
};


struct psb_gtt{
        struct drm_device *dev;
        int initialized;
	u32 gatt_start;
	u32 gtt_start;
	u32 gtt_phys_start;
	unsigned gtt_pages;
        unsigned gatt_pages;
	u32 stolen_base;
	u32 pge_ctl;
	u16 gmch_ctrl;
	unsigned long stolen_size;
	u32 *gtt_map;
	struct rw_semaphore sem;
};

/*
 * Non-GPL plugins.
 */

#define MAX_PLUGIN_IOCTLS 20

struct psb_plugin_private;
typedef int (*psb_ioctl_func) (DRM_IOCTL_ARGS);

struct drm_psb_plugin{
	struct psb_plugin_private *(*init) (struct drm_device *);
	void (*takedown)(struct psb_plugin_private *);
	int (*firstopen)(struct psb_plugin_private *);
	int (*lastclose)(struct psb_plugin_private *);
        int (*suspend)(struct psb_plugin_private *, pm_message_t);
	int (*resume) (struct psb_plugin_private *);
        irqreturn_t (*irq_handler) (struct psb_plugin_private *);
	psb_ioctl_func *ioctls;
        int num_ioctls;
};	

struct dev_list_entry {
	struct list_head head;
	struct drm_device *dev;
	int plugin_initialised;
};

typedef struct drm_psb_private {
	unsigned long chipset;
  
        struct dev_list_entry dev_list;
	struct drm_psb_dev_info_arg dev_info;
	struct drm_psb_uopt uopt;

	struct psb_gtt *pg;

	struct page *scratch_page;
	struct page *comm_page;

	volatile u32 *comm;
	u32 comm_mmu_offset;
	u32 mmu_2d_offset;
	u32 sequence;
	u32 *sgx_save;
	int engine_lockup_2d;

	struct psb_mmu_driver *mmu;
	struct psb_mmu_pd *pf_pd;

	u8 *sgx_reg;
	u8 *vdc_reg;
	u32 gatt_free_offset;

	/*
	 * Fencing / irq.
	 */

	u32 sgx_irq_mask;
	u32 vdc_irq_mask;
       
	spinlock_t irqmask_lock;
	int fence0_irq_on;
	int irq_enabled;

	/*
	 * Memory managers
	 */

	int have_vram;
	int have_tt;
	int have_mem_mmu;
	int have_mem_aper;
	int have_mem_kernel;
	struct mutex temp_mem;

	/*
	 * Command submission.
	 */

	drm_buffer_object_t *buffers[PSB_NUM_VALIDATE_BUFFERS];

	/*
	 * SAREA
	 */
	drm_psb_sarea_t *sarea_priv;

	/*
	 * LVDS info 
	 */
	int backlight_duty_cycle;  /* restore backlight to this value */
	bool panel_wants_dither;
	struct drm_display_mode *panel_fixed_mode;

	/* 
	 * Register state 
	 */
	u32 saveDSPACNTR;
	u32 saveDSPBCNTR;
	u32 savePIPEACONF;
	u32 savePIPEBCONF;
	u32 savePIPEASRC;
	u32 savePIPEBSRC;
	u32 saveFPA0;
	u32 saveFPA1;
	u32 saveDPLL_A;
	u32 saveDPLL_A_MD;
	u32 saveHTOTAL_A;
	u32 saveHBLANK_A;
	u32 saveHSYNC_A;
	u32 saveVTOTAL_A;
	u32 saveVBLANK_A;
	u32 saveVSYNC_A;
	u32 saveDSPASTRIDE;
	u32 saveDSPASIZE;
	u32 saveDSPAPOS;
	u32 saveDSPABASE;
	u32 saveDSPASURF;
	u32 saveFPB0;
	u32 saveFPB1;
	u32 saveDPLL_B;
	u32 saveDPLL_B_MD;
	u32 saveHTOTAL_B;
	u32 saveHBLANK_B;
	u32 saveHSYNC_B;
	u32 saveVTOTAL_B;
	u32 saveVBLANK_B;
	u32 saveVSYNC_B;
	u32 saveDSPBSTRIDE;
	u32 saveDSPBSIZE;
	u32 saveDSPBPOS;
	u32 saveDSPBBASE;
	u32 saveDSPBSURF;
	u32 saveVCLK_DIVISOR_VGA0;
	u32 saveVCLK_DIVISOR_VGA1;
	u32 saveVCLK_POST_DIV;
	u32 saveVGACNTRL;
	u32 saveADPA;
	u32 saveLVDS;
	u32 saveDVOA;
	u32 saveDVOB;
	u32 saveDVOC;
	u32 savePP_ON;
	u32 savePP_OFF;
	u32 savePP_CONTROL;
	u32 savePP_CYCLE;
	u32 savePFIT_CONTROL;
	u32 savePaletteA[256];
	u32 savePaletteB[256];
	u32 saveBLC_PWM_CTL;

        struct psb_plugin_private *plug_priv;
} drm_psb_private_t;

struct psb_mmu_driver;

extern struct psb_mmu_driver *psb_mmu_driver_init(u8 __iomem * registers);
extern void psb_mmu_driver_takedown(struct psb_mmu_driver *driver);
extern struct psb_mmu_pd *psb_mmu_get_default_pd(struct psb_mmu_driver *driver);
extern void psb_mmu_mirror_gtt(struct psb_mmu_pd *pd, u32 mmu_offset,
			       u32 gtt_start, __u32 gtt_pages);
extern void psb_mmu_test(struct psb_mmu_driver *driver, u32 offset);
extern struct psb_mmu_pd *psb_mmu_alloc_pd(struct psb_mmu_driver *driver);
extern void psb_mmu_free_pagedir(struct psb_mmu_pd *pd);
extern void psb_mmu_flush(struct psb_mmu_driver *driver);

/*
 * Enable / disable MMU for different requestors.
 */

extern void psb_mmu_enable_requestor(struct psb_mmu_driver *driver, u32 mask);
extern void psb_mmu_disable_requestor(struct psb_mmu_driver *driver,
				      u32 mask);
extern void psb_mmu_set_pd_context(struct psb_mmu_pd *pd, int hw_context);
extern int psb_mmu_insert_pages(struct psb_mmu_pd *pd, struct page **pages,
				unsigned long address, u32 num_pages,
				u32 desired_tile_stride, u32 hw_tile_stride,
				int type);
extern void psb_mmu_remove_pages(struct psb_mmu_pd *pd, unsigned long address,
				 u32 num_pages, __u32 desired_tile_stride,
				 u32 hw_tile_stride);
/*
 * psb_sgx.c 
 */

extern int psb_blit_sequence(drm_psb_private_t * dev_priv);
extern void psb_init_2d(drm_psb_private_t * dev_priv);
extern int drm_psb_idle(drm_device_t * dev);
extern int psb_emit_2d_copy_blit(drm_device_t * dev,
				 u32 src_offset,
				 u32 dst_offset, __u32 pages, int direction);
extern int psb_cmdbuf_ioctl(DRM_IOCTL_ARGS);

/*
 * psb_irq.c
 */

extern irqreturn_t psb_irq_handler(DRM_IRQ_ARGS);
extern void psb_irq_preinstall(drm_device_t * dev);
extern void psb_irq_postinstall(drm_device_t * dev);
extern void psb_irq_uninstall(drm_device_t * dev);

/*
 * psb_fence.c
 */

extern void psb_poke_flush(drm_device_t * dev, uint32_t class);
extern int psb_fence_emit_sequence(drm_device_t * dev, uint32_t class,
				   uint32_t flags, uint32_t * sequence,
				   uint32_t * native_type);
extern void psb_fence_handler(drm_device_t * dev, uint32_t class);
extern int psb_fence_has_irq(drm_device_t * dev, uint32_t class,
			     uint32_t flags);
extern void psb_2D_irq_off(drm_psb_private_t * dev_priv);
extern void psb_2D_irq_on(drm_psb_private_t * dev_priv);

/*
 * psb_buffer.c 
 */
extern drm_ttm_backend_t *drm_psb_tbe_init(drm_device_t * dev);
extern int psb_fence_types(drm_buffer_object_t * bo, uint32_t * class,
			   uint32_t * type);
extern uint32_t psb_evict_mask(drm_buffer_object_t * bo);
extern int psb_invalidate_caches(drm_device_t * dev, uint32_t flags);
extern int psb_init_mem_type(drm_device_t * dev, uint32_t type,
			     drm_mem_type_manager_t * man);
extern int psb_move(drm_buffer_object_t * bo,
		    int evict, int no_wait, drm_bo_mem_reg_t * new_mem);

/*
 * psb_gtt.c 
 */
extern int psb_gtt_init(struct psb_gtt *pg, int resume);
extern int psb_gtt_insert_pages(struct psb_gtt *pg, struct page **pages,
				unsigned offset_pages, unsigned num_pages,
				unsigned desired_tile_stride, 
				unsigned hw_tile_stride, 
				int type);
extern int psb_gtt_remove_pages(struct psb_gtt *pg, unsigned offset_pages, 
				unsigned num_pages, unsigned desired_tile_stride, 
				unsigned hw_tile_stride);

extern struct psb_gtt *psb_gtt_alloc(drm_device_t *dev);
extern void psb_gtt_takedown(struct psb_gtt *pg, int free);

/*
 * psb_fb.c
 */
extern int psbfb_probe(struct drm_device *dev, struct drm_crtc *crtc);
extern int psbfb_remove(struct drm_device *dev, struct drm_crtc *crtc);

/*
 * psb_drv.c
 */

extern struct drm_psb_plugin *psb_lock_plugin(int from_irq);
extern void psb_unlock_plugin(int from_irq);
extern void  psb_load_plugin(struct drm_psb_plugin *plug);
extern void psb_unload_plugin(void);

/*
 * Utilities
 */

#define PSB_ALIGN_TO(_val, _align) \
  (((_val) + ((_align) - 1)) & ~((_align) - 1))
#define PSB_WVDC32(_val, _offs) \
  iowrite32(_val, dev_priv->vdc_reg + (_offs))
#define PSB_RVDC32(_offs) \
  ioread32(dev_priv->vdc_reg + (_offs))
#define PSB_WSGX32(_val, _offs) \
  iowrite32(_val, dev_priv->sgx_reg + (_offs))
#define PSB_RSGX32(_offs) \
  ioread32(dev_priv->sgx_reg + (_offs))

#define PSB_ALPL(_val, _base)			\
  (((_val) >> (_base ## _ALIGNSHIFT)) << (_base ## _SHIFT))
#define PSB_ALPLM(_val, _base)			\
  ((((_val) >> (_base ## _ALIGNSHIFT)) << (_base ## _SHIFT)) & (_base ## _MASK))



static inline psb_fixed psb_mul_fixed(psb_fixed a, psb_fixed b)
{
	s64 tmp;
	s64 a64 = (s64) a;
	s64 b64 = (s64) b;

	tmp = a64*b64;
	return tmp / (1ULL << PSB_FIXED_SHIFT) + 
		((tmp & 0x80000000ULL) ? 1 : 0);
}

static inline psb_fixed psb_mul_ufixed(psb_ufixed a, psb_fixed b)
{
	u64 tmp;
	u64 a64 = (u64) a;
	u64 b64 = (u64) b;

	tmp = a64*b64;
	return (tmp >> PSB_FIXED_SHIFT) + 
		((tmp & 0x80000000ULL) ? 1 : 0);
}

static inline u32 psb_ufixed_to_float32(psb_ufixed a)
{
	u32 exp = 0x7f + 7;
	u32 mantissa = (u32) a;
	
	if (a == 0)
		return 0;
	while((mantissa & 0xff800000) == 0) {
		exp -= 1;
		mantissa <<= 1;
	}
	while((mantissa & 0xff800000) > 0x00800000) {
		exp += 1;
		mantissa >>= 1;
	}
	return (mantissa & ~0xff800000) | (exp << 23);
}

static inline u32 psb_fixed_to_float32(psb_fixed a)
{
	if (a < 0) 
		return psb_ufixed_to_float32(-a) | 0x80000000;
	else
		return psb_ufixed_to_float32(a);
}

#define PSB_D_RENDER  (1 << 16)

#define PSB_D_GENERAL (1 << 0)
#define PSB_D_INIT    (1 << 1)
#define PSB_D_IRQ     (1 << 2)

extern int drm_psb_debug;
extern int drm_psb_no_fb;

#define PSB_DEBUG_GENERAL(_fmt, _arg...) \
	PSB_DEBUG(PSB_D_GENERAL, _fmt, ##_arg)
#define PSB_DEBUG_INIT(_fmt, _arg...) \
	PSB_DEBUG(PSB_D_INIT, _fmt, ##_arg)
#define PSB_DEBUG_IRQ(_fmt, _arg...) \
	PSB_DEBUG(PSB_D_IRQ, _fmt, ##_arg)
#define PSB_DEBUG_RENDER(_fmt, _arg...) \
	PSB_DEBUG(PSB_D_INIT, _fmt, ##_arg)

#if DRM_DEBUG_CODE
#define PSB_DEBUG(_flag, _fmt, _arg...)					\
	do {								\
		if ((_flag) & drm_psb_debug)				\
			printk(KERN_DEBUG				\
			       "[psb:0x%02x:%s] " _fmt , _flag,	\
			       __FUNCTION__ , ##_arg);			\
	} while (0)
#else 
#define PSB_DEBUG(_fmt, _arg...)     do { } while (0)
#endif

#endif
