/**************************************************************************
 *
 * Copyright (c) 2006 Tungsten Graphics Inc. Cedar Park, TX. USA. 
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 *
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this code.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
/*
 */

#include "drmP.h"
#include "psb_drv.h"
#include "psb_reg.h"

/*
 * Video display controller interrupt.
 */

static void psb_vdc_interrupt(drm_device_t * dev, u32 vdc_stat)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	u32 pipe_stats;
	int wake = 0;

	if (vdc_stat & _PSB_VSYNC_PIPEA_FLAG) {
		pipe_stats = PSB_RVDC32(PSB_PIPEASTAT);
		atomic_inc(&dev->vbl_received);
		wake = 1;
		PSB_WVDC32(pipe_stats | _PSB_VBLANK_INTERRUPT_ENABLE |
			   _PSB_VBLANK_CLEAR, PSB_PIPEASTAT);
	}

	if (vdc_stat & _PSB_VSYNC_PIPEB_FLAG) {
		pipe_stats = PSB_RVDC32(PSB_PIPEBSTAT);
		atomic_inc(&dev->vbl_received2);
		wake = 1;
		PSB_WVDC32(pipe_stats | _PSB_VBLANK_INTERRUPT_ENABLE |
			   _PSB_VBLANK_CLEAR, PSB_PIPEBSTAT);
	}

	PSB_WVDC32(vdc_stat, PSB_INT_IDENTITY_R);
	(void)PSB_RVDC32(PSB_INT_IDENTITY_R);
	DRM_READMEMORYBARRIER();

	if (wake) {
		DRM_WAKEUP(&dev->vbl_queue);
		drm_vbl_send_signals(dev);
	}
}

/*
 * SGX interrupt source 1.
 */

static void psb_sgx_interrupt(drm_device_t * dev, u32 sgx_stat)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;

	PSB_WSGX32(sgx_stat, PSB_CR_EVENT_HOST_CLEAR);
	(void)PSB_RSGX32(PSB_CR_EVENT_HOST_CLEAR);
	DRM_READMEMORYBARRIER();

	if (sgx_stat & _PSB_CE_TWOD_COMPLETE) {
		psb_fence_handler(dev, 0);
	}
}

irqreturn_t psb_irq_handler(DRM_IRQ_ARGS)
{
	drm_device_t *dev = (drm_device_t *) arg;
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	const struct drm_psb_plugin *plug;

	u32 vdc_stat = PSB_RVDC32(PSB_INT_IDENTITY_R);
	u32 sgx_stat = PSB_RSGX32(PSB_CR_EVENT_STATUS);
	u32 sgx_mask = PSB_RSGX32(PSB_CR_EVENT_HOST_ENABLE);
	int handled = 0;

	spin_lock(&dev_priv->irqmask_lock);

	if (!(sgx_stat & _PSB_CE_MASTER_INTERRUPT) &&
	    !(vdc_stat & dev_priv->vdc_irq_mask)) {
		spin_unlock(&dev_priv->irqmask_lock);
		return IRQ_NONE;
	}
	vdc_stat &= dev_priv->vdc_irq_mask;
	sgx_stat &= (sgx_mask & ~_PSB_CE_MASTER_INTERRUPT);
	spin_unlock(&dev_priv->irqmask_lock);

	if (vdc_stat) {
		psb_vdc_interrupt(dev, vdc_stat);
		handled = 1;
	}
	if (sgx_stat) {
		psb_sgx_interrupt(dev, sgx_stat);
		handled = 1;
	}
	if (NULL != (plug = psb_lock_plugin(1))) {
		if (plug->irq_handler && 
		    plug->irq_handler(dev_priv->plug_priv) == IRQ_HANDLED)
			handled = 1;
		psb_unlock_plugin(1);
	}

	if (!handled) 
		PSB_DEBUG_IRQ("Unhandled irq\n");


	return IRQ_HANDLED;
}

void psb_irq_preinstall(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	spin_lock(&dev_priv->irqmask_lock);
	PSB_WVDC32(0xFFFFFFFF, PSB_HWSTAM);
	PSB_WVDC32(0x00000000, PSB_INT_MASK_R);
	PSB_WVDC32(0x00000000, PSB_INT_ENABLE_R);
	PSB_WSGX32(0x00000000, PSB_CR_EVENT_HOST_ENABLE);
	(void)PSB_RSGX32(PSB_CR_EVENT_HOST_ENABLE);

	dev_priv->sgx_irq_mask = 0x00000000;
	dev_priv->vdc_irq_mask = _PSB_VSYNC_PIPEA_FLAG | 
		_PSB_VSYNC_PIPEB_FLAG |
		_PSB_IRQ_SGX_FLAG;
	spin_unlock(&dev_priv->irqmask_lock);
}

void psb_irq_postinstall(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	unsigned long irqflags;

	spin_lock_irqsave(&dev_priv->irqmask_lock, irqflags);
	PSB_WVDC32(dev_priv->vdc_irq_mask, PSB_INT_ENABLE_R);
	PSB_WSGX32(dev_priv->sgx_irq_mask, PSB_CR_EVENT_HOST_ENABLE);
	(void)PSB_RSGX32(PSB_CR_EVENT_HOST_ENABLE);
	dev_priv->irq_enabled = 1;
	spin_unlock_irqrestore(&dev_priv->irqmask_lock, irqflags);
}

void psb_irq_uninstall(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	unsigned long irqflags;

	spin_lock_irqsave(&dev_priv->irqmask_lock, irqflags);
	dev_priv->sgx_irq_mask = 0x00000000;
	dev_priv->vdc_irq_mask = 0x00000000;

	PSB_WVDC32(0xFFFFFFFF, PSB_HWSTAM);
	PSB_WVDC32(0xFFFFFFFF, PSB_INT_MASK_R);
	PSB_WVDC32(dev_priv->vdc_irq_mask, PSB_INT_ENABLE_R);
	PSB_WSGX32(dev_priv->sgx_irq_mask, PSB_CR_EVENT_HOST_ENABLE);
	wmb();
	PSB_WVDC32(PSB_RVDC32(PSB_INT_IDENTITY_R), PSB_INT_IDENTITY_R);
	PSB_WSGX32(PSB_RSGX32(PSB_CR_EVENT_STATUS), PSB_CR_EVENT_HOST_CLEAR);

	dev_priv->irq_enabled = 0;
	spin_unlock_irqrestore(&dev_priv->irqmask_lock, irqflags);

}

void psb_2D_irq_off(drm_psb_private_t * dev_priv)
{
	unsigned long irqflags;
	u32 old_mask;
	u32 cleared_mask;

	spin_lock_irqsave(&dev_priv->irqmask_lock, irqflags);
	if (dev_priv->irq_enabled) {

		old_mask = dev_priv->sgx_irq_mask;
		dev_priv->sgx_irq_mask &= ~_PSB_CE_TWOD_COMPLETE;
		PSB_WSGX32(dev_priv->sgx_irq_mask, PSB_CR_EVENT_HOST_ENABLE);
		(void)PSB_RSGX32(PSB_CR_EVENT_HOST_ENABLE);
		
		cleared_mask = (old_mask ^ dev_priv->sgx_irq_mask) & old_mask;
		PSB_WSGX32(cleared_mask, PSB_CR_EVENT_HOST_CLEAR);
		(void)PSB_RSGX32(PSB_CR_EVENT_HOST_CLEAR);
	}
	spin_unlock_irqrestore(&dev_priv->irqmask_lock, irqflags);
}

void psb_2D_irq_on(drm_psb_private_t * dev_priv)
{
	unsigned long irqflags;

	spin_lock_irqsave(&dev_priv->irqmask_lock, irqflags);
	if (dev_priv->irq_enabled) {
		dev_priv->sgx_irq_mask |= _PSB_CE_TWOD_COMPLETE;
		PSB_WSGX32(dev_priv->sgx_irq_mask, PSB_CR_EVENT_HOST_ENABLE);
		(void)PSB_RSGX32(PSB_CR_EVENT_HOST_ENABLE);
	}
	spin_unlock_irqrestore(&dev_priv->irqmask_lock, irqflags);
}
