/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/

/*
 * Check for bugs #1303 and #1327: libcurl should never remove DNS entries
 * created via CURLOPT_RESOLVE, neither after DNS_CACHE_TIMEOUT elapses
 * (test1515) nor a dead connection is detected (test1616).
 */

#include "first.h"

#include "testtrace.h"

#define DNS_TIMEOUT 1L

static CURLcode do_one_request(CURLM *multi, const char *URL,
                               const char *resolve)
{
  CURL *curl;
  struct curl_slist *resolve_list = NULL;
  int still_running;
  CURLcode result = CURLE_OK;
  CURLMsg *msg;
  int msgs_left;

  resolve_list = curl_slist_append(resolve_list, resolve);

  easy_init(curl);

  easy_setopt(curl, CURLOPT_URL, URL);
  easy_setopt(curl, CURLOPT_RESOLVE, resolve_list);
  easy_setopt(curl, CURLOPT_DNS_CACHE_TIMEOUT, DNS_TIMEOUT);

  debug_config.nohex = TRUE;
  debug_config.tracetime = TRUE;
  easy_setopt(curl, CURLOPT_DEBUGDATA, &debug_config);
  easy_setopt(curl, CURLOPT_DEBUGFUNCTION, libtest_debug_cb);
  easy_setopt(curl, CURLOPT_VERBOSE, 1L);

  multi_add_handle(multi, curl);
  multi_perform(multi, &still_running);

  abort_on_test_timeout();

  while(still_running) {
    struct timeval timeout;
    fd_set fdread, fdwrite, fdexcep;
    int maxfd = -99;

    FD_ZERO(&fdread);
    FD_ZERO(&fdwrite);
    FD_ZERO(&fdexcep);
    timeout.tv_sec = 1;
    timeout.tv_usec = 0;

    multi_fdset(multi, &fdread, &fdwrite, &fdexcep, &maxfd);
    select_test(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);

    abort_on_test_timeout();
    multi_perform(multi, &still_running);

    abort_on_test_timeout();
  }

  do {
    msg = curl_multi_info_read(multi, &msgs_left);
    if(msg && msg->msg == CURLMSG_DONE && msg->easy_handle == curl) {
      result = msg->data.result;
      break;
    }
  } while(msg);

test_cleanup:

  curl_multi_remove_handle(multi, curl);
  curl_easy_cleanup(curl);
  curl_slist_free_all(resolve_list);

  return result;
}

static CURLcode test_lib1515(const char *URL)
{
  CURLM *multi = NULL;
  CURLcode result = CURLE_OK;
  const char *path = URL;
  const char *address = libtest_arg2;
  const char *port = libtest_arg3;
  char dns_entry[256];
  int i;
  int count = 2;

  curl_msnprintf(dns_entry, sizeof(dns_entry), "testserver.example.com:%s:%s",
                 port, address);

  start_test_timing();

  global_init(CURL_GLOBAL_ALL);
  curl_global_trace("all");
  multi_init(multi);

  for(i = 1; i <= count; i++) {
    char target_url[256];
    curl_msnprintf(target_url, sizeof(target_url),
                   "http://testserver.example.com:%s/%s%04d", port, path, i);

    /* second request must succeed like the first one */
    result = do_one_request(multi, target_url, dns_entry);
    if(result != CURLE_OK) {
      curl_mfprintf(stderr, "request %s failed with %d\n", target_url, result);
      goto test_cleanup;
    }

    if(i < count)
      curlx_wait_ms((DNS_TIMEOUT + 1) * 1000);
  }

test_cleanup:

  curl_multi_cleanup(multi);
  curl_global_cleanup();

  return result;
}
