# -*- coding: utf-8 -*-
#####################################################################
#  Rafael Proença <cypherbios@ubuntu.com>
#  Laudeci Oliveira <laudeci@gmail.com>
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#####################################################################

import gtk
import sys
import os
import gobject

from APTonCD.core import *
from APTonCD.core import gui 
from APTonCD.core import constants
from APTonCD.core import addmedia
from APTonCD.core import version
from APTonCD.widgets import quick_guide
from APTonCD.create import createWindow
from APTonCD.restore import restoreWindow
from APTonCD.core.utils import get_icon
from APTonCD.core.pkgmngr import SynapticManager


WINDOW_NAME = 'main'
HOME_WIDGET = 'home'

class MainWindow(controller.IController):
    """
        Main application window
    """
 
    def __init__(self, temp_path = '', packages_path = ''):
        """
            Constructor
        """
        self.temp_path = temp_path
        self.packages_path = packages_path
        self.create = None
        self.restore = None
        self.glade = gui.get_glade(constants.MAIN_GUI, WINDOW_NAME)
        
        #Main widget
        self.window = gui.get_widget(self.glade, WINDOW_NAME)
        self.window.set_icon_name(constants.APP_ICON_NAME)
        
        #Window Widgets
        #notebook widget
        self.notebook = gui.get_widget(self.glade, 'notebook')
        
        self.createContainer = gui.get_widget(self.glade,'panelCreate')
        self.restoreContainer = gui.get_widget(self.glade, 'panelRestore')
        self.panelCreate = gui.get_widget(self.glade, 'createPanel')
        self.panelRestore = gui.get_widget(self.glade, 'restorePanel')
        
        #self.image1 = gui.get_widget(self.glade, 'image1')
        #self.image1.set_from_pixbuf(get_icon(constants.APP_ICON_NAME))
        
        self.progressbar = gui.get_widget(self.glade, 'pb_main')
        self.progressbar.hide()
        
        #mode menu 
        self.mnuModeCreate = gui.get_widget(self.glade,  'mnuModeCreate')
        self.mnuModeRestore = gui.get_widget(self.glade,  'mnuModeRestore')
        
        self.mnuReload = gui.get_widget(self.glade, 'mnuReload')
        self.mnuSelectAll = gui.get_widget(self.glade, 'mnuSelectAll')
        self.mnuUnselectAll = gui.get_widget(self.glade, 'mnuUnselectAll')
        self.mnuInvertSelections = gui.get_widget(self.glade, 'mnuInvertSelections')
        
        #CheckMenuItem
        self.mnuSelectDepends = gui.get_widget(self.glade, 'mnuSelectDepends')
        self.mnuInsertOldVersions = gui.get_widget(self.glade, 'mnuInsertOldVersions')
        
        #viewMenuItens
        self.chkColumnSize = gui.get_widget(self.glade, 'chkColumnSize')
        self.chkColumnVersion = gui.get_widget(self.glade, 'chkColumnVersion')
        
        self.statusbar = gui.get_widget(self.glade, 'statusbar')
        
        #workaround to set pango markup to a statusbar
        self.status_label =  self.statusbar.get_children()[0].get_children()[0]
        if not isinstance(self.status_label, gtk.Label):
            self.status_label =  self.statusbar.get_children()[0].get_children()[0].get_children()[0]        
        self.status_label.set_use_markup(True)
        
        #signals connection
        gui.connect(gui.get_widget(self.glade, 'btnMainCreate'), 'clicked', self.on_activate_tab, 0)
        gui.connect(gui.get_widget(self.glade, 'btnMainRestore'), 'clicked', self.on_activate_tab, 1)
        gui.connect(gui.get_widget(self.glade, 'mnuQuit'), 'activate', self.on_destroy)
        gui.connect(self.mnuReload, 'activate', self.on_reload)
        gui.connect(gui.get_widget(self.glade, 'mnuAddMedia'), 'activate', self.on_add_media)
        gui.connect(gui.get_widget(self.glade, 'mnuHelp'), 'activate', self.on_help)
        gui.connect(gui.get_widget(self.glade, 'mnuQuickGuide'), 'activate', self.on_quick_guide)
        gui.connect(gui.get_widget(self.glade, 'btnHelp'), 'clicked', self.on_help)
        gui.connect(gui.get_widget(self.glade, 'mnuAbout'), 'activate', self.on_about)
        
        gui.connect(self.mnuSelectAll, 'activate', self.on_editmenu_select, 0)
        gui.connect(self.mnuUnselectAll, 'activate', self.on_editmenu_select, 1)
        gui.connect(self.mnuInvertSelections, 'activate', self.on_editmenu_select, 2)
        
        # mode menu events
        gui.connect(self.mnuModeCreate, 'toggled', self.on_change_mode, 0)
        gui.connect(self.mnuModeRestore, 'toggled', self.on_change_mode, 1)
        
        gui.connect(self.mnuSelectDepends, 'toggled', self.on_select_depends_old, 0)
        gui.connect(self.mnuInsertOldVersions, 'toggled', self.on_select_depends_old, 1)
        
        gui.connect(self.chkColumnSize, 'toggled', self.on_select_view, 0)
        gui.connect(self.chkColumnVersion, 'toggled', self.on_select_view, 1)
        
        gui.connect(self.window, 'destroy', self.on_destroy)
        
        self.show_panel('')
        
        self.show_status_message(constants.MESSAGE_0066)
    
    def on_activate_tab(self,  widget,  index):
        if index ==0:
            self.mnuModeCreate.set_active(not self.mnuModeCreate.get_active())
        elif index ==1:
            self.mnuModeRestore.set_active(not self.mnuModeRestore.get_active())
            
    def on_change_mode(self, checkmenuitem, menu_type):
        if menu_type == 0:
            self.show_panel('create')
            self.window.set_title("APTonCD - " + constants.MESSAGE_0075)
        elif menu_type == 1:
            self.show_panel('restore')
            self.window.set_title("APTonCD - " + constants.MESSAGE_0076)
            
    def on_editmenu_select(self,menu,item):
        if self.create:
            self.create.packageList.check_uncheck_all(item)
        if self.restore:
            self.restore.packageList.check_uncheck_all(item)
    
    def on_select_view(self, checkmenuitem, menu_type):
        if self.create:
            self.create.packageList.show_column(menu_type)
        if self.restore:
            self.restore.packageList.show_column(menu_type)
            
    def on_select_depends_old(self, checkmenuitem, menu_type):
        if menu_type == 0:
            if self.create:
                self.create.packageList.auto_select_packages = checkmenuitem.get_active()
            if self.restore:
                self.restore.packageList.auto_select_packages = checkmenuitem.get_active()
        elif menu_type == 1:
            if self.create:
                self.create.packageList.can_select_old = checkmenuitem.get_active()
            if self.restore:
                self.restore.packageList.can_select_old = checkmenuitem.get_active()
        else:
            pass
    
    def get_main_window(self):
        """
            implements IController interface method.
            for access the class windows object.
        """
        return self.window
    
    def get_parent_widget(self):
        """
            implements IController interface method.
            for container widget
        """
        if self.notebook.get_current_page() == 1:
            return self.createContainer
        elif self.notebook.get_current_page() == 2:
            return self.restoreContainer
            
    def show_status_message(self, text = ''):
        #if text == '':
            #self.statusbar.push(self.status_context,'Ready')
        #else:
            #self.statusbar.push(self.status_context, text)
        self.status_label.set_markup(text)
    
    def on_reload(self,*args):
        """
            This will reload the package list when clicked.
        """
        self.rebuild_list()
    
    def on_add_media(self, *args):
        """
            This will add a new cd/dvd media using synaptic way.
        """
        manager = SynapticManager(controller=self)
        showdialogfile = os.path.join(constants.CONFIG_DIR,"dontshow-addmediainfo")
        if not utils.fileExist(showdialogfile):
        	addWindow = addmedia.AddMediaDialog(self)
        	response = addWindow.run()
        	if addWindow.get_showagain() == True:
        		f = open(showdialogfile,"w")
        		f.close
        	if response == -10:	        			
        		addWindow.destroy()
        		manager.addMedia()
        	else:
        		addWindow.destroy()
        else:
        	manager.addMedia()
        	
    def on_quick_guide(self, *args):
    	"""
    		This will show a simple introduction
    	"""
    	guideWindow = quick_guide.QuickGuideDialog(self)
    	response = guideWindow.run()

    def on_help(self, *args):
        """
            This will show help contents.
        """
        helpviewer = utils.whereis("yelp")
        if helpviewer == None:
        	gui.message(constants.MESSAGE_0064, self.get_main_window())
        else:
        	os.system(helpviewer + ' ghelp:aptoncd 2> /dev/null &')
    
    def on_about(self, *args):
        """
            This will show the about window with some project's info.
        """
        gladeFileName = constants.MAIN_GUI
        self.formName = "frmAbout"
        self.gladeFile = gtk.glade.XML(gladeFileName, self.formName)
        frmAbout = self.gladeFile.get_widget(self.formName)
        frmAbout.set_position(gtk.WIN_POS_CENTER)
        frmAbout.set_name('APTonCD')
        frmAbout.set_version(version.APP_VERSION)
        frmAbout.set_modal(True)
        result = frmAbout.run()
        frmAbout.destroy()
        return 	result
            
    def on_destroy(*args):
        gtk.main_quit()
        sys.exit(0)
        
    def show(self):
        self.window.resize(680, 600)
        self.window.show()
    
    def hide(self):
        self.window.hide()
    
    def show_panel(self, panel):    
        if panel == 'restore':
            self.notebook.set_current_page(2)
            if not isinstance(self.restore, restoreWindow.RestoreWindow ):
                self.restore = restoreWindow.RestoreWindow(self, self.temp_path, None)
            else:
                self.restore.content.show()
            self.mnuReload.set_sensitive(False)
            self.restore.packageList.update_parent_count()
        elif panel == 'create':
            self.notebook.set_current_page(1)
            if not isinstance(self.create, createWindow.CreateWindow):
                self.create = createWindow.CreateWindow(self,self.temp_path, self.packages_path)
                #gui.processEvents()
                gobject.idle_add(self.create.rebuild_list)
            else:
                self.create.content.show()
            self.mnuReload.set_sensitive(True)
            self.create.packageList.update_parent_count()
        else:
            self.notebook.set_current_page(0)
            self.show_status_message(constants.MESSAGE_0066)
                
            
    def rebuild_list(self):
        if self.notebook.get_current_page() == 1:
            self.create.rebuild_list()
        elif self.notebook.get_current_page() == 2:
            #self.restore.rebuild_list()
            #restore list will not be updated
            pass
