/*
 * Cantata
 *
 * Copyright (c) 2011-2017 Craig Drummond <craig.p.drummond@gmail.com>
 *
 * ----
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "digitallyimportedsettings.h"
#include "models/digitallyimported.h"
#include "support/buddylabel.h"
#include "support/lineedit.h"
#include <QPushButton>
#include <QComboBox>

DigitallyImportedSettings::DigitallyImportedSettings(QWidget *parent)
    : Dialog(parent)
{
    setButtons(Ok|Cancel);
    setCaption(tr("Digitally Imported Settings"));
    QWidget *mainWidet = new QWidget(this);
    setupUi(mainWidet);
    setMainWidget(mainWidet);

    audio->addItem(tr("MP3 256k"), 0);
    audio->addItem(tr("AAC 64k"), 1);
    audio->addItem(tr("AAC 128k"), 2);

    connect(loginButton, SIGNAL(clicked()), this, SLOT(login()));
    connect(DigitallyImported::self(), SIGNAL(loginStatus(bool,QString)), SLOT(loginStatus(bool,QString)));

    int h=fontMetrics().height();
    user->setMinimumWidth(h*20);
    adjustSize();
    setMinimumSize(size());
}

void DigitallyImportedSettings::show()
{
    prevUser=DigitallyImported::self()->user();
    prevPass=DigitallyImported::self()->pass();
    wasLoggedIn=DigitallyImported::self()->loggedIn();
    prevAudioType=DigitallyImported::self()->audioType();

    setState();
    user->setText(prevUser);
    pass->setText(prevPass);

    for (int i=0; i<audio->count(); ++i) {
        if (audio->itemData(i).toInt()==DigitallyImported::self()->audioType()) {
            audio->setCurrentIndex(i);
            break;
        }
    }

    loginStatusLabel->setText(DigitallyImported::self()->statusString());
    if (QDialog::Accepted==Dialog::exec()) {
        QString u=user->text().trimmed();
        QString p=pass->text().trimmed();
        int at=audio->itemData(audio->currentIndex()).toInt();
        bool changed=false;
        bool needToLogin=false;
        if (u!=DigitallyImported::self()->user()) {
            DigitallyImported::self()->setUser(u);
            needToLogin=changed=true;
        }
        if (p!=DigitallyImported::self()->pass()) {
            DigitallyImported::self()->setPass(p);
            needToLogin=changed=true;
        }
        if (DigitallyImported::self()->audioType()!=at) {
            DigitallyImported::self()->setAudioType(at);
            changed=true;
        }
        if (needToLogin) {
            DigitallyImported::self()->login();
        }
        if (changed) {
            DigitallyImported::self()->save();
        }
    } else {
        DigitallyImported::self()->setUser(prevUser);
        DigitallyImported::self()->setPass(prevPass);
        DigitallyImported::self()->setAudioType(prevAudioType);
        if (wasLoggedIn) {
            DigitallyImported::self()->login();
        }
    }
}

void DigitallyImportedSettings::login()
{
    if (DigitallyImported::self()->loggedIn()) {
        loginStatusLabel->setText(tr("Not Authenticated"));
        DigitallyImported::self()->logout();
    } else {
        loginStatusLabel->setText(tr("Authenticating..."));
        messageWidget->close();
        DigitallyImported::self()->setUser(user->text().trimmed());
        DigitallyImported::self()->setPass(pass->text().trimmed());
        DigitallyImported::self()->login();
    }
}

void DigitallyImportedSettings::loginStatus(bool status, const QString &msg)
{
    loginStatusLabel->setText(status ? tr("Authenticated") : tr("Not Authenticated"));
    if (status) {
        messageWidget->close();
    } else {
        messageWidget->setError(msg, true);
        adjustSize();
    }
    setState();
}

void DigitallyImportedSettings::setState()
{
    if (DigitallyImported::self()->sessionExpiry().isValid()) {
        expiry->setText(DigitallyImported::self()->sessionExpiry().toString(Qt::ISODate));
    } else {
        loginButton->setText(tr("Login"));
        expiry->setText(QString());
    }

    expiry->setVisible(DigitallyImported::self()->sessionExpiry().isValid());
    expiryLabel->setVisible(expiry->isVisible());
    loginButton->setText(DigitallyImported::self()->loggedIn() ? tr("Logout") : tr("Login"));
}
