/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.get;

import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.action.ShardOperationFailedException;
import org.elasticsearch.action.admin.indices.alias.Alias;
import org.elasticsearch.action.admin.indices.flush.FlushResponse;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.*;
import org.elasticsearch.common.Base64;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.test.ElasticsearchIntegrationTest;
import org.elasticsearch.test.junit.annotations.TestLogging;
import org.junit.Test;

import java.io.IOException;
import java.util.Map;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;
import static org.elasticsearch.test.hamcrest.ElasticsearchAssertions.assertAcked;
import static org.hamcrest.Matchers.*;

public class GetActionTests extends ElasticsearchIntegrationTest {

    @Test
    public void simpleGetTests() {
        assertAcked(prepareCreate("test")
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1))
                .addAlias(new Alias("alias")));
        ensureGreen();

        GetResponse response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(false));

        logger.info("--> index doc 1");
        client().prepareIndex("test", "type1", "1").setSource("field1", "value1", "field2", "value2").get();

        logger.info("--> realtime get 1");
        response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2"));

        logger.info("--> realtime get 1 (no source, implicit)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFields(Strings.EMPTY_ARRAY).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getFields().size(), equalTo(0));
        assertThat(response.getSourceAsBytes(), nullValue());

        logger.info("--> realtime get 1 (no source, explicit)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFetchSource(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getFields().size(), equalTo(0));
        assertThat(response.getSourceAsBytes(), nullValue());

        logger.info("--> realtime get 1 (no type)");
        response = client().prepareGet(indexOrAlias(), null, "1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2"));

        logger.info("--> non realtime get 1");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setRealtime(false).get();
        assertThat(response.isExists(), equalTo(false));

        logger.info("--> realtime fetch of field (requires fetching parsing source)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFields("field1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsBytes(), nullValue());
        assertThat(response.getField("field1").getValues().get(0).toString(), equalTo("value1"));
        assertThat(response.getField("field2"), nullValue());

        logger.info("--> realtime fetch of field & source (requires fetching parsing source)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFields("field1").setFetchSource("field1", null).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap(), hasKey("field1"));
        assertThat(response.getSourceAsMap(), not(hasKey("field2")));
        assertThat(response.getField("field1").getValues().get(0).toString(), equalTo("value1"));
        assertThat(response.getField("field2"), nullValue());

        logger.info("--> flush the index, so we load it from it");
        flush();

        logger.info("--> realtime get 1 (loaded from index)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2"));

        logger.info("--> non realtime get 1 (loaded from index)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setRealtime(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2"));

        logger.info("--> realtime fetch of field (loaded from index)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFields("field1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsBytes(), nullValue());
        assertThat(response.getField("field1").getValues().get(0).toString(), equalTo("value1"));
        assertThat(response.getField("field2"), nullValue());

        logger.info("--> realtime fetch of field & source (loaded from index)");
        response = client().prepareGet(indexOrAlias(), "type1", "1").setFields("field1").setFetchSource(true).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsBytes(), not(nullValue()));
        assertThat(response.getField("field1").getValues().get(0).toString(), equalTo("value1"));
        assertThat(response.getField("field2"), nullValue());

        logger.info("--> update doc 1");
        client().prepareIndex("test", "type1", "1").setSource("field1", "value1_1", "field2", "value2_1").get();

        logger.info("--> realtime get 1");
        response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1_1"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2_1"));

        logger.info("--> update doc 1 again");
        client().prepareIndex("test", "type1", "1").setSource("field1", "value1_2", "field2", "value2_2").get();

        response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getSourceAsMap().get("field1").toString(), equalTo("value1_2"));
        assertThat(response.getSourceAsMap().get("field2").toString(), equalTo("value2_2"));

        DeleteResponse deleteResponse = client().prepareDelete("test", "type1", "1").get();
        assertThat(deleteResponse.isFound(), equalTo(true));

        response = client().prepareGet(indexOrAlias(), "type1", "1").get();
        assertThat(response.isExists(), equalTo(false));
    }

    private static String indexOrAlias() {
        return randomBoolean() ? "test" : "alias";
    }

    @Test
    public void simpleMultiGetTests() throws Exception {
        assertAcked(prepareCreate("test").addAlias(new Alias("alias"))
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));
        ensureGreen();

        MultiGetResponse response = client().prepareMultiGet().add(indexOrAlias(), "type1", "1").get();
        assertThat(response.getResponses().length, equalTo(1));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(false));

        for (int i = 0; i < 10; i++) {
            client().prepareIndex("test", "type1", Integer.toString(i)).setSource("field", "value" + i).get();
        }

        response = client().prepareMultiGet()
                .add(indexOrAlias(), "type1", "1")
                .add(indexOrAlias(), "type1", "15")
                .add(indexOrAlias(), "type1", "3")
                .add(indexOrAlias(), "type1", "9")
                .add(indexOrAlias(), "type1", "11").get();
        assertThat(response.getResponses().length, equalTo(5));
        assertThat(response.getResponses()[0].getId(), equalTo("1"));
        assertThat(response.getResponses()[0].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[0].getResponse().getIndex(), equalTo("test"));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[0].getResponse().getSourceAsMap().get("field").toString(), equalTo("value1"));
        assertThat(response.getResponses()[1].getId(), equalTo("15"));
        assertThat(response.getResponses()[1].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[1].getResponse().getIndex(), equalTo("test"));
        assertThat(response.getResponses()[1].getResponse().isExists(), equalTo(false));
        assertThat(response.getResponses()[2].getId(), equalTo("3"));
        assertThat(response.getResponses()[2].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[2].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[3].getId(), equalTo("9"));
        assertThat(response.getResponses()[3].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[3].getResponse().getIndex(), equalTo("test"));
        assertThat(response.getResponses()[3].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[4].getId(), equalTo("11"));
        assertThat(response.getResponses()[4].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[4].getResponse().getIndex(), equalTo("test"));
        assertThat(response.getResponses()[4].getResponse().isExists(), equalTo(false));

        // multi get with specific field
        response = client().prepareMultiGet()
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").fields("field"))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "3").fields("field"))
                .get();

        assertThat(response.getResponses().length, equalTo(2));
        assertThat(response.getResponses()[0].getResponse().getSourceAsBytes(), nullValue());
        assertThat(response.getResponses()[0].getResponse().getField("field").getValues().get(0).toString(), equalTo("value1"));
    }

    @Test
    public void realtimeGetWithCompress() throws Exception {
        assertAcked(prepareCreate("test").setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1))
                .addMapping("type", jsonBuilder().startObject().startObject("type").startObject("_source").field("compress", true).endObject().endObject().endObject()));
        ensureGreen();

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 10000; i++) {
            sb.append((char) i);
        }
        String fieldValue = sb.toString();
        client().prepareIndex("test", "type", "1").setSource("field", fieldValue).get();

        // realtime get
        GetResponse getResponse = client().prepareGet("test", "type", "1").get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getSourceAsMap().get("field").toString(), equalTo(fieldValue));
    }

    @Test
    public void getFieldsWithDifferentTypes() throws Exception {
        assertAcked(prepareCreate("test").setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1))
                .addMapping("type1", jsonBuilder().startObject().startObject("type1").startObject("_source").field("enabled", true).endObject().endObject().endObject())
                .addMapping("type2", jsonBuilder().startObject().startObject("type2")
                        .startObject("_source").field("enabled", false).endObject()
                        .startObject("properties")
                        .startObject("str").field("type", "string").field("store", "yes").endObject()
                        .startObject("strs").field("type", "string").field("store", "yes").endObject()
                        .startObject("int").field("type", "integer").field("store", "yes").endObject()
                        .startObject("ints").field("type", "integer").field("store", "yes").endObject()
                        .startObject("date").field("type", "date").field("store", "yes").endObject()
                        .startObject("binary").field("type", "binary").field("store", "yes").endObject()
                        .endObject()
                        .endObject().endObject()));
        ensureGreen();

        client().prepareIndex("test", "type1", "1").setSource(
                jsonBuilder().startObject()
                        .field("str", "test")
                        .field("strs", new String[]{"A", "B", "C"})
                        .field("int", 42)
                        .field("ints", new int[]{1, 2, 3, 4})
                        .field("date", "2012-11-13T15:26:14.000Z")
                        .field("binary", Base64.encodeBytes(new byte[]{1, 2, 3}))
                        .endObject()).get();

        client().prepareIndex("test", "type2", "1").setSource(
                jsonBuilder().startObject()
                        .field("str", "test")
                        .field("strs", new String[]{"A", "B", "C"})
                        .field("int", 42)
                        .field("ints", new int[]{1, 2, 3, 4})
                        .field("date", "2012-11-13T15:26:14.000Z")
                        .field("binary", Base64.encodeBytes(new byte[]{1, 2, 3}))
                        .endObject()).get();

        // realtime get with stored source
        logger.info("--> realtime get (from source)");
        GetResponse getResponse = client().prepareGet("test", "type1", "1").setFields("str", "strs", "int", "ints", "date", "binary").get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat((String) getResponse.getField("str").getValue(), equalTo("test"));
        assertThat(getResponse.getField("strs").getValues(), contains((Object) "A", "B", "C"));
        assertThat((Long) getResponse.getField("int").getValue(), equalTo(42l));
        assertThat(getResponse.getField("ints").getValues(), contains((Object) 1L, 2L, 3L, 4L));
        assertThat((String) getResponse.getField("date").getValue(), equalTo("2012-11-13T15:26:14.000Z"));
        assertThat(getResponse.getField("binary").getValue(), instanceOf(String.class)); // its a String..., not binary mapped

        logger.info("--> realtime get (from stored fields)");
        getResponse = client().prepareGet("test", "type2", "1").setFields("str", "strs", "int", "ints", "date", "binary").get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat((String) getResponse.getField("str").getValue(), equalTo("test"));
        assertThat(getResponse.getField("strs").getValues(), contains((Object) "A", "B", "C"));
        assertThat((Integer) getResponse.getField("int").getValue(), equalTo(42));
        assertThat(getResponse.getField("ints").getValues(), contains((Object) 1, 2, 3, 4));
        assertThat((String) getResponse.getField("date").getValue(), equalTo("2012-11-13T15:26:14.000Z"));
        assertThat((BytesReference) getResponse.getField("binary").getValue(), equalTo((BytesReference) new BytesArray(new byte[]{1, 2, 3})));

        logger.info("--> flush the index, so we load it from it");
        flush();

        logger.info("--> non realtime get (from source)");
        getResponse = client().prepareGet("test", "type1", "1").setFields("str", "strs", "int", "ints", "date", "binary").get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat((String) getResponse.getField("str").getValue(), equalTo("test"));
        assertThat(getResponse.getField("strs").getValues(), contains((Object) "A", "B", "C"));
        assertThat((Long) getResponse.getField("int").getValue(), equalTo(42l));
        assertThat(getResponse.getField("ints").getValues(), contains((Object) 1L, 2L, 3L, 4L));
        assertThat((String) getResponse.getField("date").getValue(), equalTo("2012-11-13T15:26:14.000Z"));
        assertThat(getResponse.getField("binary").getValue(), instanceOf(String.class)); // its a String..., not binary mapped

        logger.info("--> non realtime get (from stored fields)");
        getResponse = client().prepareGet("test", "type2", "1").setFields("str", "strs", "int", "ints", "date", "binary").get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat((String) getResponse.getField("str").getValue(), equalTo("test"));
        assertThat(getResponse.getField("strs").getValues(), contains((Object) "A", "B", "C"));
        assertThat((Integer) getResponse.getField("int").getValue(), equalTo(42));
        assertThat(getResponse.getField("ints").getValues(), contains((Object) 1, 2, 3, 4));
        assertThat((String) getResponse.getField("date").getValue(), equalTo("2012-11-13T15:26:14.000Z"));
        assertThat((BytesReference) getResponse.getField("binary").getValue(), equalTo((BytesReference) new BytesArray(new byte[]{1, 2, 3})));
    }

    @Test
    public void testGetDocWithMultivaluedFields() throws Exception {
        String mapping1 = XContentFactory.jsonBuilder().startObject().startObject("type1")
                .startObject("properties")
                .startObject("field").field("type", "string").field("store", "yes").endObject()
                .endObject()
                .endObject().endObject().string();
        String mapping2 = XContentFactory.jsonBuilder().startObject().startObject("type2")
                .startObject("properties")
                .startObject("field").field("type", "string").field("store", "yes").endObject()
                .endObject()
                .startObject("_source").field("enabled", false).endObject()
                .endObject().endObject().string();
        assertAcked(prepareCreate("test")
                .addMapping("type1", mapping1)
                .addMapping("type2", mapping2)
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));
        ensureGreen();

        GetResponse response = client().prepareGet("test", "type1", "1").get();
        assertThat(response.isExists(), equalTo(false));
        response = client().prepareGet("test", "type2", "1").get();
        assertThat(response.isExists(), equalTo(false));

        client().prepareIndex("test", "type1", "1")
                .setSource(jsonBuilder().startObject().field("field", "1", "2").endObject()).get();

        client().prepareIndex("test", "type2", "1")
                .setSource(jsonBuilder().startObject().field("field", "1", "2").endObject()).get();

        response = client().prepareGet("test", "type1", "1").setFields("field").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getType(), equalTo("type1"));
        assertThat(response.getFields().size(), equalTo(1));
        assertThat(response.getFields().get("field").getValues().size(), equalTo(2));
        assertThat(response.getFields().get("field").getValues().get(0).toString(), equalTo("1"));
        assertThat(response.getFields().get("field").getValues().get(1).toString(), equalTo("2"));


        response = client().prepareGet("test", "type2", "1").setFields("field").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getType(), equalTo("type2"));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getFields().size(), equalTo(1));
        assertThat(response.getFields().get("field").getValues().size(), equalTo(2));
        assertThat(response.getFields().get("field").getValues().get(0).toString(), equalTo("1"));
        assertThat(response.getFields().get("field").getValues().get(1).toString(), equalTo("2"));

        // Now test values being fetched from stored fields.
        refresh();
        response = client().prepareGet("test", "type1", "1").setFields("field").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getFields().size(), equalTo(1));
        assertThat(response.getFields().get("field").getValues().size(), equalTo(2));
        assertThat(response.getFields().get("field").getValues().get(0).toString(), equalTo("1"));
        assertThat(response.getFields().get("field").getValues().get(1).toString(), equalTo("2"));

        response = client().prepareGet("test", "type2", "1").setFields("field").get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getFields().size(), equalTo(1));
        assertThat(response.getFields().get("field").getValues().size(), equalTo(2));
        assertThat(response.getFields().get("field").getValues().get(0).toString(), equalTo("1"));
        assertThat(response.getFields().get("field").getValues().get(1).toString(), equalTo("2"));
    }

    @Test
    public void testThatGetFromTranslogShouldWorkWithExclude() throws Exception {
        String index = "test";
        String type = "type1";

        String mapping = jsonBuilder()
                .startObject()
                .startObject(type)
                .startObject("_source")
                .array("excludes", "excluded")
                .endObject()
                .endObject()
                .endObject()
                .string();

        assertAcked(prepareCreate(index)
                .addMapping(type, mapping)
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));

        client().prepareIndex(index, type, "1")
                .setSource(jsonBuilder().startObject().field("field", "1", "2").field("excluded", "should not be seen").endObject())
                .get();

        GetResponse responseBeforeFlush = client().prepareGet(index, type, "1").get();
        client().admin().indices().prepareFlush(index).get();
        GetResponse responseAfterFlush = client().prepareGet(index, type, "1").get();

        assertThat(responseBeforeFlush.isExists(), is(true));
        assertThat(responseAfterFlush.isExists(), is(true));
        assertThat(responseBeforeFlush.getSourceAsMap(), hasKey("field"));
        assertThat(responseBeforeFlush.getSourceAsMap(), not(hasKey("excluded")));
        assertThat(responseBeforeFlush.getSourceAsString(), is(responseAfterFlush.getSourceAsString()));
    }

    @Test
    public void testThatGetFromTranslogShouldWorkWithInclude() throws Exception {
        String index = "test";
        String type = "type1";

        String mapping = jsonBuilder()
                .startObject()
                .startObject(type)
                .startObject("_source")
                .array("includes", "included")
                .endObject()
                .endObject()
                .endObject()
                .string();

        assertAcked(prepareCreate(index)
                .addMapping(type, mapping)
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));

        client().prepareIndex(index, type, "1")
                .setSource(jsonBuilder().startObject().field("field", "1", "2").field("included", "should be seen").endObject())
                .get();

        GetResponse responseBeforeFlush = client().prepareGet(index, type, "1").get();
        flush();
        GetResponse responseAfterFlush = client().prepareGet(index, type, "1").get();

        assertThat(responseBeforeFlush.isExists(), is(true));
        assertThat(responseAfterFlush.isExists(), is(true));
        assertThat(responseBeforeFlush.getSourceAsMap(), not(hasKey("field")));
        assertThat(responseBeforeFlush.getSourceAsMap(), hasKey("included"));
        assertThat(responseBeforeFlush.getSourceAsString(), is(responseAfterFlush.getSourceAsString()));
    }

    @SuppressWarnings("unchecked")
    @Test
    public void testThatGetFromTranslogShouldWorkWithIncludeExcludeAndFields() throws Exception {
        String index = "test";
        String type = "type1";

        String mapping = jsonBuilder()
                .startObject()
                .startObject(type)
                .startObject("_source")
                .array("includes", "included")
                .array("exlcudes", "excluded")
                .endObject()
                .endObject()
                .endObject()
                .string();

        assertAcked(prepareCreate(index)
                .addMapping(type, mapping)
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));

        client().prepareIndex(index, type, "1")
                .setSource(jsonBuilder().startObject()
                        .field("field", "1", "2")
                        .startObject("included").field("field", "should be seen").field("field2", "extra field to remove").endObject()
                        .startObject("excluded").field("field", "should not be seen").field("field2", "should not be seen").endObject()
                        .endObject())
                .get();

        GetResponse responseBeforeFlush = client().prepareGet(index, type, "1").setFields("_source", "included.field", "excluded.field").get();
        assertThat(responseBeforeFlush.isExists(), is(true));
        assertThat(responseBeforeFlush.getSourceAsMap(), not(hasKey("excluded")));
        assertThat(responseBeforeFlush.getSourceAsMap(), not(hasKey("field")));
        assertThat(responseBeforeFlush.getSourceAsMap(), hasKey("included"));

        // now tests that extra source filtering works as expected
        GetResponse responseBeforeFlushWithExtraFilters = client().prepareGet(index, type, "1").setFields("included.field", "excluded.field")
                .setFetchSource(new String[]{"field", "*.field"}, new String[]{"*.field2"}).get();
        assertThat(responseBeforeFlushWithExtraFilters.isExists(), is(true));
        assertThat(responseBeforeFlushWithExtraFilters.getSourceAsMap(), not(hasKey("excluded")));
        assertThat(responseBeforeFlushWithExtraFilters.getSourceAsMap(), not(hasKey("field")));
        assertThat(responseBeforeFlushWithExtraFilters.getSourceAsMap(), hasKey("included"));
        assertThat((Map<String, Object>) responseBeforeFlushWithExtraFilters.getSourceAsMap().get("included"), hasKey("field"));
        assertThat((Map<String, Object>) responseBeforeFlushWithExtraFilters.getSourceAsMap().get("included"), not(hasKey("field2")));

        flush();
        GetResponse responseAfterFlush = client().prepareGet(index, type, "1").setFields("_source", "included.field", "excluded.field").get();
        GetResponse responseAfterFlushWithExtraFilters = client().prepareGet(index, type, "1").setFields("included.field", "excluded.field")
                .setFetchSource("*.field", "*.field2").get();

        assertThat(responseAfterFlush.isExists(), is(true));
        assertThat(responseBeforeFlush.getSourceAsString(), is(responseAfterFlush.getSourceAsString()));

        assertThat(responseAfterFlushWithExtraFilters.isExists(), is(true));
        assertThat(responseBeforeFlushWithExtraFilters.getSourceAsString(), is(responseAfterFlushWithExtraFilters.getSourceAsString()));
    }

    @Test
    public void testGetWithVersion() {
        assertAcked(prepareCreate("test").addAlias(new Alias("alias"))
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));
        ensureGreen();

        GetResponse response = client().prepareGet("test", "type1", "1").get();
        assertThat(response.isExists(), equalTo(false));

        logger.info("--> index doc 1");
        client().prepareIndex("test", "type1", "1").setSource("field1", "value1", "field2", "value2").get();

        // From translog:

        // version 0 means ignore version, which is the default
        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(0).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getVersion(), equalTo(1l));

        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(1).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getVersion(), equalTo(1l));

        try {
            client().prepareGet(indexOrAlias(), "type1", "1").setVersion(2).get();
            fail();
        } catch (VersionConflictEngineException e) {
            //all good
        }

        // From Lucene index:
        refresh();

        // version 0 means ignore version, which is the default
        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(0).setRealtime(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(1l));

        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(1).setRealtime(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(1l));

        try {
            client().prepareGet(indexOrAlias(), "type1", "1").setVersion(2).setRealtime(false).get();
            fail();
        } catch (VersionConflictEngineException e) {
            //all good
        }

        logger.info("--> index doc 1 again, so increasing the version");
        client().prepareIndex("test", "type1", "1").setSource("field1", "value1", "field2", "value2").get();

        // From translog:

        // version 0 means ignore version, which is the default
        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(0).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(2l));

        try {
            client().prepareGet(indexOrAlias(), "type1", "1").setVersion(1).get();
            fail();
        } catch (VersionConflictEngineException e) {
            //all good
        }

        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(2).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(2l));

        // From Lucene index:
        refresh();

        // version 0 means ignore version, which is the default
        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(0).setRealtime(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(2l));

        try {
            client().prepareGet(indexOrAlias(), "type1", "1").setVersion(1).setRealtime(false).get();
            fail();
        } catch (VersionConflictEngineException e) {
            //all good
        }

        response = client().prepareGet(indexOrAlias(), "type1", "1").setVersion(2).setRealtime(false).get();
        assertThat(response.isExists(), equalTo(true));
        assertThat(response.getId(), equalTo("1"));
        assertThat(response.getIndex(), equalTo("test"));
        assertThat(response.getVersion(), equalTo(2l));
    }

    @Test
    public void testMultiGetWithVersion() throws Exception {
        assertAcked(prepareCreate("test").addAlias(new Alias("alias"))
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));
        ensureGreen();

        MultiGetResponse response = client().prepareMultiGet().add(indexOrAlias(), "type1", "1").get();
        assertThat(response.getResponses().length, equalTo(1));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(false));

        for (int i = 0; i < 3; i++) {
            client().prepareIndex("test", "type1", Integer.toString(i)).setSource("field", "value" + i).get();
        }

        // Version from translog
        response = client().prepareMultiGet()
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(0))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(1))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(2))
                .get();
        assertThat(response.getResponses().length, equalTo(3));
        // [0] version doesn't matter, which is the default
        assertThat(response.getResponses()[0].getFailure(), nullValue());
        assertThat(response.getResponses()[0].getId(), equalTo("1"));
        assertThat(response.getResponses()[0].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[0].getResponse().getSourceAsMap().get("field").toString(), equalTo("value1"));
        assertThat(response.getResponses()[1].getId(), equalTo("1"));
        assertThat(response.getResponses()[1].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[1].getFailure(), nullValue());
        assertThat(response.getResponses()[1].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[1].getResponse().getSourceAsMap().get("field").toString(), equalTo("value1"));
        assertThat(response.getResponses()[2].getFailure(), notNullValue());
        assertThat(response.getResponses()[2].getFailure().getId(), equalTo("1"));
        assertThat(response.getResponses()[2].getFailure().getMessage(), startsWith("VersionConflictEngineException"));

        //Version from Lucene index
        refresh();
        response = client().prepareMultiGet()
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(0))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(1))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "1").version(2))
                .setRealtime(false)
                .get();
        assertThat(response.getResponses().length, equalTo(3));
        // [0] version doesn't matter, which is the default
        assertThat(response.getResponses()[0].getFailure(), nullValue());
        assertThat(response.getResponses()[0].getId(), equalTo("1"));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[0].getResponse().getSourceAsMap().get("field").toString(), equalTo("value1"));
        assertThat(response.getResponses()[1].getId(), equalTo("1"));
        assertThat(response.getResponses()[1].getFailure(), nullValue());
        assertThat(response.getResponses()[1].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[1].getResponse().getSourceAsMap().get("field").toString(), equalTo("value1"));
        assertThat(response.getResponses()[2].getFailure(), notNullValue());
        assertThat(response.getResponses()[2].getFailure().getId(), equalTo("1"));
        assertThat(response.getResponses()[2].getFailure().getMessage(), startsWith("VersionConflictEngineException"));


        for (int i = 0; i < 3; i++) {
            client().prepareIndex("test", "type1", Integer.toString(i)).setSource("field", "value" + i).get();
        }

        // Version from translog
        response = client().prepareMultiGet()
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(0))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(1))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(2))
                .get();
        assertThat(response.getResponses().length, equalTo(3));
        // [0] version doesn't matter, which is the default
        assertThat(response.getResponses()[0].getFailure(), nullValue());
        assertThat(response.getResponses()[0].getId(), equalTo("2"));
        assertThat(response.getResponses()[0].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[0].getResponse().getSourceAsMap().get("field").toString(), equalTo("value2"));
        assertThat(response.getResponses()[1].getFailure(), notNullValue());
        assertThat(response.getResponses()[1].getFailure().getId(), equalTo("2"));
        assertThat(response.getResponses()[1].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[1].getFailure().getMessage(), startsWith("VersionConflictEngineException"));
        assertThat(response.getResponses()[2].getId(), equalTo("2"));
        assertThat(response.getResponses()[2].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[2].getFailure(), nullValue());
        assertThat(response.getResponses()[2].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[2].getResponse().getSourceAsMap().get("field").toString(), equalTo("value2"));


        //Version from Lucene index
        refresh();
        response = client().prepareMultiGet()
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(0))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(1))
                .add(new MultiGetRequest.Item(indexOrAlias(), "type1", "2").version(2))
                .setRealtime(false)
                .get();
        assertThat(response.getResponses().length, equalTo(3));
        // [0] version doesn't matter, which is the default
        assertThat(response.getResponses()[0].getFailure(), nullValue());
        assertThat(response.getResponses()[0].getId(), equalTo("2"));
        assertThat(response.getResponses()[0].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[0].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[0].getResponse().getSourceAsMap().get("field").toString(), equalTo("value2"));
        assertThat(response.getResponses()[1].getFailure(), notNullValue());
        assertThat(response.getResponses()[1].getFailure().getId(), equalTo("2"));
        assertThat(response.getResponses()[1].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[1].getFailure().getMessage(), startsWith("VersionConflictEngineException"));
        assertThat(response.getResponses()[2].getId(), equalTo("2"));
        assertThat(response.getResponses()[2].getIndex(), equalTo("test"));
        assertThat(response.getResponses()[2].getFailure(), nullValue());
        assertThat(response.getResponses()[2].getResponse().isExists(), equalTo(true));
        assertThat(response.getResponses()[2].getResponse().getSourceAsMap().get("field").toString(), equalTo("value2"));
    }

    @Test
    public void testGetFields_metaData() throws Exception {
        assertAcked(prepareCreate("test").addAlias(new Alias("alias"))
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));

        client().prepareIndex("test", "my-type1", "1")
                .setRouting("1")
                .setSource(jsonBuilder().startObject().field("field1", "value").endObject())
                .get();

        GetResponse getResponse = client().prepareGet(indexOrAlias(), "my-type1", "1")
                .setRouting("1")
                .setFields("field1", "_routing")
                .get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField("field1").isMetadataField(), equalTo(false));
        assertThat(getResponse.getField("field1").getValue().toString(), equalTo("value"));
        assertThat(getResponse.getField("_routing").isMetadataField(), equalTo(true));
        assertThat(getResponse.getField("_routing").getValue().toString(), equalTo("1"));

        flush();

        client().prepareGet(indexOrAlias(), "my-type1", "1")
                .setFields("field1", "_routing")
                .setRouting("1")
                .get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField("field1").isMetadataField(), equalTo(false));
        assertThat(getResponse.getField("field1").getValue().toString(), equalTo("value"));
        assertThat(getResponse.getField("_routing").isMetadataField(), equalTo(true));
        assertThat(getResponse.getField("_routing").getValue().toString(), equalTo("1"));
    }

    @Test
    public void testGetFields_nonLeafField() throws Exception {
        assertAcked(prepareCreate("test").addAlias(new Alias("alias"))
                .addMapping("my-type1", jsonBuilder().startObject().startObject("my-type1").startObject("properties")
                        .startObject("field1").startObject("properties")
                        .startObject("field2").field("type", "string").endObject()
                        .endObject().endObject()
                        .endObject().endObject().endObject())
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1)));

        client().prepareIndex("test", "my-type1", "1")
                .setSource(jsonBuilder().startObject().startObject("field1").field("field2", "value1").endObject().endObject())
                .get();

        try {
            client().prepareGet(indexOrAlias(), "my-type1", "1").setFields("field1").get();
            fail();
        } catch (ElasticsearchIllegalArgumentException e) {
            //all well
        }

        flush();

        try {
            client().prepareGet(indexOrAlias(), "my-type1", "1").setFields("field1").get();
            fail();
        } catch (ElasticsearchIllegalArgumentException e) {
            //all well
        }
    }

    @Test
    @TestLogging("index.shard.service:TRACE,cluster.service:TRACE,action.admin.indices.flush:TRACE")
    public void testGetFields_complexField() throws Exception {
        assertAcked(prepareCreate("my-index")
                .setSettings(ImmutableSettings.settingsBuilder().put("index.refresh_interval", -1))
                .addMapping("my-type2", jsonBuilder().startObject().startObject("my-type2").startObject("properties")
                        .startObject("field1").field("type", "object")
                        .startObject("field2").field("type", "object")
                                .startObject("field3").field("type", "object")
                                    .startObject("field4").field("type", "string").field("store", "yes")
                                .endObject()
                            .endObject()
                        .endObject()
                        .endObject().endObject().endObject()));

        BytesReference source = jsonBuilder().startObject()
                    .startArray("field1")
                        .startObject()
                            .startObject("field2")
                                .startArray("field3")
                                    .startObject()
                                        .field("field4", "value1")
                                    .endObject()
                                .endArray()
                            .endObject()
                        .endObject()
                        .startObject()
                            .startObject("field2")
                                .startArray("field3")
                                    .startObject()
                                        .field("field4", "value2")
                                    .endObject()
                                .endArray()
                            .endObject()
                        .endObject()
                    .endArray()
                .endObject().bytes();

        logger.info("indexing documents");

        client().prepareIndex("my-index", "my-type1", "1").setSource(source).get();
        client().prepareIndex("my-index", "my-type2", "1").setSource(source).get();

        logger.info("checking real time retrieval");

        String field = "field1.field2.field3.field4";
        GetResponse getResponse = client().prepareGet("my-index", "my-type1", "1").setFields(field).get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField(field).isMetadataField(), equalTo(false));
        assertThat(getResponse.getField(field).getValues().size(), equalTo(2));
        assertThat(getResponse.getField(field).getValues().get(0).toString(), equalTo("value1"));
        assertThat(getResponse.getField(field).getValues().get(1).toString(), equalTo("value2"));

        getResponse = client().prepareGet("my-index", "my-type2", "1").setFields(field).get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField(field).isMetadataField(), equalTo(false));
        assertThat(getResponse.getField(field).getValues().size(), equalTo(2));
        assertThat(getResponse.getField(field).getValues().get(0).toString(), equalTo("value1"));
        assertThat(getResponse.getField(field).getValues().get(1).toString(), equalTo("value2"));

        logger.info("waiting for recoveries to complete");

        // Flush fails if shard has ongoing recoveries, make sure the cluster is settled down
        ensureGreen();

        logger.info("flushing");
        FlushResponse flushResponse = client().admin().indices().prepareFlush("my-index").setForce(true).get();
        if (flushResponse.getSuccessfulShards() == 0) {
            StringBuilder sb = new StringBuilder("failed to flush at least one shard. total shards [")
                    .append(flushResponse.getTotalShards()).append("], failed shards: [").append(flushResponse.getFailedShards()).append("]");
            for (ShardOperationFailedException failure: flushResponse.getShardFailures()) {
                sb.append("\nShard failure: ").append(failure);
            }
            fail(sb.toString());
        }

        logger.info("checking post-flush retrieval");

        getResponse = client().prepareGet("my-index", "my-type1", "1").setFields(field).get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField(field).isMetadataField(), equalTo(false));
        assertThat(getResponse.getField(field).getValues().size(), equalTo(2));
        assertThat(getResponse.getField(field).getValues().get(0).toString(), equalTo("value1"));
        assertThat(getResponse.getField(field).getValues().get(1).toString(), equalTo("value2"));

        getResponse = client().prepareGet("my-index", "my-type2", "1").setFields(field).get();
        assertThat(getResponse.isExists(), equalTo(true));
        assertThat(getResponse.getField(field).isMetadataField(), equalTo(false));
        assertThat(getResponse.getField(field).getValues().size(), equalTo(2));
        assertThat(getResponse.getField(field).getValues().get(0).toString(), equalTo("value1"));
        assertThat(getResponse.getField(field).getValues().get(1).toString(), equalTo("value2"));
    }

    @Test
    public void testGet_allField() throws Exception {
        assertAcked(prepareCreate("test")
                .addAlias(new Alias("alias"))
                .addMapping("my-type1", jsonBuilder()
                        .startObject()
                        .startObject("my-type1")
                        .startObject("_all")
                        .field("store", true)
                        .endObject()
                        .startObject("properties")
                        .startObject("some_field")
                        .field("type", "string")
                        .endObject()
                        .endObject()
                        .endObject()
                        .endObject()));
        index("test", "my-type1", "1", "some_field", "some text");
        refresh();

        GetResponse getResponse = client().prepareGet(indexOrAlias(), "my-type1", "1").setFields("_all").get();
        assertNotNull(getResponse.getField("_all").getValue());
        assertThat(getResponse.getField("_all").getValue().toString(), equalTo("some text" + " "));
    }

    @Test
    public void testUngeneratedFieldsThatAreNeverStored() throws IOException {
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"doc\": {\n" +
                "      \"_source\": {\n" +
                "        \"enabled\": \"" + randomBoolean() + "\"\n" +
                "      },\n" +
                "      \"properties\": {\n" +
                "        \"suggest\": {\n" +
                "          \"type\": \"completion\"\n" +
                "        }\n" +
                "      }\n" +
                "    }\n" +
                "  }\n" +
                "}";
        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"suggest\": {\n" +
                "    \"input\": [\n" +
                "      \"Nevermind\",\n" +
                "      \"Nirvana\"\n" +
                "    ],\n" +
                "    \"output\": \"Nirvana - Nevermind\"\n" +
                "  }\n" +
                "}";

        index("test", "doc", "1", doc);
        String[] fieldsList = {"suggest"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
    }

    @Test
    public void testUngeneratedFieldsThatAreAlwaysStored() throws IOException {
        String storedString = randomBoolean() ? "yes" : "no";
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"parentdoc\": {},\n" +
                "    \"doc\": {\n" +
                "      \"_source\": {\n" +
                "        \"enabled\": " + randomBoolean() + "\n" +
                "      },\n" +
                "      \"_parent\": {\n" +
                "        \"type\": \"parentdoc\",\n" +
                "        \"store\": \"" + storedString + "\"\n" +
                "      },\n" +
                "      \"_ttl\": {\n" +
                "        \"enabled\": true,\n" +
                "        \"store\": \"" + storedString + "\"\n" +
                "      }\n" +
                "    }\n" +
                "  }\n" +
                "}";
        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"_ttl\": \"1h\"\n" +
                "}";

        client().prepareIndex("test", "doc").setId("1").setSource(doc).setParent("1").get();

        String[] fieldsList = {"_ttl", "_parent"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
    }

    @Test
    public void testUngeneratedFieldsPartOfSourceUnstoredSourceDisabled() throws IOException {
        indexSingleDocumentWithUngeneratedFieldsThatArePartOf_source(false, false);
        String[] fieldsList = {"my_boost"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
    }

    @Test
    public void testUngeneratedFieldsPartOfSourceEitherStoredOrSourceEnabled() throws IOException {
        boolean stored = randomBoolean();
        boolean sourceEnabled = true;
        if (stored) {
            sourceEnabled = randomBoolean();
        }
        indexSingleDocumentWithUngeneratedFieldsThatArePartOf_source(stored, sourceEnabled);
        String[] fieldsList = {"my_boost"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
    }

    void indexSingleDocumentWithUngeneratedFieldsThatArePartOf_source(boolean stored, boolean sourceEnabled) {
        String storedString = stored ? "yes" : "no";
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"doc\": {\n" +
                "      \"_source\": {\n" +
                "        \"enabled\": " + sourceEnabled + "\n" +
                "      },\n" +
                "      \"_boost\": {\n" +
                "        \"name\": \"my_boost\",\n" +
                "        \"null_value\": 1,\n" +
                "        \"store\": \"" + storedString + "\"\n" +
                "      }\n" +
                "    }\n" +
                "  }\n" +
                "}";
        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"my_boost\": 5.0,\n" +
                "  \"_ttl\": \"1h\"\n" +
                "}\n";

        client().prepareIndex("test", "doc").setId("1").setSource(doc).setRouting("1").get();
    }


    @Test
    public void testUngeneratedFieldsNotPartOfSourceUnstored() throws IOException {
        indexSingleDocumentWithUngeneratedFieldsThatAreNeverPartOf_source(false, randomBoolean());
        String[] fieldsList = {"_timestamp", "_size", "_routing"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList, "1");
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList, "1");
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList, "1");
    }

    @Test
    public void testUngeneratedFieldsNotPartOfSourceStored() throws IOException {
        indexSingleDocumentWithUngeneratedFieldsThatAreNeverPartOf_source(true, randomBoolean());
        String[] fieldsList = {"_timestamp", "_size", "_routing"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList, "1");
    }

    void indexSingleDocumentWithUngeneratedFieldsThatAreNeverPartOf_source(boolean stored, boolean sourceEnabled) {
        String storedString = stored ? "yes" : "no";
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"parentdoc\": {},\n" +
                "    \"doc\": {\n" +
                "      \"_timestamp\": {\n" +
                "        \"store\": \"" + storedString + "\",\n" +
                "        \"enabled\": true\n" +
                "      },\n" +
                "      \"_routing\": {\n" +
                "        \"store\": \"" + storedString + "\"\n" +
                "      },\n" +
                "      \"_size\": {\n" +
                "        \"store\": \"" + storedString + "\",\n" +
                "        \"enabled\": true\n" +
                "      }\n" +
                "    }\n" +
                "  }\n" +
                "}";

        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"text\": \"some text.\"\n" +
                "}\n";
        client().prepareIndex("test", "doc").setId("1").setSource(doc).setRouting("1").get();
    }


    @Test
    public void testGeneratedStringFieldsUnstored() throws IOException {
        indexSingleDocumentWithStringFieldsGeneratedFromText(false, randomBoolean());
        String[] fieldsList = {"_all", "_field_names"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
    }

    @Test
    public void testGeneratedStringFieldsStored() throws IOException {
        indexSingleDocumentWithStringFieldsGeneratedFromText(true, randomBoolean());
        String[] fieldsList = {"_all", "_field_names"};
        // before refresh - document is only in translog
        assertGetFieldsNull(indexOrAlias(), "doc", "1", fieldsList);
        assertGetFieldsException(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
    }

    void indexSingleDocumentWithStringFieldsGeneratedFromText(boolean stored, boolean sourceEnabled) {

        String storedString = stored ? "yes" : "no";
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"doc\": {\n" +
                "      \"_source\" : {\"enabled\" : " + sourceEnabled + "}," +
                "      \"_all\" : {\"enabled\" : true, \"store\":\"" + storedString + "\" }," +
                "      \"_field_names\" : {\"store\":\"" + storedString + "\" }" +
                "    }\n" +
                "  }\n" +
                "}";

        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"text1\": \"some text.\"\n," +
                "  \"text2\": \"more text.\"\n" +
                "}\n";
        index("test", "doc", "1", doc);
    }


    @Test
    public void testGeneratedNumberFieldsUnstored() throws IOException {
        indexSingleDocumentWithNumericFieldsGeneratedFromText(false, randomBoolean());
        String[] fieldsList = {"token_count", "text.token_count", "murmur", "text.murmur"};
        // before refresh - document is only in translog
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysNull(indexOrAlias(), "doc", "1", fieldsList);
    }

    @Test
    public void testGeneratedNumberFieldsStored() throws IOException {
        indexSingleDocumentWithNumericFieldsGeneratedFromText(true, randomBoolean());
        String[] fieldsList = {"token_count", "text.token_count", "murmur", "text.murmur"};
        // before refresh - document is only in translog
        assertGetFieldsNull(indexOrAlias(), "doc", "1", fieldsList);
        assertGetFieldsException(indexOrAlias(), "doc", "1", fieldsList);
        refresh();
        //after refresh - document is in translog and also indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
        flush();
        //after flush - document is in not anymore translog - only indexed
        assertGetFieldsAlwaysWorks(indexOrAlias(), "doc", "1", fieldsList);
    }

    void indexSingleDocumentWithNumericFieldsGeneratedFromText(boolean stored, boolean sourceEnabled) {
        String storedString = stored ? "yes" : "no";
        String createIndexSource = "{\n" +
                "  \"settings\": {\n" +
                "    \"index.translog.disable_flush\": true,\n" +
                "    \"refresh_interval\": \"-1\"\n" +
                "  },\n" +
                "  \"mappings\": {\n" +
                "    \"doc\": {\n" +
                "      \"_source\" : {\"enabled\" : " + sourceEnabled + "}," +
                "      \"properties\": {\n" +
                "        \"token_count\": {\n" +
                "          \"type\": \"token_count\",\n" +
                "          \"analyzer\": \"standard\",\n" +
                "          \"store\": \"" + storedString + "\"" +
                "        },\n" +
                "        \"murmur\": {\n" +
                "          \"type\": \"murmur3\",\n" +
                "          \"store\": \"" + storedString + "\"" +
                "        },\n" +
                "        \"text\": {\n" +
                "          \"type\": \"string\",\n" +
                "          \"fields\": {\n" +
                "            \"token_count\": {\n" +
                "              \"type\": \"token_count\",\n" +
                "              \"analyzer\": \"standard\",\n" +
                "              \"store\": \"" + storedString + "\"" +
                "            },\n" +
                "            \"murmur\": {\n" +
                "              \"type\": \"murmur3\",\n" +
                "              \"store\": \"" + storedString + "\"" +
                "            }\n" +
                "          }\n" +
                "        }" +
                "      }\n" +
                "    }\n" +
                "  }\n" +
                "}";

        assertAcked(prepareCreate("test").addAlias(new Alias("alias")).setSource(createIndexSource));
        ensureGreen();
        String doc = "{\n" +
                "  \"murmur\": \"Some value that can be hashed\",\n" +
                "  \"token_count\": \"A text with five words.\",\n" +
                "  \"text\": \"A text with five words.\"\n" +
                "}\n";
        index("test", "doc", "1", doc);
    }

    private void assertGetFieldsAlwaysWorks(String index, String type, String docId, String[] fields) {
        assertGetFieldsAlwaysWorks(index, type, docId, fields, null);
    }

    private void assertGetFieldsAlwaysWorks(String index, String type, String docId, String[] fields, @Nullable String routing) {
        for (String field : fields) {
            assertGetFieldWorks(index, type, docId, field, false, routing);
            assertGetFieldWorks(index, type, docId, field, true, routing);
        }
    }

    private void assertGetFieldWorks(String index, String type, String docId, String field, boolean ignoreErrors, @Nullable String routing) {
        GetResponse response = getDocument(index, type, docId, field, ignoreErrors, routing);
        assertThat(response.getId(), equalTo(docId));
        assertTrue(response.isExists());
        assertNotNull(response.getField(field));
        response = multiGetDocument(index, type, docId, field, ignoreErrors, routing);
        assertThat(response.getId(), equalTo(docId));
        assertTrue(response.isExists());
        assertNotNull(response.getField(field));
    }

    protected void assertGetFieldsException(String index, String type, String docId, String[] fields) {
        for (String field : fields) {
            assertGetFieldException(index, type, docId, field);
        }
    }

    private void assertGetFieldException(String index, String type, String docId, String field) {
        try {
            client().prepareGet().setIndex(index).setType(type).setId(docId).setFields(field).setIgnoreErrorsOnGeneratedFields(false).get();
            fail();
        } catch (ElasticsearchException e) {
            assertTrue(e.getMessage().contains("You can only get this field after refresh() has been called."));
        }
        MultiGetResponse multiGetResponse = client().prepareMultiGet().add(new MultiGetRequest.Item(index, type, docId).fields(field)).setIgnoreErrorsOnGeneratedFields(false).get();
        assertNull(multiGetResponse.getResponses()[0].getResponse());
        assertTrue(multiGetResponse.getResponses()[0].getFailure().getMessage().contains("You can only get this field after refresh() has been called."));
    }

    protected void assertGetFieldsNull(String index, String type, String docId, String[] fields) {
        assertGetFieldsNull(index, type, docId, fields, null);
    }

    protected void assertGetFieldsNull(String index, String type, String docId, String[] fields, @Nullable String routing) {
        for (String field : fields) {
            assertGetFieldNull(index, type, docId, field, true, routing);
        }
    }

    protected void assertGetFieldsAlwaysNull(String index, String type, String docId, String[] fields) {
        assertGetFieldsAlwaysNull(index, type, docId, fields, null);
    }

    protected void assertGetFieldsAlwaysNull(String index, String type, String docId, String[] fields, @Nullable String routing) {
        for (String field : fields) {
            assertGetFieldNull(index, type, docId, field, true, routing);
            assertGetFieldNull(index, type, docId, field, false, routing);
        }
    }

    protected void assertGetFieldNull(String index, String type, String docId, String field, boolean ignoreErrors, @Nullable String routing) {
        //for get
        GetResponse response = getDocument(index, type, docId, field, ignoreErrors, routing);
        assertTrue(response.isExists());
        assertNull(response.getField(field));
        assertThat(response.getId(), equalTo(docId));
        //same for multi get
        response = multiGetDocument(index, type, docId, field, ignoreErrors, routing);
        assertNull(response.getField(field));
        assertThat(response.getId(), equalTo(docId));
        assertTrue(response.isExists());
    }

    private GetResponse multiGetDocument(String index, String type, String docId, String field, boolean ignoreErrors, @Nullable String routing) {
        MultiGetRequest.Item getItem = new MultiGetRequest.Item(index, type, docId).fields(field);
        if (routing != null) {
            getItem.routing(routing);
        }
        MultiGetRequestBuilder multiGetRequestBuilder = client().prepareMultiGet().add(getItem).setIgnoreErrorsOnGeneratedFields(ignoreErrors);
        MultiGetResponse multiGetResponse = multiGetRequestBuilder.get();
        assertThat(multiGetResponse.getResponses().length, equalTo(1));
        return multiGetResponse.getResponses()[0].getResponse();
    }

    private GetResponse getDocument(String index, String type, String docId, String field, boolean ignoreErrors, @Nullable String routing) {
        GetRequestBuilder getRequestBuilder = client().prepareGet().setIndex(index).setType(type).setId(docId).setFields(field).setIgnoreErrorsOnGeneratedFields(ignoreErrors);
        if (routing != null) {
            getRequestBuilder.setRouting(routing);
        }
        return getRequestBuilder.get();
    }
}
