/*
 * (C) Copyright 2002, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Local configuration
#include "config.h"

// Implementation
#include "XMLTree.hpp"

// STDC++
#include <string>
#include <memory>
#include <cassert>

// C libraries
#include <libxslt/xsltInternals.h>
#include <libxml/parser.h>

// Local
#include "Util.hpp"
#include "XMLNodeSet.hpp"
#include "XMLException.hpp"
#include "XMLDump.hpp"


namespace SP {
namespace GXML {

// XPathContext class
//
// For 1.0 stable, this is an internal class only so we do not change symbol table
// 1.2 should update this class, and use it thoroughly
// Note: xpath context associated to xmlDocs are not constant, and cannot be shared between threads.
class XPathContext
{
public:
	XPathContext(xmlDocPtr const doc)
		:_context(xmlXPathNewContext(doc))
	{
		if (!_context)
		{
			throw XMLException(XMLException::XMLTREE_CREATE_CONTEXT);
		}
	}

	~XPathContext()
	{
		xmlXPathFreeContext(_context);
	}

	xmlXPathContextPtr get() const
	{
		return _context;
	}

private:
	xmlXPathContextPtr const _context;
};

void
XMLTree::genTree(const char * xmlBuffer, int size)
{
	assert(xmlBuffer);

	// if size < 0, we must have a valid, NULL-Terminated C-String
	if (size < 0) size = ::strlen(xmlBuffer);

	// xmlParseMemory does not change Buffer xmlBuffer (const, cast)
	tree_ = xmlParseMemory(xmlBuffer, size);
	if (!tree_)
	{
	  throw XMLException( XMLException::XMLTREE_PARSE );
	}
	context_ = xmlXPathNewContext(tree_);
	if (!context_)
	{
	  xmlFreeDoc(tree_);
	  throw XMLException( XMLException::XMLTREE_CREATE_CONTEXT );
	}
}

XMLTree::XMLTree(const xmlDocPtr doc) :tree_(doc)
{
	assert(tree_);
	context_ = xmlXPathNewContext(tree_);
	if (!context_)
	{
		xmlFreeDoc(tree_);
		throw XMLException( XMLException::XMLTREE_CREATE_CONTEXT );
	}
}

XMLTree::XMLTree(const char * xmlBuffer, int size)
{
	genTree(xmlBuffer, size);
}

XMLTree::XMLTree(const std::string & xmlString)
{
	genTree(xmlString.c_str());
}

XMLTree::XMLTree(std::ifstream & xmlStream)
{
	assert(xmlStream);
	genTree(istream2String(xmlStream).c_str());
}

XMLTree::XMLTree(std::istream & xmlStream)
{
	assert(xmlStream);
	genTree(istream2String(xmlStream).c_str());
}

XMLTree::~XMLTree()
{
	xmlXPathFreeContext(context_);
	xmlFreeDoc(tree_);
}

std::string
XMLTree::getString( const std::string& xpath )
{
	xmlXPathObjectPtr obj = createXPathObject( xpath );

	if ( obj->stringval == NULL || obj->type != XPATH_STRING ) {
		throw XMLException( XMLException::XMLTREE_NO_STRING_FROM_PATH );
	}
	assert( obj->stringval );

	std::string result = (char*)obj->stringval;
	destroyXPathObject( obj );
	return result;
}

bool
XMLTree::getBool( const std::string& xpath )
{
	bool result;
	xmlXPathObjectPtr obj = createXPathObject( xpath );

	if ( obj->type != XPATH_BOOLEAN ) {
		throw XMLException( XMLException::XMLTREE_NO_BOOL_FROM_PATH );
	}

	if ( obj->boolval == 1 ) {
		result = true;
	} else if ( obj->boolval == 0 ) {
		result = false;
	} else {
		throw XMLException( XMLException::XMLTREE_NO_BOOL_FROM_PATH );
	}

	destroyXPathObject( obj );
	return result;
}

double
XMLTree::getFloat( const std::string& xpath )
{
	double result;
	xmlXPathObjectPtr obj = createXPathObject( xpath );

	if ( obj->type != XPATH_NUMBER ) {
		throw XMLException( XMLException::XMLTREE_NO_FLOAT_FROM_PATH );
	} else {
		result = obj->floatval;
	}
	destroyXPathObject( obj );
	return result;
}

std::auto_ptr<XMLNodeSet>
XMLTree::getNodeSet( const std::string& xpath )
{
	return std::auto_ptr<XMLNodeSet>( new XMLNodeSet( xpath, this ) );
}

//
// libxml2 Pointers; these are a Mockery to Abstraction
//
xmlDocPtr XMLTree::getDocPtr() const
{
	return tree_;
}

xmlXPathContextPtr
XMLTree::getXPathContextPtr() const
{
	return context_;
}

//
// Get Values from XPath Expressions
//

// This is the master "getValue" function
// STS "xmlNodeGetContent" might do the same ?
xmlChar *
XMLTree::getXmlCharValue(const xmlChar * path) const
{
	XPathContext context(tree_);
	xmlChar * result(0);
	xmlNodePtr node = nodeFromPath(context.get(), path);

	if (node && node->children) {
		result = node->children->content;
	}
	return result;
}

xmlChar *
XMLTree::getXmlCharValue(const std::string & path) const
{
	return getXmlCharValue((xmlChar *) path.c_str());
}

char *
XMLTree::getAddrValue(const std::string & path) const
{
	return (char *) getXmlCharValue(path);
}

std::string
XMLTree::getValue(const std::string & path) const
{
	char * result = getAddrValue(path);
	if (result) {
		return result;
	} else {
		return "";
	}
}

char *
XMLTree::getAddrName(const std::string & path) const
{
	XPathContext context(tree_);

	char * result(0);
	xmlNodePtr node = nodeFromPath(context.get(), path);
	if (node) {
		result = (char *) node->name;
	}
	return result;
}

std::string
XMLTree::getName(const std::string & path) const
{
	char * result(getAddrName(path));
	if (result) {
		return result;
	} else {
		return "";
	}
}

void
XMLTree::setValue(const std::string & path, char * value)
{
	XPathContext context(tree_);

	xmlNodePtr node = XMLTree::nodeFromPath(context.get(), path);
	if (!node) {
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}
	xmlNodeSetContent(node, (xmlChar *) value);
}

void
XMLTree::setValue(const std::string & path, const std::string & value)
{
	setValue(path, (char *) value.c_str());
}

std::string 
XMLTree::getSiblingXML(const std::string& path) {
	
	std::string result;
	
	if ( path == "/" ) { // Note: xmlNodeDump segfaults with path="/" see XMLDump.h
		result = getXML();
	} else {
		std::auto_ptr<XMLNodeSet> ns = this->getNodeSet( path );
		for ( XMLNodeSet::Iterator i = (ns.get())->begin() ; i != (ns.get())->end(); ++i ) {
			result.append((*i).getNodeDump());
		}
	}
	return result;
}


std::string
XMLTree::getXML(const std::string & path) const
{
	return XMLDump(this, path).get();
}

int
XMLTree::getCount(const xmlChar * path) const
{
	XPathContext context(tree_);

	int result(0);
	xmlXPathObjectPtr obj = xmlXPathEvalExpression(path, context.get());

	if (obj) {
		if (obj->nodesetval)
			result = obj->nodesetval->nodeNr;
		xmlXPathFreeObject(obj);
	}
	return result;
}

int
XMLTree::getCount(const char * path) const
{
	return getCount((xmlChar *) path);
}

int
XMLTree::getCount(const std::string & path) const
{
	return getCount(path.c_str());
}


void
XMLTree::delTag(const std::string & path)
{
	XPathContext context(tree_);

	xmlNodePtr node = XMLTree::nodeFromPath(context.get(), path);
	if (!node) {
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}
	xmlUnlinkNode(node);
	xmlFreeNode(node);
}

void
XMLTree::addTag(const std::string & path, const std::string & name, const std::string & content)
{
	XPathContext context(tree_);

	xmlNodePtr node = nodeFromPath(context.get(), path);
	if (!node) {
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}
	xmlNodePtr newNode = xmlNewChild(node, 0, (xmlChar *) name.c_str(), (xmlChar *) content.c_str());
	if (!newNode) {
		throw XMLException( XMLException::XMLTREE_CREATE_NODE );
	}
}

void
XMLTree::addSiblingTag(const std::string & path, const std::string & name, const std::string & content)
{
	XPathContext context(tree_);

	xmlNodePtr node = nodeFromPath(context.get(), path);
	if (!node) {
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}
	xmlNodePtr inNode = xmlNewNode( NULL, (xmlChar *)name.c_str() );
	if (inNode) {
		xmlNodeAddContent( inNode, (xmlChar *)content.c_str() );
	} else {
		throw XMLException( XMLException::XMLTREE_CREATE_NODE );
	}

	xmlNodePtr newNode = xmlAddNextSibling( node, inNode );
	if (!newNode)
	{
		throw XMLException( XMLException::XMLTREE_CREATE_NODE );
	}
}


void
XMLTree::addTree(const std::string & path, XMLTree * tree)
{
	assert(tree);
	XPathContext context(tree_);

	xmlNodePtr node(XMLTree::nodeFromPath(context.get(), path));
	if (!node)
	{
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}

	std::auto_ptr<XMLNodeSet> ns(tree->getNodeSet("/*"));
	for (XMLNodeSet::Iterator i(ns.get()->begin()); i != ns.get()->end(); ++i)
	{
		xmlNodePtr newNode(xmlCopyNode((*i).getNodePtr(), 1));

		if (!newNode)
		{
			throw XMLException(XMLException::XMLTREE_CREATE_NODE);
		}
		if (xmlAddChild(node, newNode) == 0)
		{
			xmlFreeNode(newNode);
			throw XMLException(XMLException::XMLTREE_ADD_NODE);
		}
	}
}

void
XMLTree::addXML(const std::string & path, const std::string & xml)
{
	std::auto_ptr<XMLTree> xmlTree(new XMLTree(xml));
	return addTree(path, xmlTree.get());
}

////////////////////////////////////////////////////////////////////////
// Utility functions
//
xmlNodePtr
XMLTree::nodeFromPath(xmlXPathContextPtr context, const xmlChar * path)
{
	assert(context);

	xmlNodePtr result(0);
	xmlXPathObjectPtr obj = xmlXPathEvalExpression(path, context);

	if (obj) {
		if (obj->nodesetval && (obj->nodesetval->nodeNr > 0)) {
			result = obj->nodesetval->nodeTab[0];
		}
		xmlXPathFreeObject(obj);
	}
	return result;
}

xmlNodePtr
XMLTree::nodeFromPath(xmlXPathContextPtr context, const std::string & path)
{
	return nodeFromPath(context, (xmlChar *) path.c_str());
}

std::auto_ptr<XMLTree>
XMLTree::getTree(const std::string & path)
{
	return std::auto_ptr<XMLTree>(new XMLTree(getXML(path)));
}

xmlXPathObjectPtr
XMLTree::createXPathObject( const std::string& xpath )
{
	XPathContext context(tree_);

	xmlXPathObjectPtr obj = xmlXPathEval( (xmlChar*)xpath.c_str(), context.get());
	if ( obj == NULL )
	{
		throw XMLException( XMLException::XMLTREE_NO_NODE_FROM_PATH );
	}
	assert( obj );
	return obj;
}

void
XMLTree::destroyXPathObject( xmlXPathObjectPtr obj )
{
	xmlXPathFreeObject(obj);
}

}}
