#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from functools import wraps
from sql import Table

from trytond.model import Workflow, fields
from trytond.pool import Pool, PoolMeta
from trytond.transaction import Transaction
from trytond import backend

__all__ = ['Invoice', 'InvoiceLine']
__metaclass__ = PoolMeta


def process_sale(func):
    @wraps(func)
    def wrapper(cls, invoices):
        pool = Pool()
        Sale = pool.get('sale.sale')
        func(cls, invoices)
        with Transaction().set_user(0, set_context=True):
            Sale.process([s for i in cls.browse(invoices)
                    for s in i.sales])
    return wrapper


class Invoice:
    __name__ = 'account.invoice'
    sales = fields.Many2Many('sale.sale-account.invoice',
            'invoice', 'sale', 'Sales', readonly=True)
    sale_exception_state = fields.Function(fields.Selection([
        ('', ''),
        ('ignored', 'Ignored'),
        ('recreated', 'Recreated'),
        ], 'Exception State'), 'get_sale_exception_state')

    @classmethod
    def __setup__(cls):
        super(Invoice, cls).__setup__()
        cls._error_messages.update({
                'delete_sale_invoice': ('You can not delete invoices '
                    'that come from a sale.'),
                'reset_invoice_sale': ('You cannot reset to draft '
                    'an invoice generated by a sale.'),
                })

    @classmethod
    def get_sale_exception_state(cls, invoices, name):
        Sale = Pool().get('sale.sale')
        with Transaction().set_user(0, set_context=True):
            sales = Sale.search([
                    ('invoices', 'in', [i.id for i in invoices]),
                    ])

        recreated = tuple(i for p in sales for i in p.invoices_recreated)
        ignored = tuple(i for p in sales for i in p.invoices_ignored)

        states = {}
        for invoice in invoices:
            states[invoice.id] = ''
            if invoice in recreated:
                states[invoice.id] = 'recreated'
            elif invoice.id in ignored:
                states[invoice.id] = 'ignored'
        return states

    @classmethod
    def copy(cls, invoices, default=None):
        if default is None:
            default = {}
        default = default.copy()
        default.setdefault('sales', None)
        return super(Invoice, cls).copy(invoices, default=default)

    @classmethod
    def delete(cls, invoices):
        pool = Pool()
        Sale_Invoice = pool.get('sale.sale-account.invoice')
        sale_invoice = Sale_Invoice.__table__()
        cursor = Transaction().cursor
        if invoices:
            cursor.execute(*sale_invoice.select(sale_invoice.id,
                    where=sale_invoice.invoice.in_(
                        [i.id for i in invoices])))
            if cursor.fetchone():
                cls.raise_user_error('delete_sale_invoice')
        super(Invoice, cls).delete(invoices)

    @classmethod
    @process_sale
    def post(cls, invoices):
        super(Invoice, cls).post(invoices)

    @classmethod
    @process_sale
    def paid(cls, invoices):
        super(Invoice, cls).paid(invoices)

    @classmethod
    @process_sale
    def cancel(cls, invoices):
        super(Invoice, cls).cancel(invoices)

    @classmethod
    @Workflow.transition('draft')
    def draft(cls, invoices):
        Sale = Pool().get('sale.sale')
        with Transaction().set_user(0, set_context=True):
            sales = Sale.search([
                    ('invoices', 'in', [i.id for i in invoices]),
                    ])
        if sales and any(i.state == 'cancel' for i in invoices):
            cls.raise_user_error('reset_invoice_sale')

        return super(Invoice, cls).draft(invoices)


class InvoiceLine:
    __name__ = 'account.invoice.line'

    @classmethod
    def __register__(cls, module_name):
        TableHandler = backend.get('TableHandler')
        cursor = Transaction().cursor
        sql_table = cls.__table__()

        super(InvoiceLine, cls).__register__(module_name)

        # Migration from 2.6: remove sale_lines
        rel_table_name = 'sale_line_invoice_lines_rel'
        if TableHandler.table_exist(cursor, rel_table_name):
            rel_table = Table(rel_table_name)
            cursor.execute(*rel_table.select(
                    rel_table.sale_line, rel_table.invoice_line))
            for sale_line, invoice_line in cursor.fetchall():
                cursor.execute(*sql_table.update(
                        columns=[sql_table.origin],
                        values=['sale.line,%s' % sale_line],
                        where=sql_table.id == invoice_line))
            TableHandler.drop_table(cursor,
                'sale.line-account.invoice.line', rel_table_name)

    @property
    def origin_name(self):
        pool = Pool()
        SaleLine = pool.get('sale.line')
        name = super(InvoiceLine, self).origin_name
        if isinstance(self.origin, SaleLine):
            name = self.origin.sale.rec_name
        return name

    @classmethod
    def _get_origin(cls):
        models = super(InvoiceLine, cls)._get_origin()
        models.append('sale.line')
        return models
