/* $NetBSD: lca.c,v 1.58 2023/12/04 00:32:10 thorpej Exp $ */

/*-
 * Copyright (c) 2000 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1995, 1996 Carnegie-Mellon University.
 * All rights reserved.
 *
 * Authors: Jeffrey Hsu and Chris G. Demetriou
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND
 * FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie the
 * rights to redistribute these changes.
 */

#include <sys/cdefs.h>			/* RCS ID & Copyright macro defns */

__KERNEL_RCSID(0, "$NetBSD: lca.c,v 1.58 2023/12/04 00:32:10 thorpej Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>

#include <uvm/uvm_extern.h>

#include <machine/autoconf.h>
#include <machine/rpb.h>
#include <machine/sysarch.h>

#include <dev/isa/isareg.h>
#include <dev/isa/isavar.h>

#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>
#include <alpha/pci/lcareg.h>
#include <alpha/pci/lcavar.h>

static int	lcamatch(device_t, cfdata_t, void *);
static void	lcaattach(device_t, device_t, void *);

CFATTACH_DECL_NEW(lca, sizeof(struct lca_softc),
    lcamatch, lcaattach, NULL, NULL);

extern struct cfdriver lca_cd;

static int	lca_bus_get_window(int, int,
		    struct alpha_bus_space_translation *);

/* There can be only one. */
struct lca_config lca_configuration;
static int lcafound;

static const u_int lca_bcache_sizes[__SHIFTOUT(MEMC_CAR_BCS, MEMC_CAR_BCS)] = {
[BCS_64K]	=	64 * 1024,
[BCS_128K]	=	128 * 1024,
[BCS_256K]	=	256 * 1024,
[BCS_512K]	=	512 * 1024,
[BCS_1M]	=	1024 * 1024,
[BCS_2M]	=	2048 * 1024,
/* Other values are reserved and we treat them as 0. */
};

static int
lcamatch(device_t parent, cfdata_t match, void *aux)
{
	struct mainbus_attach_args *ma = aux;

	/* Make sure that we're looking for a LCA. */
	if (strcmp(ma->ma_name, lca_cd.cd_name) != 0)
		return (0);

	if (lcafound)
		return (0);

	return (1);
}

/*
 * Probe the memory controller's Bcache configuration.
 */
void
lca_probe_bcache(void)
{
	const uint64_t car = REGVAL64(LCA_MEMC_CAR);

	if (lca_configuration.lc_bcache_size != 0) {
		/* Already done. */
		return;
	}

	if (car & MEMC_CAR_BCE) {
		lca_configuration.lc_bcache_size =
		    lca_bcache_sizes[__SHIFTOUT(car, MEMC_CAR_BCS)];
	} else {
		lca_configuration.lc_bcache_size = 0;
	}

	if (lca_configuration.lc_bcache_size) {
		uvmexp.ncolors = atop(lca_configuration.lc_bcache_size);
	}
}

/*
 * Set up the chipset's function pointers.
 */
void
lca_init(struct lca_config *lcp)
{

	/*
	 * The LCA HAE register is WRITE-ONLY, so we can't tell where
	 * the second sparse window is actually mapped.  Therefore,
	 * we have to guess where it is.  This seems to be the normal
	 * address.
	 */
	lcp->lc_s_mem_w2_masked_base = 0x80000000;

	if (!lcp->lc_initted) {
		/* don't do these twice since they set up extents */
		lca_bus_io_init(&lcp->lc_iot, lcp);
		lca_bus_mem_init(&lcp->lc_memt, lcp);

		/*
		 * We have 1 I/O window and 3 MEM windows.
		 */
		alpha_bus_window_count[ALPHA_BUS_TYPE_PCI_IO] = 1;
		alpha_bus_window_count[ALPHA_BUS_TYPE_PCI_MEM] = 3;
		alpha_bus_get_window = lca_bus_get_window;
	}

	lca_pci_init(&lcp->lc_pc, lcp);
	alpha_pci_chipset = &lcp->lc_pc;

	/*
	 * Refer to ``DECchip 21066 and DECchip 21068 Alpha AXP Microprocessors
	 * Hardware Reference Manual''.
	 * ...
	 */

	/*
	 * According to section 6.4.1, all bits of the IOC_HAE register are
	 * undefined after reset.  Bits <31:27> are write-only.  However, we
	 * cannot blindly set it to zero.  The serial ROM code that initializes
	 * the PCI devices' address spaces, allocates sparse memory blocks in
	 * the range that must use the IOC_HAE register for address translation,
	 * and sets this register accordingly (see section 6.4.14).
	 *
	 *	IOC_HAE left AS IS.
	 */

	/* According to section 6.4.2, all bits of the IOC_CONF register are
	 * undefined after reset.  Bits <1:0> are write-only.  Set them to
	 * 0x00 for PCI Type 0 configuration access.
	 *
	 *	IOC_CONF set to ZERO.
	 */
	REGVAL64(LCA_IOC_CONF) = 0;

	lcp->lc_initted = 1;
}

static void
lcaattach(device_t parent, device_t self, void *aux)
{
	struct lca_softc *sc = device_private(self);
	struct lca_config *lcp;
	struct pcibus_attach_args pba;

	/* note that we've attached the chipset; can't have 2 LCAs. */
	lcafound = 1;
	sc->sc_dev = self;

	/*
	 * set up the chipset's info; done once at console init time
	 * (maybe), but we must do it twice to take care of things
	 * that need to use memory allocation.
	 */
	lcp = sc->sc_lcp = &lca_configuration;
	lca_init(lcp);

	/* XXX print chipset information */
	aprint_normal("\n");
	if (lcp->lc_bcache_size != 0) {
		char buf[sizeof("256 KB")];
		if (format_bytes(buf, sizeof(buf), lcp->lc_bcache_size) > 0) {
			aprint_normal_dev(self, "%s Bcache detected\n", buf);
		}
	}

	lca_dma_init(lcp);

	alpha_pci_intr_init(lcp, &lcp->lc_iot, &lcp->lc_memt, &lcp->lc_pc);

	pba.pba_iot = &lcp->lc_iot;
	pba.pba_memt = &lcp->lc_memt;
	pba.pba_dmat =
	    alphabus_dma_get_tag(&lcp->lc_dmat_direct, ALPHA_BUS_PCI);
	pba.pba_dmat64 = NULL;
	pba.pba_pc = &lcp->lc_pc;
	pba.pba_bus = 0;
	pba.pba_bridgetag = NULL;
	pba.pba_flags = PCI_FLAGS_IO_OKAY | PCI_FLAGS_MEM_OKAY |
	    PCI_FLAGS_MRL_OKAY | PCI_FLAGS_MRM_OKAY | PCI_FLAGS_MWI_OKAY;
	config_found(self, &pba, pcibusprint, CFARGS_NONE);
}

static int
lca_bus_get_window(int type, int window,
    struct alpha_bus_space_translation *abst)
{
	struct lca_config *lcp = &lca_configuration;
	bus_space_tag_t st;

	switch (type) {
	case ALPHA_BUS_TYPE_PCI_IO:
		st = &lcp->lc_iot;
		break;

	case ALPHA_BUS_TYPE_PCI_MEM:
		st = &lcp->lc_memt;
		break;

	default:
		panic("lca_bus_get_window");
	}

	return (alpha_bus_space_get_window(st, window, abst));
}
