package index

import (
	"bytes"
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"sync"
	"sync/atomic"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	"git.sr.ht/~whynothugo/ImapGoose/internal/maildir"
	"git.sr.ht/~whynothugo/ImapGoose/internal/status"
	imap2 "github.com/emersion/go-imap/v2"
	"golang.org/x/sync/semaphore"
)

const CONCURRENCY = 16 // Amount of concurrent operations.

// LocalMessage represents a local Maildir message with U= hint.
type LocalMessage struct {
	AbsPath      string
	TentativeUID imap2.UID
	Flags        []imap2.Flag
}

// accountIndexer indexes status database for an account.
type accountIndexer struct {
	client     *imap.Client
	statusRepo *status.Repository
	localPath  string
	logger     *slog.Logger
}

// IndexAccount indexes the status database for an account from existing Maildir.
// Scans local files with U= hints and verifies them against IMAP server.
func IndexAccount(ctx context.Context, account config.Account, logger *slog.Logger) error {
	logger.Info("Indexing from local Maildir", "account", account.Name)

	statusRepo := status.New(account.Name)
	if err := statusRepo.Init(); err != nil {
		return fmt.Errorf("failed to initialize status repository: %w", err)
	}
	defer func() {
		_ = statusRepo.Close() // Best effort close
	}()

	if err := account.ResolvePassword(); err != nil {
		return fmt.Errorf("failed to resolve password: %w", err)
	}

	client, err := imap.Connect(ctx, account.Server, account.Username, account.Password, account.Plaintext, nil, logger)
	if err != nil {
		return fmt.Errorf("failed to connect to IMAP: %w", err)
	}
	defer func() {
		_ = client.Close() // Best effort close
	}()

	mailboxes, err := client.ListMailboxes(ctx)
	if err != nil {
		return fmt.Errorf("failed to list mailboxes: %w", err)
	}

	logger.Info("Discovered mailboxes", "count", len(mailboxes))

	idx := &accountIndexer{
		client:     client,
		statusRepo: statusRepo,
		localPath:  account.LocalPath,
		logger:     logger,
	}

	var totalIdentical, totalMismatched, totalMissing int
	for _, mbox := range mailboxes {
		identical, mismatched, missing, err := idx.indexMailbox(ctx, mbox)
		if err != nil {
			logger.Warn("Failed to index mailbox", "mailbox", mbox, "error", err)
			continue
		}
		totalIdentical += identical
		totalMismatched += mismatched
		totalMissing += missing
	}

	logger.Info("Account indexed",
		"account", account.Name,
		"identical", totalIdentical,
		"mismatched", totalMismatched,
		"missing", totalMissing,
	)

	return nil
}

// indexMailbox indexes a single mailbox.
func (idx *accountIndexer) indexMailbox(
	ctx context.Context,
	mailbox string,
) (identical, mismatched, missing int, err error) {
	md := maildir.New(idx.localPath, mailbox, idx.logger)

	idx.logger.Info("Processing mailbox", "mailbox", mailbox)

	selectResult, err := idx.client.SelectMailbox(ctx, mailbox, nil)
	if err != nil {
		return 0, 0, 0, fmt.Errorf("failed to select mailbox: %w", err)
	}

	if err := idx.statusRepo.SetUIDValidity(mailbox, selectResult.UIDValidity); err != nil {
		return 0, 0, 0, fmt.Errorf("failed to set UIDValidity: %w", err)
	}

	localMessages := make(chan LocalMessage, CONCURRENCY)
	scanErr := make(chan error, 1)
	go func() {
		defer close(localMessages)
		scanErr <- scanLocalWithHints(ctx, md, localMessages)
	}()

	identical, mismatched, missing, err = compareMessages(
		ctx,
		idx.client,
		idx.statusRepo,
		mailbox,
		localMessages,
		idx.logger,
	)
	if err != nil {
		return 0, 0, 0, err
	}

	if err := <-scanErr; err != nil {
		return 0, 0, 0, fmt.Errorf("failed to scan local maildir: %w", err)
	}

	idx.logger.Info("Mailbox indexed",
		"mailbox", mailbox,
		"identical", identical,
		"mismatched", mismatched,
		"missing", missing,
	)

	return identical, mismatched, missing, nil
}

// scanLocalWithHints scans local Maildir for messages with U= hints.
// Sends messages to the provided channel as they are discovered.
func scanLocalWithHints(ctx context.Context, md *maildir.Maildir, out chan<- LocalMessage) error {
	messages, err := md.List()
	if err != nil {
		return err
	}

	for filename, msg := range messages {
		select {
		case <-ctx.Done():
			return ctx.Err()
		default:
		}

		uid := md.ExtractTentativeUID(filename)
		if uid == nil {
			continue // Skip files without U= hint
		}

		select {
		case <-ctx.Done():
			return ctx.Err()
		case out <- LocalMessage{
			AbsPath:      msg.Filename,
			TentativeUID: *uid,
			Flags:        msg.Flags,
		}:
		}
	}

	return nil
}

// mailboxVerifier verifies local messages against IMAP for a single mailbox.
type mailboxVerifier struct {
	client     *imap.Client
	statusRepo *status.Repository
	mailbox    string
	logger     *slog.Logger

	identical  atomic.Int64
	mismatched atomic.Int64
	missing    atomic.Int64
	mu         sync.Mutex // For fatalErr
	fatalErr   error      // FIXME: ugly
}

// compareMessages verifies local messages against IMAP.
func compareMessages(
	ctx context.Context,
	client *imap.Client,
	statusRepo *status.Repository,
	mailbox string,
	localMessages <-chan LocalMessage,
	logger *slog.Logger,
) (identical, mismatched, missing int, err error) {
	verifier := &mailboxVerifier{
		client:     client,
		statusRepo: statusRepo,
		mailbox:    mailbox,
		logger:     logger,
	}

	sem := semaphore.NewWeighted(CONCURRENCY)
	var wg sync.WaitGroup

	wg.Add(1)
	go func() {
		defer wg.Done()
		for msg := range localMessages {
			if err := sem.Acquire(ctx, 1); err != nil {
				return
			}

			wg.Add(1)
			go func(localMsg LocalMessage) {
				defer wg.Done()
				defer sem.Release(1)

				verifier.compareAndAdd(ctx, localMsg)
			}(msg)
		}
	}()

	wg.Wait()

	return int(verifier.identical.Load()),
		int(verifier.mismatched.Load()),
		int(verifier.missing.Load()),
		verifier.fatalErr
}

// compareAndAdd verifies a single local message and adds to status if identical.
func (v *mailboxVerifier) compareAndAdd(ctx context.Context, localMsg LocalMessage) {
	remoteMsg, err := v.client.FetchMessage(ctx, localMsg.TentativeUID)
	if err != nil {
		// HACK: comparing error by string.
		if err.Error() == fmt.Sprintf("message not found: %d", localMsg.TentativeUID) {
			v.logger.Debug("Message missing on server", "uid", localMsg.TentativeUID)
			v.missing.Add(1)
			return
		}
		v.logger.Warn("Fetch error", "uid", localMsg.TentativeUID, "error", err)
		return
	}

	localBody, err := os.ReadFile(localMsg.AbsPath)
	if err != nil {
		v.logger.Warn(
			"Failed to read local file",
			"uid", localMsg.TentativeUID,
			"path", localMsg.AbsPath,
			"error", err,
		)
		return
	}

	if !bytes.Equal(localBody, remoteMsg.Body) {
		v.logger.Debug("Body mismatch", "uid", localMsg.TentativeUID)
		v.mismatched.Add(1)
		return
	}

	// Compare flags. We should maybe allow mismatching flags, but which copy do we keep?
	if !flagsEqual(localMsg.Flags, remoteMsg.Flags) {
		v.logger.Debug("Flags mismatch",
			"uid", localMsg.TentativeUID,
			"local", localMsg.Flags,
			"remote", remoteMsg.Flags,
		)
		v.mismatched.Add(1)
		return
	}

	filename := filepath.Base(localMsg.AbsPath)
	if err := v.statusRepo.Add(v.mailbox, localMsg.TentativeUID, filename, localMsg.Flags); err != nil {
		v.mu.Lock()
		if v.fatalErr == nil {
			v.fatalErr = fmt.Errorf("failed to add to status: %w", err)
		}
		v.mu.Unlock()
		return
	}
	v.logger.Debug("Message is identical; indexing as synchronised", "uid", localMsg.TentativeUID)

	v.identical.Add(1)
}

// flagsEqual compares two flag slices for equality.
func flagsEqual(a, b []imap2.Flag) bool {
	if len(a) != len(b) {
		return false
	}

	aMap := make(map[imap2.Flag]bool)
	for _, flag := range a {
		aMap[flag] = true
	}

	for _, flag := range b {
		if !aMap[flag] {
			return false
		}
	}

	return true
}
