package goldmark_test

import (
	"bytes"
	"os"
	"strconv"
	"strings"
	"testing"
	"time"

	. "github.com/yuin/goldmark"
	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/parser"
	"github.com/yuin/goldmark/renderer/html"
	"github.com/yuin/goldmark/testutil"
	"github.com/yuin/goldmark/text"
)

var testTimeoutMultiplier = 1.0

func init() {
	m, err := strconv.ParseFloat(os.Getenv("GOLDMARK_TEST_TIMEOUT_MULTIPLIER"), 64)
	if err == nil {
		testTimeoutMultiplier = m
	}
}

func TestExtras(t *testing.T) {
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))
	testutil.DoTestCaseFile(markdown, "_test/extra.txt", t, testutil.ParseCliCaseArg()...)
}

func TestEndsWithNonSpaceCharacters(t *testing.T) {
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))
	source := []byte("```\na\n```")
	var b bytes.Buffer
	err := markdown.Convert(source, &b)
	if err != nil {
		t.Error(err.Error())
	}
	if b.String() != "<pre><code>a\n</code></pre>\n" {
		t.Errorf("%s \n---------\n %s", source, b.String())
	}
}

func TestWindowsNewLine(t *testing.T) {
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
	))
	source := []byte("a  \r\nb\n")
	var b bytes.Buffer
	err := markdown.Convert(source, &b)
	if err != nil {
		t.Error(err.Error())
	}
	if b.String() != "<p>a<br />\nb</p>\n" {
		t.Errorf("%s\n---------\n%s", source, b.String())
	}

	source = []byte("a\\\r\nb\r\n")
	var b2 bytes.Buffer
	err = markdown.Convert(source, &b2)
	if err != nil {
		t.Error(err.Error())
	}
	if b2.String() != "<p>a<br />\nb</p>\n" {
		t.Errorf("\n%s\n---------\n%s", source, b2.String())
	}
}

type myIDs struct {
}

func (s *myIDs) Generate(value []byte, kind ast.NodeKind) []byte {
	return []byte("my-id")
}

func (s *myIDs) Put(value []byte) {
}

func TestAutogeneratedIDs(t *testing.T) {
	ctx := parser.NewContext(parser.WithIDs(&myIDs{}))
	markdown := New(WithParserOptions(parser.WithAutoHeadingID()))
	source := []byte("# Title1\n## Title2")
	var b bytes.Buffer
	err := markdown.Convert(source, &b, parser.WithContext(ctx))
	if err != nil {
		t.Error(err.Error())
	}
	if b.String() != `<h1 id="my-id">Title1</h1>
<h2 id="my-id">Title2</h2>
` {
		t.Errorf("%s\n---------\n%s", source, b.String())
	}
}

func nowMillis() int64 {
	// TODO: replace UnixNano to UnixMillis(drops Go1.16 support)
	return time.Now().UnixNano() / 1000000
}

func TestDeepNestedLabelPerformance(t *testing.T) {
	if testing.Short() {
		t.Skip("skipping performance test in short mode")
	}
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))

	started := nowMillis()
	n := 50000
	source := []byte(strings.Repeat("[", n) + strings.Repeat("]", n))
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	finished := nowMillis()
	if (finished - started) > int64(5000*testTimeoutMultiplier) {
		t.Error("Parsing deep nested labels took too long")
	}
}

func TestManyProcessingInstructionPerformance(t *testing.T) {
	if testing.Short() {
		t.Skip("skipping performance test in short mode")
	}
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))

	started := nowMillis()
	n := 50000
	source := []byte("a " + strings.Repeat("<?", n))
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	finished := nowMillis()
	if (finished - started) > int64(5000*testTimeoutMultiplier) {
		t.Error("Parsing processing instructions took too long")
	}
}

func TestManyCDATAPerformance(t *testing.T) {
	if testing.Short() {
		t.Skip("skipping performance test in short mode")
	}
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))

	started := nowMillis()
	n := 50000
	source := []byte(strings.Repeat("a <![CDATA[", n))
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	finished := nowMillis()
	if (finished - started) > int64(5000*testTimeoutMultiplier) {
		t.Error("Parsing processing instructions took too long")
	}
}

func TestManyDeclPerformance(t *testing.T) {
	if testing.Short() {
		t.Skip("skipping performance test in short mode")
	}
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))

	started := nowMillis()
	n := 50000
	source := []byte(strings.Repeat("a <!A ", n))
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	finished := nowMillis()
	if (finished - started) > int64(5000*testTimeoutMultiplier) {
		t.Error("Parsing processing instructions took too long")
	}
}

func TestManyCommentPerformance(t *testing.T) {
	if testing.Short() {
		t.Skip("skipping performance test in short mode")
	}
	markdown := New(WithRendererOptions(
		html.WithXHTML(),
		html.WithUnsafe(),
	))

	started := nowMillis()
	n := 50000
	source := []byte(strings.Repeat("a <!-- ", n))
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	finished := nowMillis()
	if (finished - started) > int64(5000*testTimeoutMultiplier) {
		t.Error("Parsing processing instructions took too long")
	}
}

func TestDangerousURLStringCase(t *testing.T) {
	markdown := New()

	source := []byte(`[Basic](javascript:alert('Basic'))
[CaseInsensitive](JaVaScRiPt:alert('CaseInsensitive'))
`)
	expected := []byte(`<p><a href="">Basic</a>
<a href="">CaseInsensitive</a></p>
`)
	var b bytes.Buffer
	_ = markdown.Convert(source, &b)
	if !bytes.Equal(expected, b.Bytes()) {
		t.Error("Dangerous URL should ignore cases:\n" + string(testutil.DiffPretty(expected, b.Bytes())))
	}
}

func TestNestedATXHeadingAttributes(t *testing.T) {
	markdown := New(WithParserOptions(
		parser.WithAutoHeadingID(),
		parser.WithAttribute(),
	))

	source := []byte(`# Heading {test=[a, simple, "attribute", { with="nested values" }]}`)
	c := parser.NewContext()
	n := markdown.Parser().Parse(text.NewReader(source), parser.WithContext(c))
	heading := n.FirstChild()
	if heading.Kind() != ast.KindHeading {
		t.Fatalf("expected first node to be heading, got %s", heading.Kind().String())
	}
	tv, ok := heading.Attribute([]byte("test"))
	if !ok {
		t.Fatal("expected to find attribute 'test'")
	}
	wv := tv.([]any)[3].(parser.Attributes)
	_, ok = wv.Find([]byte("with"))
	if !ok {
		t.Fatal("expected to find nested attribute 'with'")
	}

}
