//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// 電源周り
//

#pragma once

#include "device.h"
#include "message.h"

class Syncer;
class UIMessage;

// 定数
enum class PowerButtonState
{
	// 電源ボタンの状態がオフ/オン (bool にキャストできる)
	Off = 0,
	On  = 1,

	// モーメンタリスイッチなどで状態を持たない場合
	NoState = -1,
};

// 電源周り (共通部分)
class PowerDevice : public Device
{
	using inherited = Device;
 public:
	PowerDevice();
	~PowerDevice() override;

	bool Init() override;

	// 電源ボタンを押す (どのスレッドから呼んでもよい)
	void PushPowerButton();

	// リセットを指示 (どのスレッドから呼んでもよい)
	void MakeResetHard();

	// リスタートを指示 (どのスレッドから呼んでもよい)
	void MakeRestart();

	// 機種固有の電源オンオフ処理の信号を受け取る。
	virtual void SetSystemPowerOn(bool poweron);

	// 電源ボタンの状態 (オフ/オン/状態を持たない) を返す。(GUI 用)
	PowerButtonState GetPowerButtonState() const;

	// 電源が実際にオンなら true を返す。
	bool IsPower() const { return ispower; }

	// 電源 LED 点灯なら true を返す。
	bool GetPowerLED() const { return led; }

 protected:
	// メッセージコールバック
	void PowerButtonMessage(MessageID, uint32);
	void ResetMessage(MessageID, uint32);
	void PowerOffMessage(MessageID, uint32);

	// 電源ボタン操作
	virtual void DoPowerButton();

	// 電源オン、リセット、電源オフ操作
	void DoPowerOn();
	void DoResetHard();
	void DoPowerOff(MessageID);

	// 電源ボタンがオルタネートスイッチなら true。
	bool alternate_switch {};

	// 電源ボタンのオン/オフ状態。alternate_switch が false なら無効。
	bool power_button {};

	// 電源オンなら true
	bool ispower {};

	// 電源 LED
	bool led {};

	Syncer *syncer {};
	UIMessage *uimessage {};
};

class LunaPowerDevice : public PowerDevice
{
	using inherited = PowerDevice;
 public:
	LunaPowerDevice();
	~LunaPowerDevice() override;

	bool Init() override;

	// PIO1 の PC4 の状態を受け取る
	void SetSystemPowerOn(bool poweron) override;

 private:
	void PowerCallback(Event *);

	// 電源オフイベント
	Event *event {};
};

class X68030PowerDevice : public PowerDevice
{
	using inherited = PowerDevice;
 public:
	X68030PowerDevice();
	~X68030PowerDevice() override;

	// システムポートからの電源オフコマンドを受け取る
	void SetSystemPowerOn(bool poweron) override;

	// RP5C15 の ALARM_OUT を受け取る
	void SetAlarmOut(bool alarm_out_);

 private:
	// 電源ボタン操作
	void DoPowerButton() override;

	void Change();

	// システム内の電源オン信号に相当
	bool system_poweron {};

	// ALARM_OUT 信号状態
	bool alarm_out {};
};

static inline PowerDevice *GetPowerDevice() {
	return Object::GetObject<PowerDevice>(OBJ_POWER);
}
