/*	$NetBSD: fpu_emulate.c,v 1.49 2025/01/06 07:34:24 isaki Exp $	*/

/*
 * Copyright (c) 1995 Gordon W. Ross
 * some portion Copyright (c) 1995 Ken Nakata
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 4. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Gordon Ross
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * mc68881 emulator
 * XXX - Just a start at it for now...
 */

#include "fpu_emulate.h"

#define	fpe_abort(tfp, ksi, signo, code)			\
	do {							\
		(ksi)->ksi_signo = (signo);			\
		(ksi)->ksi_code = (code);			\
		(ksi)->ksi_addr = (void *)(frame)->f_pc;	\
		return -1;					\
	} while (/* CONSTCOND */ 0)

#if !defined(XM6i_FPE)
static int fpu_emul_fmovmcr(struct fpemu *, struct instruction *);
static int fpu_emul_fmovm(struct fpemu *, struct instruction *);
static int fpu_emul_arith(struct fpemu *, struct instruction *);
static int fpu_emul_type1(struct fpemu *, struct instruction *);
static int fpu_emul_brcc(struct fpemu *, struct instruction *);
static int test_cc(struct fpemu *, int);
#endif

#ifdef DEBUG_FPE
#include <stdio.h>
#define DUMP_INSN(insn)							\
	printf("%s: insn={adv=%d,siz=%d,op=%04x,w1=%04x}\n",		\
	    __func__,							\
	    (insn)->is_advance, (insn)->is_datasize,			\
	    (insn)->is_opcode, (insn)->is_word1)
#define DPRINTF(x)	printf x
#else
#define DUMP_INSN(insn)	do {} while (/* CONSTCOND */ 0)
#define DPRINTF(x)	do {} while (/* CONSTCOND */ 0)
#endif

#if !defined(XM6i_FPE)
/*
 * Emulate a floating-point instruction.
 * Return zero for success, else signal number.
 * (Typically: zero, SIGFPE, SIGILL, SIGSEGV)
 */
int
fpu_emulate(struct frame *frame, struct fpframe *fpf, ksiginfo_t *ksi)
{
	static struct instruction insn;
	static struct fpemu fe;
	int optype, sig;
	unsigned short sval;

	/* initialize insn.is_datasize to tell it is *not* initialized */
	insn.is_datasize = -1;

	fe.fe_frame = frame;
	fe.fe_fpframe = fpf;
	fe.fe_fpsr = fpf->fpf_fpsr;
	fe.fe_fpcr = fpf->fpf_fpcr;

	DPRINTF(("%s: ENTERING: FPSR=%08x, FPCR=%08x\n",
	    __func__, fe.fe_fpsr, fe.fe_fpcr));

	/* always set this (to avoid a warning) */
	insn.is_pc = frame->f_pc;
	insn.is_nextpc = 0;
	if (frame->f_format == 4) {
		/*
		 * A format 4 is generated by the 68{EC,LC}040.  The PC is
		 * already set to the instruction following the faulting
		 * instruction.  We need to calculate that, anyway.  The
		 * fslw is the PC of the faulted instruction, which is what
		 * we expect to be in f_pc.
		 *
		 * XXX - This is a hack; it assumes we at least know the
		 * sizes of all instructions we run across.
		 * XXX TODO: This may not be true, so we might want to save
		 * the PC in order to restore it later.
		 */
#if 0
		insn.is_nextpc = frame->f_pc;
#endif
		insn.is_pc = frame->f_fmt4.f_fslw;
		frame->f_pc = insn.is_pc;
	}

	if (ufetch_short((void *)(insn.is_pc), &sval)) {
		DPRINTF(("%s: fault reading opcode\n", __func__));
		fpe_abort(frame, ksi, SIGSEGV, SEGV_ACCERR);
	}

	if ((sval & 0xf000) != 0xf000) {
		DPRINTF(("%s: not coproc. insn.: opcode=0x%x\n",
		    __func__, sval));
		fpe_abort(frame, ksi, SIGILL, ILL_ILLOPC);
	}

	if ((sval & 0x0E00) != 0x0200) {
		DPRINTF(("%s: bad coproc. id: opcode=0x%x\n", __func__, sval));
		fpe_abort(frame, ksi, SIGILL, ILL_ILLOPC);
	}

	insn.is_opcode = sval;
	optype = (sval & 0x01C0);

	if (ufetch_short((void *)(insn.is_pc + 2), &sval)) {
		DPRINTF(("%s: fault reading word1\n", __func__));
		fpe_abort(frame, ksi, SIGSEGV, SEGV_ACCERR);
	}
	insn.is_word1 = sval;
	/* all FPU instructions are at least 4-byte long */
	insn.is_advance = 4;

	DUMP_INSN(&insn);

	/*
	 * Which family (or type) of opcode is it?
	 * Tests ordered by likelihood (hopefully).
	 * Certainly, type 0 is the most common.
	 */
	if (optype == 0x0000) {
		/* type=0: generic */
		if ((sval & 0x8000)) {
			if ((sval & 0x4000)) {
				DPRINTF(("%s: fmovm FPr\n", __func__));
				sig = fpu_emul_fmovm(&fe, &insn);
			} else {
				DPRINTF(("%s: fmovm FPcr\n", __func__));
				sig = fpu_emul_fmovmcr(&fe, &insn);
			}
		} else {
			if ((sval & 0xe000) == 0x6000) {
				/* fstore = fmove FPn,mem */
				DPRINTF(("%s: fmove to mem\n", __func__));
				sig = fpu_emul_fstore(&fe, &insn);
			} else if ((sval & 0xfc00) == 0x5c00) {
				/* fmovecr */
				DPRINTF(("%s: fmovecr\n", __func__));
				sig = fpu_emul_fmovecr(&fe, &insn);
			} else if ((sval & 0xa07f) == 0x26) {
				/* fscale */
				DPRINTF(("%s: fscale\n", __func__));
				sig = fpu_emul_fscale(&fe, &insn);
			} else {
				DPRINTF(("%s: other type0\n", __func__));
				/* all other type0 insns are arithmetic */
				sig = fpu_emul_arith(&fe, &insn);
			}
			if (sig == 0) {
				DPRINTF(("%s: type 0 returned 0\n", __func__));
				sig = fpu_upd_excp(&fe);
			}
		}
	} else if (optype == 0x0080 || optype == 0x00C0) {
		/* type=2 or 3: fbcc, short or long disp. */
		DPRINTF(("%s: fbcc %s\n", __func__,
		    (optype & 0x40) ? "long" : "short"));
		sig = fpu_emul_brcc(&fe, &insn);
	} else if (optype == 0x0040) {
		/* type=1: fdbcc, fscc, ftrapcc */
		DPRINTF(("%s: type1\n", __func__));
		sig = fpu_emul_type1(&fe, &insn);
		/* real FTRAPcc raises T_TRAPVINST if the condition is met. */
		if (sig == SIGFPE) {
			ksi->ksi_trap = T_TRAPVINST;
		}
	} else {
		/* type=4: fsave    (privileged) */
		/* type=5: frestore (privileged) */
		/* type=6: reserved */
		/* type=7: reserved */
		DPRINTF(("%s: bad opcode type: opcode=0x%x\n", __func__,
		    insn.is_opcode));
		sig = SIGILL;
	}

	DUMP_INSN(&insn);

	/*
	 * XXX it is not clear to me, if we should progress the PC always,
	 * for SIGFPE || 0, or only for 0; however, without SIGFPE, we
	 * don't pass the signalling regression  tests.	-is
	 */
	if ((sig == 0) || (sig == SIGFPE))
		frame->f_pc += insn.is_advance;
#if defined(DDB) && defined(DEBUG_FPE)
	else {
		printf("%s: sig=%d, opcode=%x, word1=%x\n", __func__,
		    sig, insn.is_opcode, insn.is_word1);
		kdb_trap(-1, (db_regs_t *)&frame);
	}
#endif
#if 0 /* XXX something is wrong */
	if (frame->f_format == 4) {
		/* XXX Restore PC -- 68{EC,LC}040 only */
		if (insn.is_nextpc)
			frame->f_pc = insn.is_nextpc;
	}
#endif

	DPRINTF(("%s: EXITING: w/FPSR=%08x, FPCR=%08x\n", __func__,
	    fe.fe_fpsr, fe.fe_fpcr));

	if (sig)
		fpe_abort(frame, ksi, sig, 0);
	return sig;
}
#endif /* !XM6i_FPE */

/* update accrued exception bits and see if there's an FP exception */
// 内部用 FPSR:EXCP から FPSR:AEX を更新する。
// AEX は積算型(?) なので、常に現在の内部用 FPSR:AEX に OR する。
// fe->fe_fpsr (内部用) と fe->fe_fpframe->fpf_fpsr (外部用) 両方の
// AEX を更新する。FPSR のうち AEX 以外のバイトには影響を与えない。
int
fpu_upd_excp(struct fpemu *fe)
{
	uint32_t fpsr;
	uint32_t fpcr;

	fpsr = fe->fe_fpsr;
	fpcr = fe->fe_fpcr;
	/*
	 * update fpsr accrued exception bits; each insn doesn't have to
	 * update this
	 */
	if (fpsr & (FPSR_BSUN | FPSR_SNAN | FPSR_OPERR)) {
		fpsr |= FPSR_AIOP;
	}
	if (fpsr & FPSR_OVFL) {
		fpsr |= FPSR_AOVFL;
	}
	if ((fpsr & FPSR_UNFL) && (fpsr & FPSR_INEX2)) {
		fpsr |= FPSR_AUNFL;
	}
	if (fpsr & FPSR_DZ) {
		fpsr |= FPSR_ADZ;
	}
	if (fpsr & (FPSR_INEX1 | FPSR_INEX2 | FPSR_OVFL)) {
		fpsr |= FPSR_AINEX;
	}

	/* copy AEX byte only */
	fe->fe_fpsr &= ~FPSR_AEX;
	fe->fe_fpsr |= (fpsr & FPSR_AEX);
	fe->fe_fpframe->fpf_fpsr &= ~FPSR_AEX;
	fe->fe_fpframe->fpf_fpsr |= (fpsr & FPSR_AEX);

	return (fpsr & fpcr & FPSR_EXCP) ? 1/*SIGFPE*/ : 0;
}

/* update fpsr according to fp (= result of an fp op) */
// fp によって内部 FPSR:CCB を更新し、(ここまでに反映されている分も
// 含めた) 内部 FPSR のうち CCB, EXCP, AXE バイトを外部 FPSR にもコピーする。
// 外部 FPSR:QTT は更新しない。
//
// o そのため必ず最後のほうで呼ぶこと。fpu_implode() が FPSR:INEX2 を
//   立てるため、その後で呼ばなければいけない。
// o FMOD/FREM は QTT バイトを更新するが、それ以外の命令は QTT を更新しては
//   いけないため、fpu_upd_fpsr() は QTT を外部にコピーしない。
//   QTT を更新する必要のある FMOD/FREM だけが自力で外部 FPSR を更新すること。
uint32_t
fpu_upd_fpsr(struct fpemu *fe, struct fpn *fp)
{
	uint32_t fpsr;

	DPRINTF(("%s: previous fpsr=%08x\n", __func__, fe->fe_fpsr));
	/* clear all condition code */
	fpsr = fe->fe_fpsr & ~FPSR_CCB;

	DPRINTF(("%s: result is a ", __func__));
	if (fp->fp_sign) {
		DPRINTF(("negative "));
		fpsr |= FPSR_NEG;
	} else {
		DPRINTF(("positive "));
	}

	switch (fp->fp_class) {
	case FPC_SNAN:
		DPRINTF(("signaling NAN\n"));
		fpsr |= (FPSR_NAN | FPSR_SNAN);
		break;
	case FPC_QNAN:
		DPRINTF(("quiet NAN\n"));
		fpsr |= FPSR_NAN;
		break;
	case FPC_ZERO:
		DPRINTF(("Zero\n"));
		fpsr |= FPSR_ZERO;
		break;
	case FPC_INF:
		DPRINTF(("Inf\n"));
		fpsr |= FPSR_INF;
		break;
	default:
		DPRINTF(("Number\n"));
		/* anything else is treated as if it is a number */
		break;
	}

	/* copy except QTT byte */
	fe->fe_fpsr = fpsr;
	fe->fe_fpframe->fpf_fpsr &= FPSR_QTT;
	fe->fe_fpframe->fpf_fpsr |= (fpsr & ~FPSR_QTT);

	DPRINTF(("%s: new fpsr=%08x\n", __func__, fe->fe_fpframe->fpf_fpsr));

	return fpsr;
}

#if !defined(XM6i_FPE)
static int
fpu_emul_fmovmcr(struct fpemu *fe, struct instruction *insn)
{
	struct frame *frame = fe->fe_frame;
	struct fpframe *fpf = fe->fe_fpframe;
	int sig;
	int reglist;
	int regcount;
	int fpu_to_mem;
	int modreg;
	uint32_t tmp[3];

	/* move to/from control registers */
	reglist = (insn->is_word1 & 0x1c00) >> 10;
	/* Bit 13 selects direction (FPU to/from Mem) */
	fpu_to_mem = insn->is_word1 & 0x2000;

	/* Check an illegal mod/reg. */
	modreg = insn->is_opcode & 077;
	if (fpu_to_mem) {
		/* PCrel, #imm are illegal. */
		if (modreg >= 072) {
			return SIGILL;
		}
	} else {
		/* All mod/reg can be specified. */
		if (modreg >= 075) {
			return SIGILL;
		}
	}

	/*
	 * If reglist is 0b000, treat it as FPIAR.  This is not specification
	 * but the behavior described in the 6888x user's manual.
	 */
	if (reglist == 0)
		reglist = 1;

	if (reglist == 7) {
		regcount = 3;
	} else if (reglist == 3 || reglist == 5 || reglist == 6) {
		regcount = 2;
	} else {
		regcount = 1;
	}
	insn->is_datasize = regcount * 4;
	sig = fpu_decode_ea(frame, insn, &insn->is_ea, modreg);
	if (sig)
		return sig;

	/*
	 * For data register, only single register can be transferred.
	 * For addr register, only FPIAR can be transferred.
	 */
	if ((insn->is_ea.ea_flags & EA_DIRECT)) {
		if (insn->is_ea.ea_regnum < 8) {
			if (regcount != 1) {
				return SIGILL;
			}
		} else {
			if (reglist != 1) {
				return SIGILL;
			}
		}
	}

	if (fpu_to_mem) {
		uint32_t *s = &tmp[0];

		if ((reglist & 4)) {
			*s++ = fpf->fpf_fpcr;
		}
		if ((reglist & 2)) {
			*s++ = fpf->fpf_fpsr;
		}
		if ((reglist & 1)) {
			*s++ = fpf->fpf_fpiar;
		}

		sig = fpu_store_ea(frame, insn, &insn->is_ea, (char *)tmp);
	} else {
		const uint32_t *d = &tmp[0];

		sig = fpu_load_ea(frame, insn, &insn->is_ea, (char *)tmp);
		if (sig)
			return sig;

		if ((reglist & 4)) {
			fpf->fpf_fpcr = *d++;
			fpf->fpf_fpcr &= 0x0000fff0;
		}
		if ((reglist & 2)) {
			fpf->fpf_fpsr = *d++;
			fpf->fpf_fpsr &= 0x0ffffff8;
		}
		if ((reglist & 1)) {
			fpf->fpf_fpiar = *d++;
		}
	}
	return sig;
}

/*
 * type 0: fmovem
 * Separated out of fpu_emul_type0 for efficiency.
 * In this function, we know:
 *   (opcode & 0x01C0) == 0
 *   (word1 & 0x8000) == 0x8000
 *
 * No conversion or rounding is done by this instruction,
 * and the FPSR is not affected.
 */
static int
fpu_emul_fmovm(struct fpemu *fe, struct instruction *insn)
{
	struct frame *frame = fe->fe_frame;
	struct fpframe *fpf = fe->fe_fpframe;
	int word1, sig;
	int reglist, regmask, regnum;
	int modreg;
	int fpu_to_mem, order;
	/* int w1_post_incr; */
	int *fpregs;

	insn->is_datasize = 12;
	word1 = insn->is_word1;

	/* Bit 13 selects direction (FPU to/from Mem) */
	fpu_to_mem = word1 & 0x2000;

	/*
	 * Bits 12,11 select register list mode:
	 * 0,0: Static  reg list, pre-decr.
	 * 0,1: Dynamic reg list, pre-decr.
	 * 1,0: Static  reg list, post-incr.
	 * 1,1: Dynamic reg list, post-incr
	 */
	/* w1_post_incr = word1 & 0x1000; */
	if (word1 & 0x0800) {
		/* dynamic reg list */
		reglist = frame->f_regs[(word1 & 0x70) >> 4];
	} else {
		reglist = word1;
	}
	reglist &= 0xFF;

	/* Check an illegal mod/reg. */
	modreg = insn->is_opcode & 077;
	if (fpu_to_mem) {
		/* Dn, An, (An)+, PCrel, #imm are illegal. */
		if (modreg < 020 || (modreg >> 3) == 3 || modreg >= 072) {
			return SIGILL;
		}
	} else {
		/* Dn, An, -(An), #imm are illegal. */
		if (modreg < 020 || (modreg >> 3) == 4 || modreg >= 074) {
			return SIGILL;
		}
	}

	/* Get effective address. */
	sig = fpu_decode_ea(frame, insn, &insn->is_ea, modreg);
	if (sig)
		return sig;

	/* Get address of soft coprocessor regs. */
	fpregs = &fpf->fpf_regs[0];

	if (insn->is_ea.ea_flags & EA_PREDECR) {
		regnum = 7;
		order = -1;
	} else {
		regnum = 0;
		order = 1;
	}

	regmask = 0x80;
	while ((0 <= regnum) && (regnum < 8)) {
		if (regmask & reglist) {
			if (fpu_to_mem) {
				sig = fpu_store_ea(frame, insn, &insn->is_ea,
				    (char *)&fpregs[regnum * 3]);
				DPRINTF(("%s: FP%d (%08x,%08x,%08x) saved\n",
				    __func__, regnum,
				    fpregs[regnum * 3],
				    fpregs[regnum * 3 + 1],
				    fpregs[regnum * 3 + 2]));
			} else {		/* mem to fpu */
				sig = fpu_load_ea(frame, insn, &insn->is_ea,
				    (char *)&fpregs[regnum * 3]);
				DPRINTF(("%s: FP%d (%08x,%08x,%08x) loaded\n",
				    __func__, regnum,
				    fpregs[regnum * 3],
				    fpregs[regnum * 3 + 1],
				    fpregs[regnum * 3 + 2]));
			}
			if (sig)
				break;
		}
		regnum += order;
		regmask >>= 1;
	}

	return sig;
}
#endif /* !XM6i_FPE */

#if defined(XM6i_FPE)
struct fpn *
fpu_sglmul(struct fpemu *fe)
{
	struct fpn *r;

	if (ISNAN(&fe->fe_f1))
		return &fe->fe_f1;
	if (ISNAN(&fe->fe_f2))
		return &fe->fe_f2;

	/*
	 * 仮数部を(小数以下) 23bit だけにして「精度」を単精度と同じにする。
	 * FSGLMUL では指数部が単精度の範囲を越えても inf 等にならないし、
	 * 68000PRM.pdf で (精度を落とすにあたり) FPCR の現在の丸めモードに
	 * 関係ないと言っているあたりからも、単精度型に変換するとかではなく、
	 * 仮数部のビットを落とすだけなんだと思う。
	 */
	fe->fe_f1.fp_mant[1] &= 0xf8000000;
	fe->fe_f1.fp_mant[2]  = 0;

	fe->fe_f2.fp_mant[1] &= 0xf8000000;
	fe->fe_f2.fp_mant[2]  = 0;

	r = fpu_mul(fe);

	fpu_round_prec(fe, r);
	DUMPFP("sglmul:round", r);

	// 非正規化数の最小値(最下位ビットだけ1) だったら、
	// 指数部はそのまま、仮数部を単精度の最下位ビット 1 という状態にする。
	if (r->fp_exp == -EXT_EXP_BIAS - EXT_FRACBITS + 1 &&
	    r->fp_mant[0] == FP_1 &&
	    r->fp_mant[1] == 0 &&
	    r->fp_mant[2] == 0)
	{
		r->fp_exp += EXT_FRACBITS - SNG_FRACBITS - 1;
	}
	DUMPFP("sglmul:mod  ", r);

	return r;
}

struct fpn *
fpu_sgldiv(struct fpemu *fe)
{
	struct fpn *r;

	if (ISNAN(&fe->fe_f1))
		return &fe->fe_f1;
	if (ISNAN(&fe->fe_f2))
		return &fe->fe_f2;

	fe->fe_f1.fp_mant[1] &= 0xf8000000;
	fe->fe_f1.fp_mant[2]  = 0;

	fe->fe_f2.fp_mant[1] &= 0xf8000000;
	fe->fe_f2.fp_mant[2]  = 0;

	r = fpu_div(fe);

	fpu_round_prec(fe, r);
	DUMPFP("sgldiv:round", r);

	if (r->fp_exp == -EXT_EXP_BIAS - EXT_FRACBITS + 1 &&
	    r->fp_mant[0] == FP_1 &&
	    r->fp_mant[1] == 0 &&
	    r->fp_mant[2] == 0)
	{
		r->fp_exp += EXT_FRACBITS - SNG_FRACBITS - 1;
	}
	DUMPFP("sgldiv:mod  ", r);

	return r;
}
#endif /* XM6i_FPE */

#if !defined(XM6i_FPE)
/*
 * arithmetic operations
 */
static int
fpu_emul_arith(struct fpemu *fe, struct instruction *insn)
{
	struct frame *frame = fe->fe_frame;
	uint32_t *fpregs = &(fe->fe_fpframe->fpf_regs[0]);
	struct fpn *res;
	int word1, sig = 0;
	int regnum, format;
	int modreg;
	int discard_result = 0;
	uint32_t buf[3];
#ifdef DEBUG_FPE
	int flags;
	char regname;
#endif

	fe->fe_fpsr &= ~FPSR_EXCP;

	DUMP_INSN(insn);

	DPRINTF(("%s: FPSR = %08x, FPCR = %08x\n", __func__,
	    fe->fe_fpsr, fe->fe_fpcr));

	word1 = insn->is_word1;
	format = (word1 >> 10) & 7;
	regnum = (word1 >> 7) & 7;

	/* fetch a source operand : may not be used */
	DPRINTF(("%s: dst/src FP%d=%08x,%08x,%08x\n", __func__,
	    regnum, fpregs[regnum * 3], fpregs[regnum * 3 + 1],
	    fpregs[regnum * 3 + 2]));

	fpu_explode(fe, &fe->fe_f1, FTYPE_EXT, &fpregs[regnum * 3]);

	DUMP_INSN(insn);

	/* get the other operand which is always the source */
	if ((word1 & 0x4000) == 0) {
		DPRINTF(("%s: FP%d op FP%d => FP%d\n", __func__,
		    format, regnum, regnum));
		DPRINTF(("%s: src opr FP%d=%08x,%08x,%08x\n", __func__,
		    format, fpregs[format * 3], fpregs[format * 3 + 1],
		    fpregs[format * 3 + 2]));
		fpu_explode(fe, &fe->fe_f2, FTYPE_EXT, &fpregs[format * 3]);
	} else {
		/* the operand is in memory */
		if (format == FTYPE_DBL) {
			insn->is_datasize = 8;
		} else if (format == FTYPE_SNG || format == FTYPE_LNG) {
			insn->is_datasize = 4;
		} else if (format == FTYPE_WRD) {
			insn->is_datasize = 2;
		} else if (format == FTYPE_BYT) {
			insn->is_datasize = 1;
		} else if (format == FTYPE_EXT) {
			insn->is_datasize = 12;
		} else {
			/* invalid or unsupported operand format */
			sig = SIGFPE;
			return sig;
		}

		/* Check an illegal mod/reg. */
		modreg = insn->is_opcode & 077;
		if ((modreg >> 3) == 1/*An*/ || modreg >= 075) {
			return SIGILL;
		}

		/* Get effective address. */
		sig = fpu_decode_ea(frame, insn, &insn->is_ea, modreg);
		if (sig) {
			DPRINTF(("%s: error in fpu_decode_ea\n", __func__));
			return sig;
		}

		if (insn->is_ea.ea_flags == EA_DIRECT &&
		    insn->is_datasize > 4) {
			DPRINTF(("%s: attempted to fetch dbl/ext from reg\n",
			    __func__));
			return SIGILL;
		}

		DUMP_INSN(insn);

#ifdef DEBUG_FPE
		printf("%s: addr mode = ", __func__);
		flags = insn->is_ea.ea_flags;
		regname = (insn->is_ea.ea_regnum & 8) ? 'a' : 'd';

		if (flags & EA_DIRECT) {
			printf("%c%d\n", regname, insn->is_ea.ea_regnum & 7);
		} else if (flags & EA_PC_REL) {
			if (flags & EA_OFFSET) {
				printf("pc@(%d)\n", insn->is_ea.ea_offset);
			} else if (flags & EA_INDEXED) {
				printf("pc@(...)\n");
			}
		} else if (flags & EA_PREDECR) {
			printf("%c%d@-\n", regname, insn->is_ea.ea_regnum & 7);
		} else if (flags & EA_POSTINCR) {
			printf("%c%d@+\n", regname, insn->is_ea.ea_regnum & 7);
		} else if (flags & EA_OFFSET) {
			printf("%c%d@(%d)\n", regname,
			    insn->is_ea.ea_regnum & 7,
			    insn->is_ea.ea_offset);
		} else if (flags & EA_INDEXED) {
			printf("%c%d@(...)\n", regname,
			    insn->is_ea.ea_regnum & 7);
		} else if (flags & EA_ABS) {
			printf("0x%08x\n", insn->is_ea.ea_absaddr);
		} else if (flags & EA_IMMED) {
			printf("#0x%08x,%08x,%08x\n",
			    insn->is_ea.ea_immed[0],
			    insn->is_ea.ea_immed[1],
			    insn->is_ea.ea_immed[2]);
		} else {
			printf("%c%d@\n", regname, insn->is_ea.ea_regnum & 7);
		}
#endif /* DEBUG_FPE */

		fpu_load_ea(frame, insn, &insn->is_ea, (char*)buf);
		if (format == FTYPE_WRD) {
			/* sign-extend */
			buf[0] &= 0xffff;
			if (buf[0] & 0x8000)
				buf[0] |= 0xffff0000;
			format = FTYPE_LNG;
		} else if (format == FTYPE_BYT) {
			/* sign-extend */
			buf[0] &= 0xff;
			if (buf[0] & 0x80)
				buf[0] |= 0xffffff00;
			format = FTYPE_LNG;
		}
		DPRINTF(("%s: src = %08x %08x %08x, siz = %d\n", __func__,
		    buf[0], buf[1], buf[2], insn->is_datasize));
		fpu_explode(fe, &fe->fe_f2, format, buf);
	}

	DUMP_INSN(insn);

	/*
	 * An arithmetic instruction emulate function has a prototype of
	 * struct fpn *fpu_op(struct fpemu *);
	 *
	 * 1) If the instruction is monadic, then fpu_op() must use
	 *    fe->fe_f2 as its operand, and return a pointer to the
	 *    result.
	 *
	 * 2) If the instruction is diadic, then fpu_op() must use
	 *    fe->fe_f1 and fe->fe_f2 as its two operands, and return a
	 *    pointer to the result.
	 *
	 */
	res = NULL;
	switch (word1 & 0x7f) {
	case 0x00:		/* fmove */
		res = &fe->fe_f2;
		break;

	case 0x01:		/* fint */
		res = fpu_int(fe);
		break;

	case 0x02:		/* fsinh */
		res = fpu_sinh(fe);
		break;

	case 0x03:		/* fintrz */
		res = fpu_intrz(fe);
		break;

	case 0x04:		/* fsqrt */
		res = fpu_sqrt(fe);
		break;

	case 0x06:		/* flognp1 */
		res = fpu_lognp1(fe);
		break;

	case 0x08:		/* fetoxm1 */
		res = fpu_etoxm1(fe);
		break;

	case 0x09:		/* ftanh */
		res = fpu_tanh(fe);
		break;

	case 0x0A:		/* fatan */
		res = fpu_atan(fe);
		break;

	case 0x0C:		/* fasin */
		res = fpu_asin(fe);
		break;

	case 0x0D:		/* fatanh */
		res = fpu_atanh(fe);
		break;

	case 0x0E:		/* fsin */
		res = fpu_sin(fe);
		break;

	case 0x0F:		/* ftan */
		res = fpu_tan(fe);
		break;

	case 0x10:		/* fetox */
		res = fpu_etox(fe);
		break;

	case 0x11:		/* ftwotox */
		res = fpu_twotox(fe);
		break;

	case 0x12:		/* ftentox */
		res = fpu_tentox(fe);
		break;

	case 0x14:		/* flogn */
		res = fpu_logn(fe);
		break;

	case 0x15:		/* flog10 */
		res = fpu_log10(fe);
		break;

	case 0x16:		/* flog2 */
		res = fpu_log2(fe);
		break;

	case 0x18:		/* fabs */
		fe->fe_f2.fp_sign = 0;
		res = &fe->fe_f2;
		break;

	case 0x19:		/* fcosh */
		res = fpu_cosh(fe);
		break;

	case 0x1A:		/* fneg */
		fe->fe_f2.fp_sign = !fe->fe_f2.fp_sign;
		res = &fe->fe_f2;
		break;

	case 0x1C:		/* facos */
		res = fpu_acos(fe);
		break;

	case 0x1D:		/* fcos */
		res = fpu_cos(fe);
		break;

	case 0x1E:		/* fgetexp */
		res = fpu_getexp(fe);
		break;

	case 0x1F:		/* fgetman */
		res = fpu_getman(fe);
		break;

	case 0x20:		/* fdiv */
	case 0x24:		/* fsgldiv: cheating - better than nothing */
		res = fpu_div(fe);
		break;

	case 0x21:		/* fmod */
		res = fpu_mod(fe);
		break;

	case 0x28:		/* fsub */
		fe->fe_f2.fp_sign = !fe->fe_f2.fp_sign; /* f2 = -f2 */
		/* FALLTHROUGH */
	case 0x22:		/* fadd */
		res = fpu_add(fe);
		break;

	case 0x23:		/* fmul */
	case 0x27:		/* fsglmul: cheating - better than nothing */
		res = fpu_mul(fe);
		break;

	case 0x25:		/* frem */
		res = fpu_rem(fe);
		break;

	case 0x26:
		/* fscale is handled by a separate function */
		break;

	case 0x30:
	case 0x31:
	case 0x32:
	case 0x33:
	case 0x34:
	case 0x35:
	case 0x36:
	case 0x37:		/* fsincos */
		res = fpu_sincos(fe, word1 & 7);
		break;

	case 0x38:		/* fcmp */
		res = fpu_cmp(fe);
		discard_result = 1;
		break;

	case 0x3A:		/* ftst */
		res = &fe->fe_f2;
		discard_result = 1;
		break;

	default:		/* possibly 040/060 instructions */
		DPRINTF(("%s: bad opcode=0x%x, word1=0x%x\n", __func__,
		    insn->is_opcode, insn->is_word1));
		sig = SIGILL;
	}

	/* for sanity */
	if (res == NULL)
		sig = SIGILL;

	if (sig == 0) {
		if (!discard_result)
			fpu_implode(fe, res, FTYPE_EXT, &fpregs[regnum * 3]);

		/* update fpsr according to the result of operation */
		fpu_upd_fpsr(fe, res);
#ifdef DEBUG_FPE
		if (!discard_result) {
			printf("%s: %08x,%08x,%08x stored in FP%d\n", __func__,
			    fpregs[regnum * 3],
			    fpregs[regnum * 3 + 1],
			    fpregs[regnum * 3 + 2],
			    regnum);
		} else {
			static const char *class_name[] =
			    { "SNAN", "QNAN", "ZERO", "NUM", "INF" };
			printf("%s: result(%s,%c,%d,%08x,%08x,%08x) "
			    "discarded\n", __func__,
			    class_name[res->fp_class + 2],
			    res->fp_sign ? '-' : '+', res->fp_exp,
			    res->fp_mant[0], res->fp_mant[1],
			    res->fp_mant[2]);
		}
#endif
	} else {
		DPRINTF(("%s: received signal %d\n", __func__, sig));
	}

	DPRINTF(("%s: FPSR = %08x, FPCR = %08x\n", __func__,
	    fe->fe_fpsr, fe->fe_fpcr));

	DUMP_INSN(insn);

	return sig;
}
#endif /* !XM6i_FPE */

/*
 * test condition code according to the predicate in the opcode.
 * returns -1 when the predicate evaluates to true, 0 when false.
 * signal numbers are returned when an error is detected.
 * ここではシグナルではなく正数なら未実装命令パターン。
 */
#if defined(XM6i_FPE)
int
#else
static int
#endif
test_cc(struct fpemu *fe, int pred)
{
	int result, sig_bsun;
	int fpsr;

	fpsr = fe->fe_fpsr;
	DPRINTF(("%s: fpsr=0x%08x\n", __func__, fpsr));
	pred &= 0x3f;		/* lowest 6 bits */

	DPRINTF(("%s: ", __func__));

	if (pred >= 0x20) {
		DPRINTF(("Illegal condition code\n"));
		return 1;
	} else if (pred & 0x10) {
		/* IEEE nonaware tests */
		sig_bsun = 1;
		pred &= 0x0f;		/* lower 4 bits */
	} else {
		/* IEEE aware tests */
		DPRINTF(("IEEE "));
		sig_bsun = 0;
	}

	/*
	 *           condition   real 68882
	 * mnemonic  in manual   condition
	 * --------  ----------  ----------
	 * 0000 F    0           <-         = ~NAN &  0 & ~Z | 0
	 * 0001 EQ   Z           <-         = ~NAN &  0 |  Z | 0
	 * 0010 OGT  ~(NAN|Z|N)  <-         = ~NAN & ~N & ~Z | 0
	 * 0011 OGE  Z|~(NAN|N)  <-         = ~NAN & ~N |  Z | 0
	 * 0100 OLT  N&~(NAN|Z)  <-         = ~NAN &  N & ~Z | 0
	 * 0101 OLE  Z|(N&~NAN)  <-         = ~NAN &  N |  Z | 0
	 * 0110 OGL  ~(NAN|Z)    <-         = ~NAN &  1 & ~Z | 0
	 * 0111 OR   ~NAN        Z|~NAN     = ~NAN &  1 |  Z | 0
	 *
	 * 1000 UN   NAN         <-         =  1   &  0 & ~Z | NAN
	 * 1001 UEQ  NAN|Z       <-         =  1   &  0 |  Z | NAN
	 * 1010 UGT  NAN|~(N|Z)  <-         =  1   & ~N & ~Z | NAN
	 * 1011 UGE  NAN|(Z|~N)  <-         =  1   & ~N |  Z | NAN
	 * 1100 ULT  NAN|(N&~Z)  <-         =  1   &  N & ~Z | NAN
	 * 1101 ULE  NAN|(Z|N)   <-         =  1   &  N |  Z | NAN
	 * 1110 NE   ~Z          NAN|(~Z)   =  1   &  1 & ~Z | NAN
	 * 1111 T    1           <-         =  1   &  1 |  Z | NAN
	 */
	if ((pred & 0x08) == 0) {
		result = ((fpsr & FPSR_NAN) == 0);
	} else {
		result = 1;
	}
	switch (pred & 0x06) {
	 case 0x00:	// 'AND 0'
		result &= 0;
		break;
	 case 0x02:	// 'AND ~N'
		result &= ((fpsr & FPSR_NEG) == 0);
		break;
	 case 0x04:	// 'AND N'
		result &= ((fpsr & FPSR_NEG) != 0);
		break;
	 case 0x06:	// 'AND 1'
		result &= 1;
		break;
	}
	if ((pred & 0x01) == 0) {
		result &= ((fpsr & FPSR_ZERO) == 0);
	} else {
		result |= ((fpsr & FPSR_ZERO) != 0);
	}
	if ((pred & 0x08) != 0) {
		result |= ((fpsr & FPSR_NAN) != 0);
	}

	DPRINTF(("=> %s (%d)\n", result ? "true" : "false", result));
	/* if it's an IEEE unaware test and NAN is set, BSUN is set */
	if (sig_bsun && (fpsr & FPSR_NAN)) {
		fpsr |= FPSR_BSUN;
	}
	/* if BSUN is set, IOP is set too */
	if ((fpsr & FPSR_BSUN)) {
		fpsr |= FPSR_AIOP;
	}

	/* put fpsr back */
	fe->fe_fpframe->fpf_fpsr = fe->fe_fpsr = fpsr;

	return -result;
}

#if !defined(XM6i_FPE)
/*
 * type 1: fdbcc, fscc, ftrapcc
 * In this function, we know:
 *   (opcode & 0x01C0) == 0x0040
 * return SIGILL for an illegal instruction.
 * return SIGFPE if FTRAPcc's condition is met.
 */
static int
fpu_emul_type1(struct fpemu *fe, struct instruction *insn)
{
	struct frame *frame = fe->fe_frame;
	int advance, sig, branch, displ;
	unsigned short sval;

	branch = test_cc(fe, insn->is_word1);
	if (branch > 0)
		return branch;
	fe->fe_fpframe->fpf_fpsr = fe->fe_fpsr;

	sig = 0;
	switch (insn->is_opcode & 070) {
	case 010:			/* fdbcc */
		if (branch) {
			/* advance */
			insn->is_advance = 6;
		} else {
			/* decrement Dn and if (Dn != -1) branch */
			uint16_t count = frame->f_regs[insn->is_opcode & 7];

			if (count-- != 0) {
				if (ufetch_short((void *)(insn->is_pc +
							   insn->is_advance),
						  &sval)) {
					DPRINTF(("%s: fault reading "
					    "displacement\n", __func__));
					return SIGSEGV;
				}
				displ = sval;
				/* sign-extend the displacement */
				displ &= 0xffff;
				if (displ & 0x8000) {
					displ |= 0xffff0000;
				}
				insn->is_advance += displ;
#if 0				/* XXX */
				insn->is_nextpc = insn->is_pc +
				    insn->is_advance;
#endif
			} else {
				insn->is_advance = 6;
			}
			/* write it back */
			frame->f_regs[insn->is_opcode & 7] &= 0xffff0000;
			frame->f_regs[insn->is_opcode & 7] |= (uint32_t)count;
		}
		break;

	case 070:			/* ftrapcc or fscc */
		advance = 4;
		if ((insn->is_opcode & 07) >= 2) {
			switch (insn->is_opcode & 07) {
			case 3:		/* long opr */
				advance += 2;
			case 2:		/* word opr */
				advance += 2;
			case 4:		/* no opr */
				break;
			default:
				return SIGILL;
				break;
			}
			insn->is_advance = advance;

			if (branch) {
				/* trap */
				sig = SIGFPE;
			}
			break;
		}

		/* FALLTHROUGH */
	default:			/* fscc */
		insn->is_datasize = 1;	/* always byte */
		sig = fpu_decode_ea(frame, insn, &insn->is_ea, insn->is_opcode);
		if (sig) {
			break;
		}
		/* set result */
		sig = fpu_store_ea(frame, insn, &insn->is_ea, (char *)&branch);
		break;
	}
	return sig;
}

/*
 * Type 2 or 3: fbcc (also fnop)
 * In this function, we know:
 *   (opcode & 0x0180) == 0x0080
 */
static int
fpu_emul_brcc(struct fpemu *fe, struct instruction *insn)
{
	int displ, word2;
	int sig;
	unsigned short sval;

	/*
	 * Get branch displacement.
	 */
	displ = insn->is_word1;

	if (insn->is_opcode & 0x40) {
		if (ufetch_short((void *)(insn->is_pc + insn->is_advance),
				  &sval)) {
			DPRINTF(("%s: fault reading word2\n", __func__));
			return SIGSEGV;
		}
		word2 = sval;
		displ <<= 16;
		displ |= word2;
		insn->is_advance += 2;
	} else {
		/* displacement is word sized */
		if (displ & 0x8000)
			displ |= 0xFFFF0000;
	}

	/* XXX: If CC, insn->is_pc += displ */
	sig = test_cc(fe, insn->is_opcode);
	fe->fe_fpframe->fpf_fpsr = fe->fe_fpsr;

	if (fe->fe_fpsr & fe->fe_fpcr & FPSR_EXCP) {
		return SIGFPE;		/* caught an exception */
	}
	if (sig == -1) {
		/*
		 * branch does take place; 2 is the offset to the 1st disp word
		 */
		insn->is_advance = displ + 2;
#if 0		/* XXX */
		insn->is_nextpc = insn->is_pc + insn->is_advance;
#endif
	} else if (sig)
		return SIGILL;		/* got a signal */
	DPRINTF(("%s: %s insn @ %x (%x+%x) (disp=%x)\n", __func__,
	    (sig == -1) ? "BRANCH to" : "NEXT",
	    insn->is_pc + insn->is_advance, insn->is_pc, insn->is_advance,
	    displ));
	return 0;
}
#endif /* !XM6i_FPE */

#if defined(XM6i_FPE)
// fpn のタグ情報のうち、ゼロ、Inf、NAN、それ以外(ここでは正規化数)を返す。
// 非正規化数とアンノーマル数はここに来る前に弾いてある。
// 68040 の FSAVE で使う。
int
fpu_gettag(const struct fpn *fpn)
{
	if (ISZERO(fpn)) {
		return 1;
	}
	if (ISINF(fpn)) {
		return 2;
	}
	if (ISNAN(fpn)) {
		return 3;
	}
	return 0;
}
#endif
