//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

#include "optestxp_subr.h"

static std::vector<uint8> values1 = {
	0x00,
	0x01,
	0x0f,
	0x10,
	0x7f,
	0x80,
	0xff,
};
static std::vector<uint8> values2 = {
	0x00,
	0x01,
	0x03,
	0x55,
	0x7f,
	0xaa,
	0xff,
};

static void
gentest_add()
{
	init("add", "testrun2");

	out("testexec_%s:\n", testname);
	out("	ADD	A,D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (auto src : values1) {
			uint32 res32 = dst + src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.N = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = ((res32 & 0x0100) != 0);
			flag.PV = vflag_add(dst, src, res);
			flag.H = hflag_add(dst, src);
			flag.N = false;
			out_testdata_run2(dst, inflag, src, res, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_adc()
{
	init("adc", "testrun2");

	out("testexec_%s:\n", testname);
	out("	ADC	A,D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (auto src : values1) {
			for (int cin = 0; cin <= 1; cin++) {
				uint32 res32 = dst + src + cin;
				uint8 res = res32;
				hd64180flag inflag;
				inflag.C = cin;
				inflag.N = true;
				hd64180flag flag;
				flag.Z = (res == 0);
				flag.S = ((res & 0x80) != 0);
				flag.C = ((res32 & 0x0100) != 0);
				flag.PV = vflag_add(dst, src, res);
				flag.H = hflag_add(dst, src) | hflag_add(src + dst, cin);
				flag.N = false;
				out_testdata_run2(dst, inflag, src, res, flag);
			}
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_sub()
{
	init("sub", "testrun2");

	out("testexec_%s:\n", testname);
	out("	SUB	D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (auto src : values1) {
			uint32 res32 = dst - src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.N = false;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = ((res32 & 0x0100) != 0);
			flag.PV = vflag_sub(dst, src, res);
			flag.H = hflag_sub(dst, src);
			flag.N = true;
			out_testdata_run2(dst, inflag, src, res, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_sbc()
{
	init("sbc", "testrun2");

	out("testexec_%s:\n", testname);
	out("	SBC	A,D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (auto src : values1) {
			for (int cin = 0; cin <= 1; cin++) {
				uint32 res32 = dst - src - cin;
				uint8 res = res32;
				hd64180flag inflag;
				inflag.C = cin;
				inflag.N = false;
				hd64180flag flag;
				flag.Z = (res == 0);
				flag.S = ((res & 0x80) != 0);
				flag.C = ((res32 & 0x0100) != 0);
				flag.PV = vflag_sub(dst, src, res);
				flag.H = hflag_sub(dst, src) | hflag_sub(dst - src, cin);
				flag.N = true;
				out_testdata_run2(dst, inflag, src, res, flag);
			}
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_and()
{
	init("and", "testrun2");

	out("testexec_%s:\n", testname);
	out("	AND	D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		for (auto src : values2) {
			uint32 res32 = dst & src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.H = false;
			inflag.N = true;
			inflag.C = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = false;
			flag.PV = pflag(res);
			flag.H = true;
			flag.N = false;
			out_testdata_run2(dst, inflag, src, res, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_or()
{
	init("or", "testrun2");

	out("testexec_%s:\n", testname);
	out("	OR	D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		for (auto src : values2) {
			uint32 res32 = dst | src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.H = true;
			inflag.N = true;
			inflag.C = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = false;
			flag.PV = pflag(res);
			flag.H = false;
			flag.N = false;
			out_testdata_run2(dst, inflag, src, res, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_xor()
{
	init("xor", "testrun2");

	out("testexec_%s:\n", testname);
	out("	XOR	D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		for (auto src : values2) {
			uint32 res32 = dst ^ src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.H = true;
			inflag.N = true;
			inflag.C = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = false;
			flag.PV = pflag(res);
			flag.H = false;
			flag.N = false;
			out_testdata_run2(dst, inflag, src, res, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_cp()
{
	init("cp", "testrun2");

	out("testexec_%s:\n", testname);
	out("	CP	D\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (auto src : values1) {
			uint32 res32 = dst - src;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.N = false;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = ((res32 & 0x0100) != 0);
			flag.PV = vflag_sub(dst, src, res);
			flag.H = hflag_sub(dst, src);
			flag.N = true;
			out_testdata_run2(dst, inflag, src, dst, flag);
		}
	}
	out("	defb	0,0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_inc()
{
	init("inc", "testrun1");

	out("testexec_%s:\n", testname);
	out("	INC	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst + 1;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.C = cin;
			inflag.N = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = cin;
			flag.PV = vflag_add(dst, 1, res);
			flag.H = hflag_add(dst, 1);
			flag.N = false;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_dec()
{
	init("dec", "testrun1");

	out("testexec_%s:\n", testname);
	out("	DEC	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst - 1;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.C = cin;
			inflag.N = false;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = cin;
			flag.PV = vflag_sub(dst, 1, res);
			flag.H = hflag_sub(dst, 1);
			flag.N = true;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_neg()
{
	init("neg", "testrun1");

	out("testexec_%s:\n", testname);
	out("	NEG\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values1) {
		uint32 res32 = 0 - dst;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = false;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((res32 & 0x100) != 0);
		flag.PV = vflag_sub(0, dst, res);
		flag.H = hflag_sub(0, dst);
		flag.N = true;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_cpl()
{
	init("cpl", "testrun1");

	out("testexec_%s:\n", testname);
	out("	CPL\n");
	out("	RET\n");

	out_testset();

	bool fin = false;
	for (auto dst : values2) {
		fin = !fin;
		uint32 res32 = ~dst;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.Z = fin;
		inflag.S = fin;
		inflag.C = fin;
		inflag.PV = fin;
		inflag.N = false;
		inflag.H = false;
		hd64180flag flag;
		flag.Z = fin;
		flag.S = fin;
		flag.C = fin;
		flag.PV = fin;
		flag.H = true;
		flag.N = true;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rla()
{
	init("rla", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RLA\n");
	out("	RET\n");

	out_testset();

	bool fin = false;
	for (auto dst : values2) {
		fin = !fin;
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst << 1;
			res32 |= cin;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.Z = fin;
			inflag.S = fin;
			inflag.C = cin;
			inflag.PV = fin;
			inflag.N = true;
			inflag.H = true;
			hd64180flag flag;
			flag.Z = fin;
			flag.S = fin;
			flag.C = ((res32 & 0x100) != 0);
			flag.PV = fin;
			flag.H = false;
			flag.N = false;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rra()
{
	init("rra", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RRA\n");
	out("	RET\n");

	out_testset();

	bool fin = false;
	for (auto dst : values2) {
		fin = !fin;
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst >> 1;
			res32 |= cin ? 0x80 : 0;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.Z = fin;
			inflag.S = fin;
			inflag.C = cin;
			inflag.PV = fin;
			inflag.N = true;
			inflag.H = true;
			hd64180flag flag;
			flag.Z = fin;
			flag.S = fin;
			flag.C = ((dst & 1) != 0);
			flag.PV = fin;
			flag.H = false;
			flag.N = false;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rlca()
{
	init("rlca", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RLCA\n");
	out("	RET\n");

	out_testset();

	bool fin = false;
	for (auto dst : values2) {
		fin = !fin;
		uint32 res32 = dst << 1;
		res32 |= (res32 >> 8);
		uint8 res = res32;
		hd64180flag inflag;
		inflag.Z = fin;
		inflag.S = fin;
		inflag.PV = fin;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = fin;
		flag.S = fin;
		flag.C = ((res32 & 0x100) != 0);
		flag.PV = fin;
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rrca()
{
	init("rrca", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RRCA\n");
	out("	RET\n");

	out_testset();

	bool fin = false;
	for (auto dst : values2) {
		fin = !fin;
		uint32 res32 = dst >> 1;
		res32 |= dst << 7;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.Z = fin;
		inflag.S = fin;
		inflag.PV = fin;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = fin;
		flag.S = fin;
		flag.C = ((dst & 1) != 0);
		flag.PV = fin;
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rl()
{
	init("rl", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RL	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst << 1;
			res32 |= cin;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.C = cin;
			inflag.N = true;
			inflag.H = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = ((res32 & 0x100) != 0);
			flag.PV = pflag(res);
			flag.H = false;
			flag.N = false;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rr()
{
	init("rr", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RR	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		for (int cin = 0; cin <= 1; cin++) {
			uint32 res32 = dst >> 1;
			res32 |= cin ? 0x80 : 0;
			uint8 res = res32;
			hd64180flag inflag;
			inflag.C = cin;
			inflag.N = true;
			inflag.H = true;
			hd64180flag flag;
			flag.Z = (res == 0);
			flag.S = ((res & 0x80) != 0);
			flag.C = ((dst & 1) != 0);
			flag.PV = pflag(res);
			flag.H = false;
			flag.N = false;
			out_testdata_run1(dst, inflag, res, flag);
		}
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rlc()
{
	init("rlc", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RLC	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		uint32 res32 = dst << 1;
		res32 |= (res32 >> 8);
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((res32 & 0x100) != 0);
		flag.PV = pflag(res);
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_rrc()
{
	init("rrc", "testrun1");

	out("testexec_%s:\n", testname);
	out("	RRC	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		uint32 res32 = dst >> 1;
		res32 |= dst << 7;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((dst & 1) != 0);
		flag.PV = pflag(res);
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_sla()
{
	init("sla", "testrun1");

	out("testexec_%s:\n", testname);
	out("	SLA	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		uint32 res32 = dst << 1;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((res32 & 0x100) != 0);
		flag.PV = pflag(res);
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_sra()
{
	init("sra", "testrun1");

	out("testexec_%s:\n", testname);
	out("	SRA	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		uint32 res32 = dst >> 1;
		res32 |= (dst & 0x80);
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((dst & 1) != 0);
		flag.PV = pflag(res);
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

static void
gentest_srl()
{
	init("srl", "testrun1");

	out("testexec_%s:\n", testname);
	out("	SRL	A\n");
	out("	RET\n");

	out_testset();

	for (auto dst : values2) {
		uint32 res32 = dst >> 1;
		uint8 res = res32;
		hd64180flag inflag;
		inflag.N = true;
		inflag.H = true;
		hd64180flag flag;
		flag.Z = (res == 0);
		flag.S = ((res & 0x80) != 0);
		flag.C = ((dst & 1) != 0);
		flag.PV = pflag(res);
		flag.H = false;
		flag.N = false;
		out_testdata_run1(dst, inflag, res, flag);
	}
	out("	defb	0,0,0,0FFH\n");
	out("\n");
}

int
main(int ac, char *av[])
{
	out("	org	0100H\n");
	out("	JP	main\n");
	out("\n");

	gentest_add();
	gentest_adc();
	gentest_sub();
	gentest_sbc();
	gentest_and();
	gentest_or();
	gentest_xor();
	gentest_cp();

	gentest_inc();
	gentest_dec();
	gentest_neg();
	gentest_cpl();

	gentest_rla();
	gentest_rra();
	gentest_rlca();
	gentest_rrca();

	gentest_rl();
	gentest_rr();
	gentest_rlc();
	gentest_rrc();
	gentest_sla();
	gentest_sra();
	gentest_srl();

	out("main:\n");
	for (auto name : testlist) {
		out("	CALL	test_%s\n", name.c_str());
	}
	out("	RET\n");

	return 0;
}
