# MD072 - Frontmatter key sort

Aliases: `frontmatter-key-sort`

## Rule Details

This rule ensures that top-level keys in frontmatter are sorted alphabetically. It supports YAML, TOML, and JSON frontmatter formats.

Consistent key ordering makes frontmatter easier to scan and reduces merge conflicts when multiple people edit the same files.

## Supported Formats

| Format | Delimiters | Comments | Auto-fix                      |
| ------ | ---------- | -------- | ----------------------------- |
| YAML   | `---`      | `#`      | Yes (unless comments present) |
| TOML   | `+++`      | `#`      | Yes (unless comments present) |
| JSON   | `{` `}`    | None     | Always                        |

## Examples

### YAML - Incorrect

```markdown
---
title: My Document
author: John Doe
date: 2024-01-15
---

# Heading
```

### YAML - Correct

```markdown
---
author: John Doe
date: 2024-01-15
title: My Document
---

# Heading
```

### TOML - Incorrect

```markdown
+++
title = "My Document"
author = "John Doe"
+++

# Heading
```

### TOML - Correct

```markdown
+++
author = "John Doe"
title = "My Document"
+++

# Heading
```

### JSON - Incorrect

```markdown
{
"title": "My Document",
"author": "John Doe"
}

# Heading
```

### JSON - Correct

```markdown
{
"author": "John Doe",
"title": "My Document"
}

# Heading
```

## Rationale

- Consistent ordering makes it easier to find specific fields
- Reduces merge conflicts when multiple contributors edit frontmatter
- Enforces a predictable structure across all documents
- Aligns with common practices in configuration file management

## Auto-fix Behavior

### YAML and TOML

Auto-fix is available unless the frontmatter contains comments (`#`). Comments would be lost during re-serialization, so the fix is skipped to preserve them.

When comments are present, the warning will indicate:
> "Auto-fix unavailable: frontmatter contains comments."

### JSON

JSON has no comment syntax, so auto-fix is always available.

## Sorting Behavior

- Only top-level keys are sorted (nested keys within objects are not reordered)
- Sorting is case-insensitive (`Author` and `author` are treated equivalently)
- Complex values (lists, nested objects) are preserved during auto-fix

## Configuration

This rule is **disabled by default** (opt-in) because alphabetical key sorting is an opinionated style choice. Many projects prefer semantic ordering (title first, date second) rather than
alphabetical.

| Option      | Type     | Default | Description                                                |
| ----------- | -------- | ------- | ---------------------------------------------------------- |
| `enabled`   | boolean  | `false` | Enable the rule                                            |
| `key-order` | string[] | (none)  | Custom key order (unlisted keys sort alphabetically after) |

### Enabling the Rule

**.rumdl.toml:**

```toml
[MD072]
enabled = true
```

**pyproject.toml:**

```toml
[tool.rumdl.MD072]
enabled = true
```

### Custom Key Order

You can specify a preferred key order instead of alphabetical sorting. Keys not in the list will be sorted alphabetically after the specified keys.

**.rumdl.toml:**

```toml
[MD072]
enabled = true
key-order = ["title", "date", "author", "tags", "categories"]
```

**pyproject.toml:**

```toml
[tool.rumdl.MD072]
enabled = true
key-order = ["title", "date", "author", "tags", "categories"]
```

With this configuration:

- `title` always comes first
- `date` comes second
- `author` comes third
- `tags` and `categories` follow
- Any other keys (e.g., `description`, `draft`) are sorted alphabetically after the specified keys

This is useful for projects that want semantic ordering (title first for readability) while still ensuring consistency for unlisted keys.

## Related Rules

- [MD071](md071.md) - Blank line after frontmatter

## Version

Added in rumdl 0.0.210.
