/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.kafka.connect.integration;

import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.provider.FileConfigProvider;
import org.apache.kafka.common.utils.LogCaptureAppender;
import org.apache.kafka.connect.connector.Task;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.json.JsonConverter;
import org.apache.kafka.connect.json.JsonConverterConfig;
import org.apache.kafka.connect.runtime.distributed.DistributedConfig;
import org.apache.kafka.connect.runtime.rest.entities.ConnectorOffset;
import org.apache.kafka.connect.runtime.rest.entities.ConnectorOffsets;
import org.apache.kafka.connect.runtime.rest.entities.CreateConnectorRequest;
import org.apache.kafka.connect.runtime.rest.errors.ConnectRestException;
import org.apache.kafka.connect.runtime.rest.resources.ConnectorsResource;
import org.apache.kafka.connect.sink.SinkConnector;
import org.apache.kafka.connect.sink.SinkRecord;
import org.apache.kafka.connect.sink.SinkTask;
import org.apache.kafka.connect.storage.StringConverter;
import org.apache.kafka.connect.util.SinkUtils;
import org.apache.kafka.connect.util.clusters.EmbeddedConnectCluster;
import org.apache.kafka.connect.util.clusters.WorkerHandle;
import org.apache.kafka.test.IntegrationTest;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestRule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import java.io.File;
import java.io.FileOutputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static javax.ws.rs.core.Response.Status.INTERNAL_SERVER_ERROR;
import static org.apache.kafka.clients.CommonClientConfigs.BOOTSTRAP_SERVERS_CONFIG;
import static org.apache.kafka.common.config.AbstractConfig.CONFIG_PROVIDERS_CONFIG;
import static org.apache.kafka.common.config.TopicConfig.DELETE_RETENTION_MS_CONFIG;
import static org.apache.kafka.common.config.TopicConfig.SEGMENT_MS_CONFIG;
import static org.apache.kafka.connect.integration.MonitorableSourceConnector.TOPIC_CONFIG;
import static org.apache.kafka.connect.runtime.ConnectorConfig.CONNECTOR_CLASS_CONFIG;
import static org.apache.kafka.connect.runtime.ConnectorConfig.CONNECTOR_CLIENT_PRODUCER_OVERRIDES_PREFIX;
import static org.apache.kafka.connect.runtime.ConnectorConfig.KEY_CONVERTER_CLASS_CONFIG;
import static org.apache.kafka.connect.runtime.ConnectorConfig.TASKS_MAX_CONFIG;
import static org.apache.kafka.connect.runtime.ConnectorConfig.VALUE_CONVERTER_CLASS_CONFIG;
import static org.apache.kafka.connect.runtime.SinkConnectorConfig.TOPICS_CONFIG;
import static org.apache.kafka.connect.runtime.TopicCreationConfig.DEFAULT_TOPIC_CREATION_PREFIX;
import static org.apache.kafka.connect.runtime.TopicCreationConfig.PARTITIONS_CONFIG;
import static org.apache.kafka.connect.runtime.TopicCreationConfig.REPLICATION_FACTOR_CONFIG;
import static org.apache.kafka.connect.runtime.WorkerConfig.CONNECTOR_CLIENT_POLICY_CLASS_CONFIG;
import static org.apache.kafka.connect.runtime.WorkerConfig.OFFSET_COMMIT_INTERVAL_MS_CONFIG;
import static org.apache.kafka.connect.runtime.distributed.DistributedConfig.CONFIG_STORAGE_PREFIX;
import static org.apache.kafka.connect.runtime.distributed.DistributedConfig.CONFIG_TOPIC_CONFIG;
import static org.apache.kafka.connect.runtime.distributed.DistributedConfig.SCHEDULED_REBALANCE_MAX_DELAY_MS_CONFIG;
import static org.apache.kafka.connect.runtime.rest.RestServer.DEFAULT_REST_REQUEST_TIMEOUT_MS;
import static org.apache.kafka.connect.util.clusters.ConnectAssertions.CONNECTOR_SETUP_DURATION_MS;
import static org.apache.kafka.test.TestUtils.waitForCondition;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

/**
 * Test simple operations on the workers of a Connect cluster.
 */
@Category(IntegrationTest.class)
public class ConnectWorkerIntegrationTest {
    private static final Logger log = LoggerFactory.getLogger(ConnectWorkerIntegrationTest.class);

    private static final int NUM_TOPIC_PARTITIONS = 3;
    private static final long OFFSET_COMMIT_INTERVAL_MS = TimeUnit.SECONDS.toMillis(30);
    private static final int NUM_WORKERS = 3;
    private static final int NUM_TASKS = 4;
    private static final int MESSAGES_PER_POLL = 10;
    private static final String CONNECTOR_NAME = "simple-connector";
    private static final String TOPIC_NAME = "test-topic";

    private EmbeddedConnectCluster.Builder connectBuilder;
    private EmbeddedConnectCluster connect;
    private Map<String, String> workerProps;
    private Properties brokerProps;

    @Rule
    public TestRule watcher = ConnectIntegrationTestUtils.newTestWatcher(log);

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    @Before
    public void setup() {
        // setup Connect worker properties
        workerProps = new HashMap<>();
        workerProps.put(OFFSET_COMMIT_INTERVAL_MS_CONFIG, String.valueOf(OFFSET_COMMIT_INTERVAL_MS));
        workerProps.put(CONNECTOR_CLIENT_POLICY_CLASS_CONFIG, "All");

        // setup Kafka broker properties
        brokerProps = new Properties();
        brokerProps.put("auto.create.topics.enable", String.valueOf(false));

        // build a Connect cluster backed by Kafka and Zk
        connectBuilder = new EmbeddedConnectCluster.Builder()
                .name("connect-cluster")
                .numWorkers(NUM_WORKERS)
                .workerProps(workerProps)
                .brokerProps(brokerProps)
                .maskExitProcedures(true); // true is the default, setting here as example
    }

    @After
    public void close() {
        // stop all Connect, Kafka and Zk threads.
        connect.stop();
    }

    /**
     * Simple test case to add and then remove a worker from the embedded Connect cluster while
     * running a simple source connector.
     */
    @Test
    public void testAddAndRemoveWorker() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        // create test topic
        connect.kafka().createTopic(TOPIC_NAME, NUM_TOPIC_PARTITIONS);

        // set up props for the source connector
        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        // start a source connector
        connect.configureConnector(CONNECTOR_NAME, props);

        connect.assertions().assertConnectorAndAtLeastNumTasksAreRunning(CONNECTOR_NAME, NUM_TASKS,
                "Connector tasks did not start in time.");

        WorkerHandle extraWorker = connect.addWorker();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS + 1,
                "Expanded group of workers did not start in time.");

        connect.assertions().assertConnectorAndAtLeastNumTasksAreRunning(CONNECTOR_NAME, NUM_TASKS,
                "Connector tasks are not all in running state.");

        Set<WorkerHandle> workers = connect.activeWorkers();
        assertTrue(workers.contains(extraWorker));

        connect.removeWorker(extraWorker);

        connect.assertions().assertExactlyNumWorkersAreUp(NUM_WORKERS,
                "Group of workers did not shrink in time.");

        workers = connect.activeWorkers();
        assertFalse(workers.contains(extraWorker));
    }

    /**
     * Verify that a failed task can be restarted successfully.
     */
    @Test
    public void testRestartFailedTask() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        int numTasks = 1;

        // setup up props for the source connector
        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);
        // Properties for the source connector. The task should fail at startup due to the bad broker address.
        props.put(TASKS_MAX_CONFIG, Objects.toString(numTasks));
        props.put(CONNECTOR_CLIENT_PRODUCER_OVERRIDES_PREFIX + BOOTSTRAP_SERVERS_CONFIG, "nobrokerrunningatthisaddress");

        connect.assertions().assertExactlyNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        // Try to start the connector and its single task.
        connect.configureConnector(CONNECTOR_NAME, props);

        connect.assertions().assertConnectorIsRunningAndTasksHaveFailed(CONNECTOR_NAME, numTasks,
                "Connector tasks did not fail in time");

        // Reconfigure the connector without the bad broker address.
        props.remove(CONNECTOR_CLIENT_PRODUCER_OVERRIDES_PREFIX + BOOTSTRAP_SERVERS_CONFIG);
        connect.configureConnector(CONNECTOR_NAME, props);

        // Restart the failed task
        String taskRestartEndpoint = connect.endpointForResource(
            String.format("connectors/%s/tasks/0/restart", CONNECTOR_NAME));
        connect.requestPost(taskRestartEndpoint, "", Collections.emptyMap());

        // Ensure the task started successfully this time
        connect.assertions().assertConnectorAndAtLeastNumTasksAreRunning(CONNECTOR_NAME, numTasks,
            "Connector tasks are not all in running state.");
    }

    /**
     * Verify that a set of tasks restarts correctly after a broker goes offline and back online
     */
    @Test
    public void testBrokerCoordinator() throws Exception {
        ConnectorHandle connectorHandle = RuntimeHandles.get().connectorHandle(CONNECTOR_NAME);
        workerProps.put(DistributedConfig.SCHEDULED_REBALANCE_MAX_DELAY_MS_CONFIG, String.valueOf(5000));
        connect = connectBuilder.workerProps(workerProps).build();
        // start the clusters
        connect.start();
        int numTasks = 4;
        // create test topic
        connect.kafka().createTopic(TOPIC_NAME, NUM_TOPIC_PARTITIONS);

        // set up props for the source connector
        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        // start a source connector
        connect.configureConnector(CONNECTOR_NAME, props);

        connect.assertions().assertConnectorAndAtLeastNumTasksAreRunning(CONNECTOR_NAME, numTasks,
                "Connector tasks did not start in time.");

        // expect that the connector will be stopped once the coordinator is detected to be down
        StartAndStopLatch stopLatch = connectorHandle.expectedStops(1, false);

        connect.kafka().stopOnlyKafka();

        connect.assertions().assertExactlyNumWorkersAreUp(NUM_WORKERS,
                "Group of workers did not remain the same after broker shutdown");

        // Allow for the workers to discover that the coordinator is unavailable, wait is
        // heartbeat timeout * 2 + 4sec
        Thread.sleep(TimeUnit.SECONDS.toMillis(10));

        // Wait for the connector to be stopped
        assertTrue("Failed to stop connector and tasks after coordinator failure within "
                        + CONNECTOR_SETUP_DURATION_MS + "ms",
                stopLatch.await(CONNECTOR_SETUP_DURATION_MS, TimeUnit.MILLISECONDS));

        StartAndStopLatch startLatch = connectorHandle.expectedStarts(1, false);
        connect.kafka().startOnlyKafkaOnSamePorts();

        // Allow for the kafka brokers to come back online
        Thread.sleep(TimeUnit.SECONDS.toMillis(10));

        connect.assertions().assertExactlyNumWorkersAreUp(NUM_WORKERS,
                "Group of workers did not remain the same within the designated time.");

        // Allow for the workers to rebalance and reach a steady state
        Thread.sleep(TimeUnit.SECONDS.toMillis(10));

        connect.assertions().assertConnectorAndAtLeastNumTasksAreRunning(CONNECTOR_NAME, numTasks,
                "Connector tasks did not start in time.");

        // Expect that the connector has started again
        assertTrue("Failed to stop connector and tasks after coordinator failure within "
                        + CONNECTOR_SETUP_DURATION_MS + "ms",
                startLatch.await(CONNECTOR_SETUP_DURATION_MS, TimeUnit.MILLISECONDS));
    }

    /**
     * Verify that the number of tasks listed in the REST API is updated correctly after changes to
     * the "tasks.max" connector configuration.
     */
    @Test
    public void testTaskStatuses() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        // base connector props
        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);
        props.put(CONNECTOR_CLASS_CONFIG, MonitorableSourceConnector.class.getSimpleName());

        // start the connector with only one task
        int initialNumTasks = 1;
        props.put(TASKS_MAX_CONFIG, String.valueOf(initialNumTasks));
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(CONNECTOR_NAME,
                initialNumTasks, "Connector tasks did not start in time");

        // then reconfigure it to use more tasks
        int increasedNumTasks = 5;
        props.put(TASKS_MAX_CONFIG, String.valueOf(increasedNumTasks));
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(CONNECTOR_NAME,
                increasedNumTasks, "Connector task statuses did not update in time.");

        // then reconfigure it to use fewer tasks
        int decreasedNumTasks = 3;
        props.put(TASKS_MAX_CONFIG, String.valueOf(decreasedNumTasks));
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(CONNECTOR_NAME,
                decreasedNumTasks, "Connector task statuses did not update in time.");
    }

    @Test
    public void testSourceTaskNotBlockedOnShutdownWithNonExistentTopic() throws Exception {
        // When automatic topic creation is disabled on the broker
        brokerProps.put("auto.create.topics.enable", "false");
        connect = connectBuilder
            .brokerProps(brokerProps)
            .numWorkers(1)
            .numBrokers(1)
            .build();
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(1, "Initial group of workers did not start in time.");

        // and when the connector is not configured to create topics
        Map<String, String> props = defaultSourceConnectorProps("nonexistenttopic");
        props.remove(DEFAULT_TOPIC_CREATION_PREFIX + REPLICATION_FACTOR_CONFIG);
        props.remove(DEFAULT_TOPIC_CREATION_PREFIX + PARTITIONS_CONFIG);
        props.put("throughput", "-1");

        ConnectorHandle connector = RuntimeHandles.get().connectorHandle(CONNECTOR_NAME);
        connector.expectedRecords(NUM_TASKS * MESSAGES_PER_POLL);
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(CONNECTOR_NAME,
            NUM_TASKS, "Connector tasks did not start in time");
        connector.awaitRecords(TimeUnit.MINUTES.toMillis(1));

        // Then if we delete the connector, it and each of its tasks should be stopped by the framework
        // even though the producer is blocked because there is no topic
        StartAndStopLatch stopCounter = connector.expectedStops(1);
        connect.deleteConnector(CONNECTOR_NAME);

        assertTrue("Connector and all tasks were not stopped in time", stopCounter.await(1, TimeUnit.MINUTES));
    }

    /**
     * Verify that the target state (started, paused, stopped) of a connector can be updated, with
     * an emphasis on ensuring that the transitions between each state are correct.
     * <p>
     * The transitions we need to cover are:
     * <ol>
     *     <li>RUNNING -> PAUSED</li>
     *     <li>RUNNING -> STOPPED</li>
     *     <li>PAUSED -> RUNNING</li>
     *     <li>PAUSED -> STOPPED</li>
     *     <li>STOPPED -> RUNNING</li>
     *     <li>STOPPED -> PAUSED</li>
     * </ol>
     * With some reordering, we can perform each transition just once:
     * <ul>
     *     <li>Start with RUNNING</li>
     *     <li>Transition to STOPPED (2)</li>
     *     <li>Transition to RUNNING (5)</li>
     *     <li>Transition to PAUSED (1)</li>
     *     <li>Transition to STOPPED (4)</li>
     *     <li>Transition to PAUSED (6)</li>
     *     <li>Transition to RUNNING (3)</li>
     * </ul>
     */
    @Test
    public void testPauseStopResume() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        // Want to make sure to use multiple tasks
        final int numTasks = 4;
        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);
        props.put(TASKS_MAX_CONFIG, Integer.toString(numTasks));

        // Start with RUNNING
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                numTasks,
                "Connector tasks did not start in time"
        );

        // Transition to STOPPED
        connect.stopConnector(CONNECTOR_NAME);
        // Issue a second request to ensure that this operation is idempotent
        connect.stopConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorIsStopped(
                CONNECTOR_NAME,
                "Connector did not stop in time"
        );
        // If the connector is truly stopped, we should also see an empty set of tasks and task configs
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Transition to RUNNING
        connect.resumeConnector(CONNECTOR_NAME);
        // Issue a second request to ensure that this operation is idempotent
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                numTasks,
                "Connector tasks did not resume in time"
        );

        // Transition to PAUSED
        connect.pauseConnector(CONNECTOR_NAME);
        // Issue a second request to ensure that this operation is idempotent
        connect.pauseConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksArePaused(
                CONNECTOR_NAME,
                numTasks,
                "Connector did not pause in time"
        );

        // Transition to STOPPED
        connect.stopConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorIsStopped(
                CONNECTOR_NAME,
                "Connector did not stop in time"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Transition to PAUSED
        connect.pauseConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksArePaused(
                CONNECTOR_NAME,
                0,
                "Connector did not pause in time"
        );

        // Transition to RUNNING
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                numTasks,
                "Connector tasks did not resume in time"
        );

        // Delete the connector
        connect.deleteConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorDoesNotExist(
                CONNECTOR_NAME,
                "Connector wasn't deleted in time"
        );
    }

    /**
     * Test out the {@code STOPPED} state introduced in
     * <a href="https://cwiki.apache.org/confluence/display/KAFKA/KIP-875%3A+First-class+offsets+support+in+Kafka+Connect#KIP875:FirstclassoffsetssupportinKafkaConnect-Newtargetstate:STOPPED">KIP-875</a>,
     * with an emphasis on correctly handling errors thrown from the connector.
     */
    @Test
    public void testStoppedState() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
                "Initial group of workers did not start in time.");

        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);
        // Fail the connector on startup
        props.put("connector.start.inject.error", "true");

        // Start the connector (should fail immediately and generate no tasks)
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.assertions().assertConnectorIsFailedAndTasksHaveFailed(
                CONNECTOR_NAME,
                0,
                "Connector should have failed and not generated any tasks"
        );

        // Stopping a failed connector updates its state to STOPPED in the REST API
        connect.stopConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorIsStopped(
                CONNECTOR_NAME,
                "Connector did not stop in time"
        );
        // If the connector is truly stopped, we should also see an empty set of tasks and task configs
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Can resume a connector after its Connector has failed before shutdown after receiving a stop request
        props.remove("connector.start.inject.error");
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                NUM_TASKS,
                "Connector or tasks did not start running healthily in time"
        );

        // Fail the connector on shutdown
        props.put("connector.stop.inject.error", "true");
        // Stopping a connector that fails during shutdown after receiving a stop request updates its state to STOPPED in the REST API
        connect.configureConnector(CONNECTOR_NAME, props);
        connect.stopConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorIsStopped(
                CONNECTOR_NAME,
                "Connector did not stop in time"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Can resume a connector after its Connector has failed during shutdown after receiving a stop request
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                NUM_TASKS,
                "Connector or tasks did not start running healthily in time"
        );

        // Can delete a stopped connector
        connect.deleteConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorDoesNotExist(
                CONNECTOR_NAME,
                "Connector wasn't deleted in time"
        );
    }

    /**
     * The <strong><em>GET /connectors/{connector}/tasks-config</em></strong> endpoint was deprecated in
     * <a href="https://cwiki.apache.org/confluence/display/KAFKA/KIP-970%3A+Deprecate+and+remove+Connect%27s+redundant+task+configurations+endpoint">KIP-970</a>
     * and is slated for removal in the next major release. This test verifies that the deprecation warning log is emitted on trying to use the
     * deprecated endpoint.
     */
    @Test
    public void testTasksConfigDeprecation() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(
            NUM_WORKERS,
            "Initial group of workers did not start in time."
        );

        connect.configureConnector(CONNECTOR_NAME, defaultSourceConnectorProps(TOPIC_NAME));
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
            CONNECTOR_NAME,
            NUM_TASKS,
            "Connector tasks did not start in time"
        );

        try (LogCaptureAppender logCaptureAppender = LogCaptureAppender.createAndRegister(ConnectorsResource.class)) {
            connect.requestGet(connect.endpointForResource("connectors/" + CONNECTOR_NAME + "/tasks-config"));
            List<LogCaptureAppender.Event> logEvents = logCaptureAppender.getEvents();
            assertEquals(1, logEvents.size());
            assertEquals(Level.WARN.toString(), logEvents.get(0).getLevel());
            assertThat(logEvents.get(0).getMessage(), containsString("deprecated"));
        }

    }

    @Test
    public void testCreateConnectorWithPausedInitialState() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
            "Initial group of workers did not start in time.");

        CreateConnectorRequest createConnectorRequest = new CreateConnectorRequest(
            CONNECTOR_NAME,
            defaultSourceConnectorProps(TOPIC_NAME),
            CreateConnectorRequest.InitialState.PAUSED
        );
        connect.configureConnector(createConnectorRequest);

        // Verify that the connector's status is PAUSED and also that no tasks were spawned for the connector
        connect.assertions().assertConnectorAndExactlyNumTasksArePaused(
            CONNECTOR_NAME,
            0,
            "Connector was not created in a paused state"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Verify that a connector created in the PAUSED state can be resumed successfully
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
            CONNECTOR_NAME,
            NUM_TASKS,
            "Connector or tasks did not start running healthily in time"
        );
    }

    @Test
    public void testCreateSourceConnectorWithStoppedInitialStateAndModifyOffsets() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
            "Initial group of workers did not start in time.");

        Map<String, String> props = defaultSourceConnectorProps(TOPIC_NAME);

        // Configure the connector to produce a maximum of 10 messages
        props.put("max.messages", "10");
        props.put(TASKS_MAX_CONFIG, "1");
        CreateConnectorRequest createConnectorRequest = new CreateConnectorRequest(
            CONNECTOR_NAME,
            props,
            CreateConnectorRequest.InitialState.STOPPED
        );
        connect.configureConnector(createConnectorRequest);

        // Verify that the connector's status is STOPPED and also that no tasks were spawned for the connector
        connect.assertions().assertConnectorIsStopped(
            CONNECTOR_NAME,
            "Connector was not created in a stopped state"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Verify that the offsets can be modified for a source connector created in the STOPPED state

        // Alter the offsets so that only 5 messages are produced
        connect.alterSourceConnectorOffset(
            CONNECTOR_NAME,
            Collections.singletonMap("task.id", CONNECTOR_NAME + "-0"),
            Collections.singletonMap("saved", 5L)
        );

        // Verify that a connector created in the STOPPED state can be resumed successfully
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
            CONNECTOR_NAME,
            1,
            "Connector or tasks did not start running healthily in time"
        );

        // Verify that only 5 messages were produced. We verify this by consuming all the messages from the topic after we've already ensured that at
        // least 5 messages can be consumed.
        long timeoutMs = TimeUnit.SECONDS.toMillis(10);
        connect.kafka().consume(5, timeoutMs, TOPIC_NAME);
        assertEquals(5, connect.kafka().consumeAll(timeoutMs, TOPIC_NAME).count());
    }

    @Test
    public void testCreateSinkConnectorWithStoppedInitialStateAndModifyOffsets() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
            "Initial group of workers did not start in time.");

        // Create topic and produce 10 messages
        connect.kafka().createTopic(TOPIC_NAME);
        for (int i = 0; i < 10; i++) {
            connect.kafka().produce(TOPIC_NAME, "Message " + i);
        }

        Map<String, String> props = defaultSinkConnectorProps(TOPIC_NAME);
        props.put(TASKS_MAX_CONFIG, "1");

        CreateConnectorRequest createConnectorRequest = new CreateConnectorRequest(
            CONNECTOR_NAME,
            props,
            CreateConnectorRequest.InitialState.STOPPED
        );
        connect.configureConnector(createConnectorRequest);

        // Verify that the connector's status is STOPPED and also that no tasks were spawned for the connector
        connect.assertions().assertConnectorIsStopped(
            CONNECTOR_NAME,
            "Connector was not created in a stopped state"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Verify that the offsets can be modified for a sink connector created in the STOPPED state

        // Alter the offsets so that the first 5 messages in the topic are skipped
        connect.alterSinkConnectorOffset(CONNECTOR_NAME, new TopicPartition(TOPIC_NAME, 0), 5L);

        // This will cause the connector task to fail if it encounters a record with offset < 5
        TaskHandle taskHandle = RuntimeHandles.get().connectorHandle(CONNECTOR_NAME).taskHandle(CONNECTOR_NAME + "-0",
            sinkRecord -> {
                if (sinkRecord.kafkaOffset() < 5L) {
                    throw new ConnectException("Unexpected record encountered: " + sinkRecord);
                }
            });

        // We produced 10 records and altered the connector offsets to skip over the first 5, so we expect 5 records to be consumed
        taskHandle.expectedRecords(5);

        // Verify that a connector created in the STOPPED state can be resumed successfully
        connect.resumeConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
            CONNECTOR_NAME,
            1,
            "Connector or tasks did not start running healthily in time"
        );

        taskHandle.awaitRecords(TimeUnit.SECONDS.toMillis(10));

        // Confirm that the task is still running (i.e. it didn't fail due to encountering any records with offset < 5)
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
            CONNECTOR_NAME,
            1,
            "Connector or tasks did not start running healthily in time"
        );
    }

    @Test
    public void testDeleteConnectorCreatedWithPausedOrStoppedInitialState() throws Exception {
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(NUM_WORKERS,
            "Initial group of workers did not start in time.");

        // Create a connector with PAUSED initial state
        CreateConnectorRequest createConnectorRequest = new CreateConnectorRequest(
            CONNECTOR_NAME,
            defaultSourceConnectorProps(TOPIC_NAME),
            CreateConnectorRequest.InitialState.PAUSED
        );
        connect.configureConnector(createConnectorRequest);

        // Verify that the connector's status is PAUSED and also that no tasks were spawned for the connector
        connect.assertions().assertConnectorAndExactlyNumTasksArePaused(
            CONNECTOR_NAME,
            0,
            "Connector was not created in a paused state"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Verify that a connector created in the PAUSED state can be deleted successfully
        connect.deleteConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorDoesNotExist(CONNECTOR_NAME, "Connector wasn't deleted in time");


        // Create a connector with STOPPED initial state
        createConnectorRequest = new CreateConnectorRequest(
            CONNECTOR_NAME,
            defaultSourceConnectorProps(TOPIC_NAME),
            CreateConnectorRequest.InitialState.STOPPED
        );
        connect.configureConnector(createConnectorRequest);

        // Verify that the connector's status is STOPPED and also that no tasks were spawned for the connector
        connect.assertions().assertConnectorIsStopped(
            CONNECTOR_NAME,
            "Connector was not created in a stopped state"
        );
        assertEquals(Collections.emptyList(), connect.connectorInfo(CONNECTOR_NAME).tasks());
        assertEquals(Collections.emptyList(), connect.taskConfigs(CONNECTOR_NAME));

        // Verify that a connector created in the STOPPED state can be deleted successfully
        connect.deleteConnector(CONNECTOR_NAME);
        connect.assertions().assertConnectorDoesNotExist(CONNECTOR_NAME, "Connector wasn't deleted in time");
    }

    private Map<String, String> defaultSinkConnectorProps(String topics) {
        // setup props for the sink connector
        Map<String, String> props = new HashMap<>();
        props.put(CONNECTOR_CLASS_CONFIG, MonitorableSinkConnector.class.getSimpleName());
        props.put(TASKS_MAX_CONFIG, String.valueOf(NUM_TASKS));
        props.put(TOPICS_CONFIG, topics);

        return props;
    }

    @Test
    public void testRequestTimeouts() throws Exception {
        final String configTopic = "test-request-timeout-configs";
        workerProps.put(CONFIG_TOPIC_CONFIG, configTopic);
        // Workaround for KAFKA-15676, which can cause the scheduled rebalance delay to
        // be spuriously triggered after the group coordinator for a Connect cluster is bounced
        workerProps.put(SCHEDULED_REBALANCE_MAX_DELAY_MS_CONFIG, "0");
        connect = connectBuilder
                .numBrokers(1)
                .numWorkers(1)
                .build();
        connect.start();
        connect.assertions().assertAtLeastNumWorkersAreUp(1,
                "Worker did not start in time");

        Map<String, String> connectorConfig1 = defaultSourceConnectorProps(TOPIC_NAME);
        Map<String, String> connectorConfig2 = new HashMap<>(connectorConfig1);
        connectorConfig2.put(TASKS_MAX_CONFIG, Integer.toString(NUM_TASKS + 1));

        // Create a connector to ensure that the worker has completed startup
        log.info("Creating initial connector");
        connect.configureConnector(CONNECTOR_NAME, connectorConfig1);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME, NUM_TASKS, "connector and tasks did not start in time"
        );

        // Bring down Kafka, which should cause some REST requests to fail
        log.info("Stopping Kafka cluster");
        connect.kafka().stopOnlyKafka();

        // Try to reconfigure the connector, which should fail with a timeout error
        log.info("Trying to reconfigure connector while Kafka cluster is down");
        assertTimeoutException(
                () -> connect.configureConnector(CONNECTOR_NAME, connectorConfig2),
                "flushing updates to the status topic"
        );
        log.info("Restarting Kafka cluster");
        connect.kafka().startOnlyKafkaOnSamePorts();
        connect.assertions().assertExactlyNumBrokersAreUp(1, "Broker did not complete startup in time");
        log.info("Kafka cluster is restarted");

        // Reconfigure the connector to ensure that the broker has completed startup
        log.info("Reconfiguring connector with one more task");
        connect.configureConnector(CONNECTOR_NAME, connectorConfig2);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME, NUM_TASKS + 1, "connector and tasks did not start in time"
        );

        // Delete the config topic--WCGW?
        log.info("Deleting Kafka Connect config topic");
        connect.kafka().deleteTopic(configTopic);

        // Try to delete the connector, which should fail with a slightly-different timeout error
        log.info("Trying to reconfigure connector after config topic has been deleted");
        assertTimeoutException(
                () -> connect.deleteConnector(CONNECTOR_NAME),
                "removing the config for connector " + CONNECTOR_NAME + " from the config topic"
        );

        // The worker should still be blocked on the same operation
        log.info("Trying again to reconfigure connector after config topic has been deleted");
        assertTimeoutException(
                () -> connect.configureConnector(CONNECTOR_NAME, connectorConfig1),
                "removing the config for connector " + CONNECTOR_NAME + " from the config topic"
        );
    }

    private void assertTimeoutException(Runnable operation, String expectedStageDescription) throws InterruptedException {
        connect.requestTimeout(1_000);
        AtomicReference<Throwable> latestError = new AtomicReference<>();
        waitForCondition(
                () -> {
                    try {
                        operation.run();
                        latestError.set(null);
                        return false;
                    } catch (Throwable t) {
                        latestError.set(t);
                        assertTrue(t instanceof ConnectRestException);
                        ConnectRestException restException = (ConnectRestException) t;

                        assertEquals(INTERNAL_SERVER_ERROR.getStatusCode(), restException.statusCode());
                        assertNotNull(restException.getMessage());
                        assertTrue(
                                "Message '" + restException.getMessage() + "' does not match expected format",
                                restException.getMessage().contains("Request timed out. The worker is currently " + expectedStageDescription)
                        );

                        return true;
                    }
                },
                30_000,
                () -> {
                    String baseMessage = "REST request did not time out with expected error message in time. ";
                    Throwable t = latestError.get();
                    if (t == null) {
                        return baseMessage + "The most recent request did not fail.";
                    } else {
                        return baseMessage + "Most recent error: " + t;
                    }
                }
        );
        connect.requestTimeout(DEFAULT_REST_REQUEST_TIMEOUT_MS);
    }

    /**
     * Task configs are not removed from the config topic after a connector is deleted.
     * When topic compaction takes place, this can cause the tombstone message for the
     * connector config to be deleted, leaving the task configs in the config topic with no
     * explicit record of the connector's deletion.
     * <p>
     * This test guarantees that those older task configs are never used, even when the
     * connector is recreated later.
     */
    @Test
    public void testCompactedDeletedOlderConnectorConfig() throws Exception {
        brokerProps.put("log.cleaner.backoff.ms", "100");
        brokerProps.put("log.cleaner.delete.retention.ms", "1");
        brokerProps.put("log.cleaner.max.compaction.lag.ms", "1");
        brokerProps.put("log.cleaner.min.cleanable.ratio", "0");
        brokerProps.put("log.cleaner.min.compaction.lag.ms", "1");
        brokerProps.put("log.cleaner.threads", "1");

        final String configTopic = "kafka-16838-configs";
        final int offsetCommitIntervalMs = 100;
        workerProps.put(CONFIG_TOPIC_CONFIG, configTopic);
        workerProps.put(CONFIG_STORAGE_PREFIX + SEGMENT_MS_CONFIG, "100");
        workerProps.put(CONFIG_STORAGE_PREFIX + DELETE_RETENTION_MS_CONFIG, "1");
        workerProps.put(OFFSET_COMMIT_INTERVAL_MS_CONFIG, Integer.toString(offsetCommitIntervalMs));

        final int numWorkers = 1;
        connect = connectBuilder
                .numWorkers(numWorkers)
                .build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(
                numWorkers,
                "Initial group of workers did not start in time."
        );

        final String connectorTopic = "connector-topic";
        connect.kafka().createTopic(connectorTopic, 1);

        ConnectorHandle connectorHandle = RuntimeHandles.get().connectorHandle(CONNECTOR_NAME);
        connectorHandle.expectedCommits(NUM_TASKS * 2);

        Map<String, String> connectorConfig = defaultSourceConnectorProps(connectorTopic);
        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                NUM_TASKS,
                "Connector or its tasks did not start in time"
        );
        connectorHandle.awaitCommits(offsetCommitIntervalMs * 3);

        connect.deleteConnector(CONNECTOR_NAME);

        // Roll the entire cluster
        connect.activeWorkers().forEach(connect::removeWorker);

        // Miserable hack: produce directly to the config topic and then wait a little bit
        // in order to trigger segment rollover and allow compaction to take place
        connect.kafka().produce(configTopic, "garbage-key-1", null);
        Thread.sleep(1_000);
        connect.kafka().produce(configTopic, "garbage-key-2", null);
        Thread.sleep(1_000);

        for (int i = 0; i < numWorkers; i++)
            connect.addWorker();

        connect.assertions().assertAtLeastNumWorkersAreUp(
                numWorkers,
                "Initial group of workers did not start in time."
        );

        final TopicPartition connectorTopicPartition = new TopicPartition(connectorTopic, 0);
        final long initialEndOffset = connect.kafka().endOffset(connectorTopicPartition);
        assertTrue(
                "Source connector should have published at least one record to Kafka",
                initialEndOffset > 0
        );

        connectorHandle.expectedCommits(NUM_TASKS * 2);

        // Re-create the connector with a different config (targets a different topic)
        final String otherConnectorTopic = "other-topic";
        connect.kafka().createTopic(otherConnectorTopic, 1);
        connectorConfig.put(TOPIC_CONFIG, otherConnectorTopic);
        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                NUM_TASKS,
                "Connector or its tasks did not start in time"
        );
        connectorHandle.awaitCommits(offsetCommitIntervalMs * 3);

        // See if any new records got written to the old topic
        final long nextEndOffset = connect.kafka().endOffset(connectorTopicPartition);
        assertEquals(
                "No new records should have been written to the older topic",
                initialEndOffset,
                nextEndOffset
        );
    }

    /**
     * If a connector has existing tasks, and then generates new task configs, workers compare the
     * new and existing configs before publishing them to the config topic. If there is no difference,
     * workers do not publish task configs (this is a workaround to prevent infinite loops with eager
     * rebalancing).
     * <p>
     * This test tries to guarantee that, if the old task configs become invalid because of
     * an invalid config provider reference, it will still be possible to reconfigure the connector.
     */
    @Test
    public void testReconfigureConnectorWithFailingTaskConfigs() throws Exception {
        final int offsetCommitIntervalMs = 100;
        workerProps.put(CONFIG_PROVIDERS_CONFIG, "file");
        workerProps.put(CONFIG_PROVIDERS_CONFIG + ".file.class", FileConfigProvider.class.getName());
        workerProps.put(OFFSET_COMMIT_INTERVAL_MS_CONFIG, Integer.toString(offsetCommitIntervalMs));

        final int numWorkers = 1;
        connect = connectBuilder
                .numWorkers(numWorkers)
                .build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(
                numWorkers,
                "Initial group of workers did not start in time."
        );

        final String firstConnectorTopic = "connector-topic-1";
        connect.kafka().createTopic(firstConnectorTopic);

        final File secretsFile = tmp.newFile("test-secrets");
        final Properties secrets = new Properties();
        final String throughputSecretKey = "secret-throughput";
        secrets.put(throughputSecretKey, "10");
        try (FileOutputStream secretsOutputStream = new FileOutputStream(secretsFile)) {
            secrets.store(secretsOutputStream, null);
        }

        ConnectorHandle connectorHandle = RuntimeHandles.get().connectorHandle(CONNECTOR_NAME);
        connectorHandle.expectedCommits(NUM_TASKS * 2);

        Map<String, String> connectorConfig = defaultSourceConnectorProps(firstConnectorTopic);
        connectorConfig.put(
                "throughput",
                "${file:" + secretsFile.getAbsolutePath() + ":" + throughputSecretKey + "}"
        );
        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                NUM_TASKS,
                "Connector or its tasks did not start in time"
        );
        connectorHandle.awaitCommits(offsetCommitIntervalMs * 3);

        // Delete the secrets file, which should render the old task configs invalid
        assertTrue("Failed to delete secrets file", secretsFile.delete());

        // Use a start latch here instead of assertConnectorAndExactlyNumTasksAreRunning
        // since failure to reconfigure the tasks (which may occur if the bug this test was written
        // to help catch resurfaces) will not cause existing tasks to fail or stop running
        StartAndStopLatch restarts = connectorHandle.expectedStarts(1);
        connectorHandle.expectedCommits(NUM_TASKS * 2);

        final String secondConnectorTopic = "connector-topic-2";
        connect.kafka().createTopic(secondConnectorTopic, 1);

        // Stop using the config provider for this connector, and instruct it to start writing to the
        // old topic again
        connectorConfig.put("throughput", "10");
        connectorConfig.put(TOPIC_CONFIG, secondConnectorTopic);
        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        assertTrue(
                "Connector tasks were not restarted in time",
                restarts.await(10, TimeUnit.SECONDS)
        );
        connectorHandle.awaitCommits(offsetCommitIntervalMs * 3);

        final long endOffset = connect.kafka().endOffset(new TopicPartition(secondConnectorTopic, 0));
        assertTrue(
                "Source connector should have published at least one record to new Kafka topic "
                    + "after being reconfigured",
                endOffset > 0
        );
    }

    @Test
    public void testRuntimePropertyReconfiguration() throws Exception {
        final int offsetCommitIntervalMs = 1_000;
        // force fast offset commits
        workerProps.put(OFFSET_COMMIT_INTERVAL_MS_CONFIG, Integer.toString(offsetCommitIntervalMs));
        connect = connectBuilder.build();
        // start the clusters
        connect.start();

        connect.assertions().assertAtLeastNumWorkersAreUp(
                NUM_WORKERS,
                "Initial group of workers did not start in time."
        );

        final String topic = "kafka9228";
        connect.kafka().createTopic(topic, 1);
        connect.kafka().produce(topic, "non-json-value");

        Map<String, String> connectorConfig = new HashMap<>();
        connectorConfig.put(CONNECTOR_CLASS_CONFIG, EmptyTaskConfigsConnector.class.getName());
        connectorConfig.put(TASKS_MAX_CONFIG, "1");
        connectorConfig.put(TOPICS_CONFIG, topic);
        // Initially configure the connector to use the JSON converter, which should cause task failure(s)
        connectorConfig.put(VALUE_CONVERTER_CLASS_CONFIG, JsonConverter.class.getName());
        connectorConfig.put(
                VALUE_CONVERTER_CLASS_CONFIG + "." + JsonConverterConfig.SCHEMAS_ENABLE_CONFIG,
                "false"
        );

        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        connect.assertions().assertConnectorIsRunningAndTasksHaveFailed(
                CONNECTOR_NAME,
                1,
                "Connector did not start or task did not fail in time"
        );
        assertEquals(
                "Connector should not have any committed offsets when only task fails on first record",
                new ConnectorOffsets(Collections.emptyList()),
                connect.connectorOffsets(CONNECTOR_NAME)
        );

        // Reconfigure the connector to use the string converter, which should not cause any more task failures
        connectorConfig.put(VALUE_CONVERTER_CLASS_CONFIG, StringConverter.class.getName());
        connectorConfig.remove(
                KEY_CONVERTER_CLASS_CONFIG + "." + JsonConverterConfig.SCHEMAS_ENABLE_CONFIG
        );
        connect.configureConnector(CONNECTOR_NAME, connectorConfig);
        connect.assertions().assertConnectorAndExactlyNumTasksAreRunning(
                CONNECTOR_NAME,
                1,
                "Connector or tasks did not start in time"
        );

        Map<String, Object> expectedOffsetKey = new HashMap<>();
        expectedOffsetKey.put(SinkUtils.KAFKA_TOPIC_KEY, topic);
        expectedOffsetKey.put(SinkUtils.KAFKA_PARTITION_KEY, 0);
        Map<String, Object> expectedOffsetValue = Collections.singletonMap(SinkUtils.KAFKA_OFFSET_KEY, 1);
        ConnectorOffset expectedOffset = new ConnectorOffset(expectedOffsetKey, expectedOffsetValue);
        ConnectorOffsets expectedOffsets = new ConnectorOffsets(Collections.singletonList(expectedOffset));

        // Wait for it to commit offsets, signaling that it has successfully processed the record we produced earlier
        waitForCondition(
                () -> expectedOffsets.equals(connect.connectorOffsets(CONNECTOR_NAME)),
                offsetCommitIntervalMs * 2,
                "Task did not successfully process record and/or commit offsets in time"
        );
    }

    private Map<String, String> defaultSourceConnectorProps(String topic) {
        // setup props for the source connector
        Map<String, String> props = new HashMap<>();
        props.put(CONNECTOR_CLASS_CONFIG, MonitorableSourceConnector.class.getSimpleName());
        props.put(TASKS_MAX_CONFIG, String.valueOf(NUM_TASKS));
        props.put(TOPIC_CONFIG, topic);
        props.put("throughput", "10");
        props.put("messages.per.poll", String.valueOf(MESSAGES_PER_POLL));
        props.put(KEY_CONVERTER_CLASS_CONFIG, StringConverter.class.getName());
        props.put(VALUE_CONVERTER_CLASS_CONFIG, StringConverter.class.getName());
        props.put(DEFAULT_TOPIC_CREATION_PREFIX + REPLICATION_FACTOR_CONFIG, String.valueOf(1));
        props.put(DEFAULT_TOPIC_CREATION_PREFIX + PARTITIONS_CONFIG, String.valueOf(1));
        return props;
    }

    public static class EmptyTaskConfigsConnector extends SinkConnector {
        @Override
        public String version() {
            return "0.0";
        }

        @Override
        public void start(Map<String, String> props) {
            // no-op
        }

        @Override
        public Class<? extends Task> taskClass() {
            return SimpleTask.class;
        }

        @Override
        public List<Map<String, String>> taskConfigs(int maxTasks) {
            return IntStream.range(0, maxTasks)
                    .mapToObj(i -> Collections.<String, String>emptyMap())
                    .collect(Collectors.toList());
        }

        @Override
        public void stop() {
            // no-op
        }

        @Override
        public ConfigDef config() {
            return new ConfigDef();
        }
    }

    public static class SimpleTask extends SinkTask {
        @Override
        public String version() {
            return "0.0";
        }

        @Override
        public void start(Map<String, String> props) {
            // no-op
        }

        @Override
        public void put(Collection<SinkRecord> records) {
            // no-op
        }

        @Override
        public void stop() {
            // no-op
        }
    }
}
