/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.nar;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.nar.NarLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NarAutoLoaderTask
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NarAutoLoaderTask.class);
    private static final long MIN_FILE_AGE = 5000L;
    private final Path autoLoadPath;
    private final WatchService watchService;
    private final long pollIntervalMillis;
    private final NarLoader narLoader;
    private final List<File> candidateNars;
    private volatile boolean stopped = false;

    private NarAutoLoaderTask(Builder builder) {
        this.autoLoadPath = builder.autoLoadPath;
        this.watchService = builder.watchService;
        this.pollIntervalMillis = builder.pollIntervalMillis;
        this.narLoader = builder.narLoader;
        this.candidateNars = new ArrayList<File>();
    }

    @Override
    public void run() {
        while (!this.stopped) {
            try {
                WatchKey key;
                try {
                    LOGGER.debug("Polling for new NARs at {}", (Object)this.autoLoadPath);
                    key = this.watchService.poll(this.pollIntervalMillis, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException x) {
                    LOGGER.info("WatchService interrupted, returning...");
                    return;
                }
                if (key != null) {
                    for (WatchEvent<?> event : key.pollEvents()) {
                        WatchEvent.Kind<?> kind = event.kind();
                        if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                        WatchEvent<?> ev = event;
                        Path filename = (Path)ev.context();
                        Path autoLoadFile = this.autoLoadPath.resolve(filename);
                        String autoLoadFilename = autoLoadFile.toFile().getName().toLowerCase();
                        if (!autoLoadFilename.endsWith(".nar")) {
                            LOGGER.info("Skipping non-nar file {}", (Object)autoLoadFilename);
                            continue;
                        }
                        if (autoLoadFilename.startsWith(".")) {
                            LOGGER.debug("Skipping partially written file {}", (Object)autoLoadFilename);
                            continue;
                        }
                        LOGGER.info("Found {} in auto-load directory", (Object)autoLoadFile);
                        this.candidateNars.add(autoLoadFile.toFile());
                    }
                    boolean valid = key.reset();
                    if (!valid) {
                        LOGGER.error("NAR auto-load directory is no longer valid");
                        this.stop();
                    }
                }
                ArrayList<File> readyNars = new ArrayList<File>();
                Iterator<File> candidateNarIter = this.candidateNars.iterator();
                while (candidateNarIter.hasNext()) {
                    File candidateNar = candidateNarIter.next();
                    long fileAge = System.currentTimeMillis() - candidateNar.lastModified();
                    if (fileAge >= 5000L) {
                        readyNars.add(candidateNar);
                        candidateNarIter.remove();
                        continue;
                    }
                    LOGGER.debug("Candidate NAR {} not ready yet, will check again next time", (Object)candidateNar.getName());
                }
                if (readyNars.isEmpty()) continue;
                this.narLoader.load(readyNars);
            }
            catch (Throwable t) {
                LOGGER.error("Error loading NARs", t);
            }
        }
    }

    public void stop() {
        LOGGER.info("Stopping NAR Auto-loader");
        this.stopped = true;
    }

    public static class Builder {
        private Path autoLoadPath;
        private WatchService watchService;
        private long pollIntervalMillis;
        private NarLoader narLoader;

        public Builder autoLoadPath(Path autoLoadPath) {
            this.autoLoadPath = autoLoadPath;
            return this;
        }

        public Builder watchService(WatchService watchService) {
            this.watchService = watchService;
            return this;
        }

        public Builder pollIntervalMillis(long pollIntervalMillis) {
            this.pollIntervalMillis = pollIntervalMillis;
            return this;
        }

        public Builder narLoader(NarLoader narLoader) {
            this.narLoader = narLoader;
            return this;
        }

        public NarAutoLoaderTask build() {
            return new NarAutoLoaderTask(this);
        }
    }
}

