//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.simulator.options;

import static org.eclipse.escet.common.java.Strings.fmt;

import java.awt.Color;
import java.util.Locale;
import java.util.Map;

import org.eclipse.escet.common.app.framework.options.Options;
import org.eclipse.escet.common.app.framework.options.StringOption;
import org.eclipse.escet.common.java.Strings;
import org.eclipse.escet.common.java.exceptions.InvalidOptionException;
import org.eclipse.escet.common.svg.selector.SvgSelectorOutlineStyle;

/** SVG input interactive element outline style option. */
public class SvgInOutlineStyleOption extends StringOption {
    /** The default value of the {@link SvgInOutlineStyleOption}. */
    private static final String DEFAULT = "red:green:1.5";

    /** Mapping of names of colors to their colors. */
    private static final Map<String, Color> NAMED_COLORS = Map.ofEntries(
            Map.entry("black", Color.BLACK),
            Map.entry("blue", Color.BLUE),
            Map.entry("cyan", Color.CYAN),
            Map.entry("dark-gray", Color.DARK_GRAY),
            Map.entry("gray", Color.GRAY),
            Map.entry("green", Color.GREEN),
            Map.entry("light-gray", Color.LIGHT_GRAY),
            Map.entry("magenta", Color.MAGENTA),
            Map.entry("orange", Color.ORANGE),
            Map.entry("pink", Color.PINK),
            Map.entry("red", Color.RED),
            Map.entry("white", Color.WHITE),
            Map.entry("yellow", Color.YELLOW));

    /** The description of the {@link SvgInOutlineStyleOption}. */
    private static final String DESCRIPTION = fmt("The style to use for the outline of interactive elements of the SVG "
            + "input mode. Specify \"off\" to not show any outline. Specify a hover color, click color and stroke "
            + "width, separated by colons, to specify a specific style. For the colors, use: one of the named colors "
            + "%s; or comma-separated red, green and blue components, each in range [0..255]. For the stroke width, "
            + "specify the width in pixels as a positive floating point number.",
            Strings.makeElementsChoiceText(NAMED_COLORS.keySet().stream().sorted().toList(),
                    name -> "\"" + name + "\""));

    /** Constructor for the {@link SvgInOutlineStyleOption} class. */
    public SvgInOutlineStyleOption() {
        super(// name
                "SVG input outline style",

                // description
                DESCRIPTION + " [DEFAULT=\"" + DEFAULT + "\"]",

                // cmdShort
                null,

                // cmdLong
                "svgin-outline-style",

                // cmdValue
                "STYLE",

                // defaultValue
                DEFAULT,

                // emptyAsNull
                false,

                // showInDialog
                true,

                // optDialogDescr
                DESCRIPTION,

                // optDialogLabelText
                "Style:"

        );
    }

    /**
     * Returns the outline style as configured by the {@link SvgInOutlineStyleOption}.
     *
     * @return The outline style, or {@code null} to not paint any outlines.
     */
    public static SvgSelectorOutlineStyle getStyle() {
        // Get text.
        String text = Options.get(SvgInOutlineStyleOption.class);
        String originalText = text;
        text = text.strip().toLowerCase(Locale.US);

        // Handle 'off'.
        if (text.equals("off")) {
            return null;
        }

        // Parser the style.
        String[] parts = text.split(":");
        if (parts.length != 3) {
            throw new InvalidOptionException(fmt("Invalid SVG input outline style \"%s\": expected 3 values separated "
                    + "by colons, but got %d.", originalText, parts.length));
        }

        Color hoverColor = parseColor(parts[0]);
        Color clickColor = parseColor(parts[1]);
        float strokeWidth = parseWidth(parts[2]);

        return new SvgSelectorOutlineStyle(hoverColor, clickColor, strokeWidth);
    }

    /**
     * Parse a color.
     *
     * @param text The color text.
     * @return The color.
     */
    private static Color parseColor(String text) {
        String originalText = text;
        text = text.strip().toLowerCase(Locale.US);

        // Try to parse as named color.
        Color color = NAMED_COLORS.get(text);
        if (color != null) {
            return color;
        }

        // Parse as RGB color.
        String[] parts = text.split(",");
        if (parts.length != 3) {
            throw new InvalidOptionException(fmt("Invalid SVG input outline style color \"%s\": expected a known named "
                    + "color or 3 color components.", originalText));
        }

        int red = parseColorComponent(parts[0], "red");
        int green = parseColorComponent(parts[1], "green");
        int blue = parseColorComponent(parts[2], "blue");
        return new Color(red, green, blue);
    }

    /**
     * Parse an RGB color component.
     *
     * @param text The text of the component.
     * @param name The name of the component.
     * @return The parsed component value.
     */
    private static int parseColorComponent(String text, String name) {
        int value;
        try {
            value = Integer.parseInt(text);
            if (value < 0 || value > 255) {
                throw new NumberFormatException("Number outside of the [0..255] range.");
            }
        } catch (NumberFormatException e) {
            throw new InvalidOptionException(
                    fmt("Invalid SVG input outline style color %s component \"%s\": invalid number.", name, text), e);
        }
        return value;
    }

    /**
     * Parse a width.
     *
     * @param text The width text.
     * @return The width.
     */
    private static float parseWidth(String text) {
        float width;
        try {
            width = Float.parseFloat(text.strip());
            if (Float.isNaN(width)) {
                throw new NumberFormatException("Width is not a number.");
            }
            if (Float.isInfinite(width)) {
                throw new NumberFormatException("Width is infinite.");
            }
            if (width <= 0) {
                throw new NumberFormatException("Width is not positive.");
            }
        } catch (NumberFormatException e) {
            throw new InvalidOptionException(fmt("Invalid SVG input outline style width \"%s\".", text), e);
        }
        return width;
    }
}
