/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.builder;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.builder.OperatorBuilder;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.utils.AspectToEcore;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Builder for the {@link Weave} operator.
 */
@SuppressWarnings("all")
public class WeaveBuilder extends OperatorBuilder<Weave> {
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;
  
  @Inject
  @Extension
  private AspectToEcore _aspectToEcore;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  /**
   * The {@link EPackage}s on which the aspect pointed by the current
   * {@link Weave} operator should be woven.
   */
  private Set<EPackage> baseModel;
  
  public WeaveBuilder(final Weave op, final Set<EPackage> baseModel) {
    super(op);
    this.baseModel = baseModel;
  }
  
  /**
   * Process the current {@link Weave} operator and build the fragment
   * of {@link EPackage} corresponding to it.
   */
  @Override
  public void make() {
    final JvmTypeReference aspRef = this.source.getAspectTypeRef();
    JvmType _type = null;
    if (aspRef!=null) {
      _type=aspRef.getType();
    }
    if ((_type instanceof JvmDeclaredType)) {
      final String aspectedClass = this._aspectExtensions.getAspectAnnotationValue(aspRef);
      EClass _xifexpression = null;
      if ((aspectedClass != null)) {
        _xifexpression = this._ecoreExtensions.findClass(this.baseModel, aspectedClass);
      }
      EClass baseClass = _xifexpression;
      if (((baseClass == null) && (aspectedClass != null))) {
        final List<PackageBinding> renamings = this._languageExtensions.collectMappings(this.source.getOwningLanguage());
        final String newName = this._languageExtensions.rename(aspectedClass, renamings);
        baseClass = this._ecoreExtensions.findClass(this.baseModel, newName);
      }
      JvmType _type_1 = aspRef.getType();
      final JvmDeclaredType aspect = ((JvmDeclaredType) _type_1);
      this.model = CollectionLiterals.<EPackage>newHashSet(this._aspectToEcore.inferEcoreFragment(aspect, baseClass, this.baseModel));
      this.alignInferredClassifiers(this.baseModel, this.model);
    }
  }
  
  /**
   * Because aspect inference happens sequentially, one may infer
   * a concept as a datatype and the other as a plain class.
   * Thus, we need to align the inferred types, classes taking priority
   * over datatypes.
   */
  private void alignInferredClassifiers(final Set<EPackage> base, final Set<EPackage> fragment) {
    final Function1<EPackage, EList<EClassifier>> _function = new Function1<EPackage, EList<EClassifier>>() {
      @Override
      public EList<EClassifier> apply(final EPackage it) {
        return it.getEClassifiers();
      }
    };
    final Function1<EDataType, Boolean> _function_1 = new Function1<EDataType, Boolean>() {
      @Override
      public Boolean apply(final EDataType dt) {
        final Function1<EPackage, EList<EClassifier>> _function = new Function1<EPackage, EList<EClassifier>>() {
          @Override
          public EList<EClassifier> apply(final EPackage it) {
            return it.getEClassifiers();
          }
        };
        final Function1<EClass, Boolean> _function_1 = new Function1<EClass, Boolean>() {
          @Override
          public Boolean apply(final EClass cls) {
            String _name = cls.getName();
            String _name_1 = dt.getName();
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          }
        };
        return Boolean.valueOf(IterableExtensions.<EClass>exists(Iterables.<EClass>filter(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(fragment, _function)), EClass.class), _function_1));
      }
    };
    final EDataType find = IterableExtensions.<EDataType>findFirst(Iterables.<EDataType>filter(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(fragment, _function)), EDataType.class), _function_1);
    if ((find != null)) {
      this._ecoreExtensions.replaceDataTypeWithEClass(fragment, find);
    }
    final Function1<EPackage, EList<EClassifier>> _function_2 = new Function1<EPackage, EList<EClassifier>>() {
      @Override
      public EList<EClassifier> apply(final EPackage it) {
        return it.getEClassifiers();
      }
    };
    final Consumer<EClassifier> _function_3 = new Consumer<EClassifier>() {
      @Override
      public void accept(final EClassifier cls1) {
        final Function1<EPackage, EList<EClassifier>> _function = new Function1<EPackage, EList<EClassifier>>() {
          @Override
          public EList<EClassifier> apply(final EPackage it) {
            return it.getEClassifiers();
          }
        };
        final Function1<EClassifier, Boolean> _function_1 = new Function1<EClassifier, Boolean>() {
          @Override
          public Boolean apply(final EClassifier cls2) {
            return Boolean.valueOf((Objects.equal(cls2.getName(), cls1.getName()) && (cls2.eClass() != cls1.eClass())));
          }
        };
        final EClassifier cls2 = IterableExtensions.<EClassifier>findFirst(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(base, _function)), _function_1);
        if ((cls2 != null)) {
          if ((cls2 instanceof EClass)) {
            if ((cls1 instanceof EDataType)) {
              WeaveBuilder.this._ecoreExtensions.replaceDataTypeWithEClass(fragment, ((EDataType)cls1));
            }
          } else {
            if ((cls1 instanceof EClass)) {
              if ((cls2 instanceof EDataType)) {
                WeaveBuilder.this._ecoreExtensions.replaceDataTypeWithEClass(base, ((EDataType)cls2));
              }
            }
          }
        }
      }
    };
    Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(fragment, _function_2)).forEach(_function_3);
  }
  
  public Weave getSource() {
    return this.source;
  }
}
