/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.gemoc.generator.util;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.codegen.ecore.generator.Generator;
import org.eclipse.emf.codegen.ecore.genmodel.GenJDKLevel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelFactory;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.codegen.ecore.genmodel.GenParameter;
import org.eclipse.emf.codegen.ecore.genmodel.generator.GenBaseGeneratorAdapter;
import org.eclipse.emf.codegen.ecore.genmodel.util.GenModelUtil;
import org.eclipse.emf.codegen.util.CodeGenUtil;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.Monitor;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gemoc.trace.gemoc.generator.util.PluginProjectHelper;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

@SuppressWarnings("all")
public class StandaloneEMFProjectGenerator {
  protected final EPackage ecoreModel;
  
  protected final String projectName;
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected IProject project;
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected final Set<GenPackage> referencedGenPackages = new HashSet<GenPackage>();
  
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  protected final Set<EPackage> rootPackages = new HashSet<EPackage>();
  
  protected GenModel genModel;
  
  /**
   * Helper method to generate code without a job.
   */
  public void generateModelCode() {
    try {
      IWorkbench _workbench = PlatformUI.getWorkbench();
      IWorkbenchWindow _activeWorkbenchWindow = _workbench.getActiveWorkbenchWindow();
      final IRunnableWithProgress _function = new IRunnableWithProgress() {
        @Override
        public void run(final IProgressMonitor m) throws InvocationTargetException, InterruptedException {
          try {
            StandaloneEMFProjectGenerator.this.generateModelCode(m);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      };
      _activeWorkbenchWindow.run(false, true, _function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public final static String MODEL_GEN_FOLDER = "model";
  
  protected IProgressMonitor progressMonitor;
  
  protected ResourceSet resourceSet;
  
  protected URI modelGenFolderURI;
  
  protected String srcFolderPathString;
  
  protected Resource ecoreModelResource;
  
  public StandaloneEMFProjectGenerator(final String projectName, final EPackage p) {
    this.ecoreModel = p;
    this.projectName = projectName;
  }
  
  /**
   * Creates a new EMF project with the ecore file and the genmodel in the "model" folder
   * also mages project, referencedGenPackages and rootPackages available.
   */
  public void generateBaseEMFProject(final IProgressMonitor m) {
    try {
      this.progressMonitor = m;
      ResourceSetImpl _resourceSetImpl = new ResourceSetImpl();
      this.resourceSet = _resourceSetImpl;
      IProject _createPluginProject = PluginProjectHelper.createPluginProject(this.projectName, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("src")), 
        Collections.<IProject>unmodifiableList(CollectionLiterals.<IProject>newArrayList()), 
        Collections.<String>unmodifiableSet(CollectionLiterals.<String>newHashSet()), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), m);
      this.project = _createPluginProject;
      URI _setupModelGenFolder = this.setupModelGenFolder();
      this.modelGenFolderURI = _setupModelGenFolder;
      String _setupSrcFolder = this.setupSrcFolder();
      this.srcFolderPathString = _setupSrcFolder;
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(this.projectName, "");
      _builder.append("/");
      _builder.append(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER, "");
      _builder.append("/");
      String _name = this.ecoreModel.getName();
      _builder.append(_name, "");
      _builder.append(".ecore");
      URI _createPlatformResourceURI = URI.createPlatformResourceURI(_builder.toString(), true);
      Resource _createResource = this.resourceSet.createResource(_createPlatformResourceURI);
      this.ecoreModelResource = _createResource;
      EList<EObject> _contents = this.ecoreModelResource.getContents();
      _contents.add(this.ecoreModel);
      this.save(this.ecoreModelResource);
      this.ecoreModelResource.unload();
      this.ecoreModelResource.load(null);
      this.ecoreModelResource.unload();
      this.ecoreModelResource.load(null);
      this.checkReferencedPackages(this.ecoreModelResource);
      EList<EObject> _contents_1 = this.ecoreModelResource.getContents();
      EObject _get = _contents_1.get(0);
      GenModel _generateGenModel = this.generateGenModel(((EPackage) _get), this.modelGenFolderURI);
      this.genModel = _generateGenModel;
      EList<EObject> _contents_2 = this.ecoreModelResource.getContents();
      Iterable<EPackage> _filter = Iterables.<EPackage>filter(_contents_2, EPackage.class);
      Set<EPackage> _set = IterableExtensions.<EPackage>toSet(_filter);
      this.rootPackages.addAll(_set);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Generates the code using the genmodel (within a Job).
   */
  public void generateModelCode(final IProgressMonitor m) throws Exception {
    this.generateCode(this.progressMonitor);
  }
  
  private URI setupModelGenFolder() {
    URI modelGenFolderURI = null;
    final IFolder modelGenFolder = this.project.getFolder(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER);
    boolean _exists = modelGenFolder.exists();
    boolean _not = (!_exists);
    if (_not) {
      try {
        modelGenFolder.create(true, true, null);
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException e = (CoreException)_t;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("The folder \'");
          _builder.append(StandaloneEMFProjectGenerator.MODEL_GEN_FOLDER, "");
          _builder.append("\' could not be created.");
          throw new RuntimeException(_builder.toString(), e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    IPath _fullPath = modelGenFolder.getFullPath();
    String _string = _fullPath.toString();
    URI _createPlatformResourceURI = URI.createPlatformResourceURI(_string, true);
    modelGenFolderURI = _createPlatformResourceURI;
    return modelGenFolderURI;
  }
  
  private String setupSrcFolder() {
    String srcFolderPathString = null;
    final IFolder srcFolder = this.project.getFolder("src");
    boolean _exists = srcFolder.exists();
    boolean _not = (!_exists);
    if (_not) {
      try {
        srcFolder.create(true, true, null);
      } catch (final Throwable _t) {
        if (_t instanceof CoreException) {
          final CoreException e = (CoreException)_t;
          throw new RuntimeException("The source folder \'src\' could not be created.", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    IPath _fullPath = srcFolder.getFullPath();
    String _string = _fullPath.toString();
    srcFolderPathString = _string;
    return srcFolderPathString;
  }
  
  private void checkReferencedPackages(final Resource xmofModelResource) {
    final Set<URI> missingPackages = new HashSet<URI>();
    final Map<EObject, Collection<EStructuralFeature.Setting>> externalCrossReferences = EcoreUtil.ExternalCrossReferencer.find(xmofModelResource);
    Set<EObject> _keySet = externalCrossReferences.keySet();
    for (final EObject eObject : _keySet) {
      boolean _eIsProxy = eObject.eIsProxy();
      if (_eIsProxy) {
        URI _uRI = EcoreUtil.getURI(eObject);
        URI _trimFragment = _uRI.trimFragment();
        missingPackages.add(_trimFragment);
      }
    }
    int _size = missingPackages.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      String _xifexpression = null;
      int _size_1 = missingPackages.size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        _xifexpression = "";
      } else {
        _xifexpression = "s";
      }
      String _plus = ("Unable to load the following referenced resource" + _xifexpression);
      String _plus_1 = (_plus + ": ");
      String _string = missingPackages.toString();
      final String message = (_plus_1 + _string);
      throw new RuntimeException(message);
    }
  }
  
  protected GenModel generateGenModel(final EPackage rootEPackage, final URI modelGenFolderURI) {
    final Resource genModelResource = this.createGenModel(rootEPackage);
    EList<EObject> _contents = genModelResource.getContents();
    EObject _get = _contents.get(0);
    final GenModel genModel = ((GenModel) _get);
    this.setInitializeByLoad(genModel);
    this.save(genModelResource);
    return genModel;
  }
  
  protected Resource createGenModel(final EPackage rootEPackage) {
    final Resource ecoreModelResource = rootEPackage.eResource();
    URI _uRI = ecoreModelResource.getURI();
    URI _trimFileExtension = _uRI.trimFileExtension();
    URI _appendFileExtension = _trimFileExtension.appendFileExtension("genmodel");
    String _lastSegment = _appendFileExtension.lastSegment();
    final String genModelFileName = _lastSegment.toString();
    final URI genModelURI = this.modelGenFolderURI.appendSegment(genModelFileName);
    final Resource genModelResource = this.resourceSet.createResource(genModelURI);
    final GenModel genModel = GenModelFactory.eINSTANCE.createGenModel();
    EList<EObject> _contents = genModelResource.getContents();
    _contents.add(genModel);
    final IFolder srcFolder = this.project.getFolder("src");
    IPath _fullPath = srcFolder.getFullPath();
    String _string = _fullPath.toString();
    genModel.setModelDirectory(_string);
    EList<String> _foreignModel = genModel.getForeignModel();
    URI _uRI_1 = ecoreModelResource.getURI();
    String _string_1 = _uRI_1.toString();
    _foreignModel.add(_string_1);
    String _modelName = this.getModelName(genModelURI);
    genModel.setModelName(_modelName);
    String _pluginID = this.getPluginID(genModelURI);
    genModel.setModelPluginID(_pluginID);
    genModel.setRootExtendsClass("org.eclipse.emf.ecore.impl.MinimalEObjectImpl$Container");
    GenJDKLevel _complicanceLevel = this.getComplicanceLevel();
    genModel.setComplianceLevel(_complicanceLevel);
    genModel.setImportOrganizing(true);
    Set<EPackage> _singleton = Collections.<EPackage>singleton(rootEPackage);
    genModel.initialize(_singleton);
    this.setMissingParameterTypes(genModel);
    this.fixUsedGenPackages(genModel);
    return genModelResource;
  }
  
  private Set<GenModel> fixedGenModels = new HashSet<GenModel>();
  
  /**
   * Tries to fix the "usedGenPackages" collection of a genmodel (and recursively of all genmodels it references)
   * 1) remove all usedGenPackages that have a null genModel (for a mysterious reason...)
   * 2) use the magical method 'computeMissingGenPackages' to find missing packages, and add them to usedGenPackages
   * 3) as a bonus, store all referenced gen packages in 'referencedGenPackages' for later use
   */
  private void fixUsedGenPackages(final GenModel genModel) {
    boolean _contains = this.fixedGenModels.contains(genModel);
    boolean _not = (!_contains);
    if (_not) {
      this.fixedGenModels.add(genModel);
      EList<GenPackage> _usedGenPackages = genModel.getUsedGenPackages();
      EList<GenPackage> _usedGenPackages_1 = genModel.getUsedGenPackages();
      List<GenPackage> _immutableCopy = ImmutableList.<GenPackage>copyOf(_usedGenPackages_1);
      final Function1<GenPackage, Boolean> _function = new Function1<GenPackage, Boolean>() {
        @Override
        public Boolean apply(final GenPackage p) {
          GenModel _genModel = p.getGenModel();
          return Boolean.valueOf(Objects.equal(_genModel, null));
        }
      };
      Iterable<GenPackage> _filter = IterableExtensions.<GenPackage>filter(_immutableCopy, _function);
      CollectionExtensions.<GenPackage>removeAll(_usedGenPackages, _filter);
      final List<GenPackage> missingGenPackages = this.computeMissingGenPackages(genModel);
      for (final GenPackage genPackage : missingGenPackages) {
        GenModel _genModel = genPackage.getGenModel();
        this.fixUsedGenPackages(_genModel);
      }
      this.referencedGenPackages.addAll(missingGenPackages);
      EList<GenPackage> _genPackages = genModel.getGenPackages();
      this.referencedGenPackages.addAll(_genPackages);
      EList<GenPackage> _usedGenPackages_2 = genModel.getUsedGenPackages();
      _usedGenPackages_2.addAll(missingGenPackages);
    }
  }
  
  protected String getModelName(final URI genModelURI) {
    URI _trimFileExtension = genModelURI.trimFileExtension();
    final String genModelFileName = _trimFileExtension.lastSegment();
    String _substring = genModelFileName.substring(0, 1);
    String _upperCase = _substring.toUpperCase();
    String _substring_1 = genModelFileName.substring(1);
    final String modelName = (_upperCase + _substring_1);
    return modelName;
  }
  
  protected String getPluginID(final URI uri) {
    String pluginID = "";
    IFolder _folder = this.project.getFolder("META-INF");
    final IFile manifestFile = _folder.getFile("MANIFEST.MF");
    try {
      InputStream _contents = manifestFile.getContents();
      final Manifest manifest = new Manifest(_contents);
      Attributes _mainAttributes = manifest.getMainAttributes();
      String symbolicName = _mainAttributes.getValue("Bundle-SymbolicName");
      boolean _notEquals = (!Objects.equal(symbolicName, null));
      if (_notEquals) {
        final int index = symbolicName.indexOf(";");
        if ((index > 0)) {
          String _substring = symbolicName.substring(0, index);
          symbolicName = _substring;
        }
        String _trim = symbolicName.trim();
        pluginID = _trim;
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        IPath _fullPath = manifestFile.getFullPath();
        String _string = _fullPath.toString();
        String _plus = ("Could not find manifest file \'" + _string);
        String _plus_1 = (_plus + "\'.");
        throw new RuntimeException(_plus_1, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return pluginID;
  }
  
  private GenJDKLevel getComplicanceLevel() {
    final String complianceLevel = CodeGenUtil.EclipseUtil.getJavaComplianceLevel(this.project);
    boolean _equals = "1.4".equals(complianceLevel);
    if (_equals) {
      return GenJDKLevel.JDK14_LITERAL;
    } else {
      boolean _equals_1 = "1.5".equals(complianceLevel);
      if (_equals_1) {
        return GenJDKLevel.JDK50_LITERAL;
      } else {
        boolean _equals_2 = "1.6".equals(complianceLevel);
        if (_equals_2) {
          return GenJDKLevel.JDK60_LITERAL;
        } else {
          boolean _equals_3 = "1.7".equals(complianceLevel);
          if (_equals_3) {
            return GenJDKLevel.JDK70_LITERAL;
          } else {
            return GenJDKLevel.JDK80_LITERAL;
          }
        }
      }
    }
  }
  
  /**
   * In case of missing parameter types, the types are temporarily set to
   * EObject
   * 
   * @param genModel
   */
  private void setMissingParameterTypes(final GenModel genModel) {
    TreeIterator<EObject> _eAllContents = genModel.eAllContents();
    Iterator<GenParameter> _filter = Iterators.<GenParameter>filter(_eAllContents, GenParameter.class);
    Set<GenParameter> _set = IteratorExtensions.<GenParameter>toSet(_filter);
    for (final GenParameter genModelElement : _set) {
      {
        final GenParameter genParameter = ((GenParameter) genModelElement);
        final EParameter ecoreParameter = genParameter.getEcoreParameter();
        EClassifier _eType = ecoreParameter.getEType();
        boolean _equals = Objects.equal(_eType, null);
        if (_equals) {
          EClass _eObject = EcorePackage.eINSTANCE.getEObject();
          ecoreParameter.setEType(_eObject);
        }
      }
    }
  }
  
  protected List<GenPackage> computeMissingGenPackages(final GenModel genModel) {
    final List<GenPackage> missingGenPackages = new UniqueEList<GenPackage>();
    final Map<String, URI> genModelLocationMapTargetEnvironment = EcorePlugin.getEPackageNsURIToGenModelLocationMap(true);
    final Map<String, URI> genModelLocationMapEnvironment = EcorePlugin.getEPackageNsURIToGenModelLocationMap(false);
    List<EPackage> _missingPackages = genModel.getMissingPackages();
    for (final EPackage ePackage : _missingPackages) {
      boolean _notEquals = (!Objects.equal(ePackage, null));
      if (_notEquals) {
        String _nsURI = ePackage.getNsURI();
        URI missingGenModelURI = genModelLocationMapEnvironment.get(_nsURI);
        boolean _equals = Objects.equal(missingGenModelURI, null);
        if (_equals) {
          String _nsURI_1 = ePackage.getNsURI();
          URI _get = genModelLocationMapTargetEnvironment.get(_nsURI_1);
          missingGenModelURI = _get;
        }
        boolean _equals_1 = Objects.equal(missingGenModelURI, null);
        if (_equals_1) {
          String _nsURI_2 = ePackage.getNsURI();
          String _plus = ("Unable to load generator model of required package \'" + _nsURI_2);
          String _plus_1 = (_plus + "\'.");
          throw new RuntimeException(_plus_1);
        }
        Resource missingGenModelResource = null;
        try {
          Resource _resource = this.resourceSet.getResource(missingGenModelURI, true);
          missingGenModelResource = _resource;
        } catch (final Throwable _t) {
          if (_t instanceof RuntimeException) {
            final RuntimeException e = (RuntimeException)_t;
            String _nsURI_3 = ePackage.getNsURI();
            String _plus_2 = ("Unable to load generator model of required package \'" + _nsURI_3);
            String _plus_3 = (_plus_2 + "\'.");
            throw new RuntimeException(_plus_3);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        EList<EObject> _contents = missingGenModelResource.getContents();
        EObject _get_1 = _contents.get(0);
        final GenModel missingGenModel = ((GenModel) _get_1);
        EList<GenPackage> _genPackages = missingGenModel.getGenPackages();
        missingGenPackages.addAll(_genPackages);
      }
    }
    return missingGenPackages;
  }
  
  protected void setInitializeByLoad(final GenModel genModel) {
    EList<GenPackage> _genPackages = genModel.getGenPackages();
    for (final GenPackage genPackage : _genPackages) {
      this.setInitializeByLoad(genPackage);
    }
  }
  
  private void setInitializeByLoad(final GenPackage genPackage) {
    genPackage.setLoadInitialization(false);
    List<GenPackage> _subGenPackages = genPackage.getSubGenPackages();
    for (final GenPackage subGenPackage : _subGenPackages) {
      this.setInitializeByLoad(subGenPackage);
    }
  }
  
  private List<Diagnostic> expandDiagnostics(final Diagnostic d) {
    final ArrayList<Diagnostic> result = new ArrayList<Diagnostic>();
    result.add(d);
    List<Diagnostic> _children = d.getChildren();
    result.addAll(_children);
    List<Diagnostic> _children_1 = d.getChildren();
    for (final Diagnostic c : _children_1) {
      List<Diagnostic> _expandDiagnostics = this.expandDiagnostics(c);
      result.addAll(_expandDiagnostics);
    }
    return result;
  }
  
  private String exceptionToStackString(final Throwable e) {
    final StringWriter sw = new StringWriter();
    PrintWriter _printWriter = new PrintWriter(sw);
    e.printStackTrace(_printWriter);
    final String exceptionAsString = sw.toString();
    return exceptionAsString;
  }
  
  private String diagnosticErrorsToString(final Diagnostic diagnostic) {
    List<Diagnostic> _expandDiagnostics = this.expandDiagnostics(diagnostic);
    final Function1<Diagnostic, Boolean> _function = new Function1<Diagnostic, Boolean>() {
      @Override
      public Boolean apply(final Diagnostic d) {
        int _severity = d.getSeverity();
        return Boolean.valueOf((_severity == Diagnostic.ERROR));
      }
    };
    Iterable<Diagnostic> _filter = IterableExtensions.<Diagnostic>filter(_expandDiagnostics, _function);
    final Set<Diagnostic> errors = IterableExtensions.<Diagnostic>toSet(_filter);
    final Function1<Diagnostic, Boolean> _function_1 = new Function1<Diagnostic, Boolean>() {
      @Override
      public Boolean apply(final Diagnostic e) {
        Throwable _exception = e.getException();
        return Boolean.valueOf((!Objects.equal(_exception, null)));
      }
    };
    Iterable<Diagnostic> _filter_1 = IterableExtensions.<Diagnostic>filter(errors, _function_1);
    final Function1<Diagnostic, String> _function_2 = new Function1<Diagnostic, String>() {
      @Override
      public String apply(final Diagnostic e) {
        Throwable _exception = e.getException();
        return StandaloneEMFProjectGenerator.this.exceptionToStackString(_exception);
      }
    };
    Iterable<String> _map = IterableExtensions.<Diagnostic, String>map(_filter_1, _function_2);
    final Set<String> exceptions = IterableExtensions.<String>toSet(_map);
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _hasElements = false;
      for(final String e : exceptions) {
        if (!_hasElements) {
          _hasElements = true;
          _builder.append("Encountered exceptions:\n", "");
        } else {
          _builder.appendImmediate("\n", "");
        }
        _builder.append("- ");
        _builder.append(e, "");
        _builder.newLineIfNotEmpty();
      }
      if (_hasElements) {
        _builder.append("\n", "");
      }
    }
    {
      boolean _hasElements_1 = false;
      for(final Diagnostic e_1 : errors) {
        if (!_hasElements_1) {
          _hasElements_1 = true;
          _builder.append("Encountered diagnostic errors:\n", "");
        } else {
          _builder.appendImmediate("\n", "");
        }
        _builder.append("- ");
        String _message = e_1.getMessage();
        _builder.append(_message, "");
        _builder.append(" ");
        _builder.append("\t\t\t");
      }
      if (_hasElements_1) {
        _builder.append("\n", "");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
  
  protected void generateCode(final IProgressMonitor progressMonitor) throws Exception {
    boolean success = false;
    this.prepareGenModelForCodeGeneration(this.genModel);
    final Generator generator = GenModelUtil.createGenerator(this.genModel);
    final boolean canGenerate = generator.canGenerate(this.genModel, GenBaseGeneratorAdapter.MODEL_PROJECT_TYPE);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Cannot generate code of EPackage ");
    String _name = this.ecoreModel.getName();
    _builder.append(_name, "");
    String message = _builder.toString();
    if (canGenerate) {
      Diagnostic diagnostic = null;
      String otherMessage = "";
      try {
        Monitor _monitor = BasicMonitor.toMonitor(progressMonitor);
        Diagnostic _generate = generator.generate(this.genModel, GenBaseGeneratorAdapter.MODEL_PROJECT_TYPE, _monitor);
        diagnostic = _generate;
      } catch (final Throwable _t) {
        if (_t instanceof Throwable) {
          final Throwable t = (Throwable)_t;
          String _exceptionToStackString = this.exceptionToStackString(t);
          otherMessage = _exceptionToStackString;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      if (((!Objects.equal(diagnostic, null)) && (diagnostic.getSeverity() == Diagnostic.OK))) {
        success = true;
      } else {
        boolean _notEquals = (!Objects.equal(diagnostic, null));
        if (_notEquals) {
          String _message = message;
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append(": ");
          String _diagnosticErrorsToString = this.diagnosticErrorsToString(diagnostic);
          _builder_1.append(_diagnosticErrorsToString, "");
          _builder_1.append(".");
          message = (_message + _builder_1);
        } else {
          String _message_1 = message;
          message = (_message_1 + otherMessage);
        }
      }
    } else {
      String _message_2 = message;
      message = (_message_2 + "generator.canGenerate returns false.");
    }
    if ((!success)) {
      throw new Exception(message);
    }
  }
  
  protected void prepareGenModelForCodeGeneration(final GenModel genModel) {
    genModel.reconcile();
    genModel.setCanGenerate(true);
  }
  
  private void save(final Resource resource) {
    try {
      resource.save(Collections.EMPTY_MAP);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        URI _uRI = resource.getURI();
        String _plus = ("Could not save resource \'" + _uRI);
        String _plus_1 = (_plus + "\'.");
        throw new RuntimeException(_plus_1, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Pure
  public IProject getProject() {
    return this.project;
  }
  
  protected void setProject(final IProject project) {
    this.project = project;
  }
  
  @Pure
  public Set<GenPackage> getReferencedGenPackages() {
    return this.referencedGenPackages;
  }
  
  @Pure
  public Set<EPackage> getRootPackages() {
    return this.rootPackages;
  }
}
