/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package distributions.math;

import java.util.Arrays;

import org.apache.commons.math3.exception.NotStrictlyPositiveException;
import org.apache.commons.math3.random.RandomGenerator;

public class PoissonModeDistribution extends org.apache.commons.math3.distribution.PoissonDistribution
        implements ModeDistribution
{
    private static final long serialVersionUID = 1L;

    private final Double itsDefault;

    public PoissonModeDistribution(RandomGenerator rng, double p, Double epsilon, Double _default)
            throws NotStrictlyPositiveException
    {
        super(rng, p,
                epsilon == null ? org.apache.commons.math3.distribution.PoissonDistribution.DEFAULT_EPSILON : epsilon,
                org.apache.commons.math3.distribution.PoissonDistribution.DEFAULT_MAX_ITERATIONS);
        itsDefault = _default;
    }

    /**
     * Mode equals numerical mean for a gamma distribution
     *
     * @see distributions.math.ModeDistribution#getMode()
     */
    @Override
    public double getMode() {
        return getNumericalMean();
    }

    @Override
    public double getDefault() {
        if (null != itsDefault) {
            return itsDefault;
        }
        return getNumericalMean();
    }

    /**
     * An attempt to implement a RealDistribution.density using an IntegerDistribution.
     */
    @Override
    public double density(double x) {
        // Check if x is (very close to) an integer
        if (Math.abs(x - Math.round(x)) < 1e-10) {
            // It's effectively an integer, return the probability mass
            return probability(Math.round((int)x));
        } else {
            // For non-integer points in a discrete distribution, density is 0
            return 0.0;
        }
    }

    @Override
    public double getLowerBound() {
        return getSupportLowerBound();
    }

    @Override
    public double getUpperBound() {
        return getSupportUpperBound();
    }

    @Override
    public double nextSample() {
        return sample();
    }

    @Override
    public double[] nextSamples(int sampleSize) {
        return Arrays.stream(super.sample(sampleSize)).mapToDouble(i -> i).toArray();
    }
}
