/*******************************************************************************
 * Copyright (c) 2013 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.compare.uml2.internal.postprocessor.util;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import java.util.Collections;
import java.util.Iterator;

import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Extension;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Some utility methods that may be tweaked to allow EMFCompare to scale.
 * 
 * @author <a href="mailto:mikael.barbero@obeo.fr">Mikael Barbero</a>
 */
public final class UMLCompareUtil {

	/** Constructor. */
	private UMLCompareUtil() {
	}

	/**
	 * Retrieves the base element for the specified stereotype application, i.e. the element to which the
	 * stereotype is applied.
	 * <p>
	 * It first calls {@link UMLUtil#getBaseElement(EObject)}. If it returns null, it then tries to find a
	 * {@link EStructuralFeature} with a name starting with {@link Extension#METACLASS_ROLE_PREFIX}. It
	 * <em>does not</em> verify if the the given {@code stereotypeApplication}'s eClass is defined as a
	 * Stereotype within a Profile because it may lead to load the resource of the Profile.
	 * 
	 * @param stereotypeApplication
	 *            The stereotype application.
	 * @return The base element.
	 */
	public static Element getBaseElement(EObject stereotypeApplication) {
		if (stereotypeApplication == null) {
			return null;
		}

		Element baseElement = UMLUtil.getBaseElement(stereotypeApplication);
		final Iterator<EStructuralFeature> features = stereotypeApplication.eClass()
				.getEAllStructuralFeatures().iterator();
		while (features.hasNext() && baseElement == null) {
			final EStructuralFeature feature = features.next();
			if (feature.getName().startsWith(Extension.METACLASS_ROLE_PREFIX)) {
				final Object value = stereotypeApplication.eGet(feature);
				if (value instanceof Element) {
					baseElement = (Element)value;
				}
			}
		}

		return baseElement;
	}

	/**
	 * From the given EReference, it returns the list of EReference which are superset and non union.
	 * 
	 * @param reference
	 *            The EReference subset from which is requested the non union superset.
	 * @return The list of EReference non union superset.
	 */
	public static Iterable<EReference> getNonUnionSupersetReferences(EReference reference) {
		return Iterables.filter(getSupersetReferences(reference), isNonUnionReference());
	}

	private static Predicate<? super EReference> isNonUnionReference() {
		return new Predicate<EReference>() {
			public boolean apply(EReference input) {
				return input != null
						&& !Iterables.any(input.getEAnnotations(), UMLUtilForCompare.isUnionAnnotation());
			}
		};
	}

	/**
	 * From the given EReference, it returns the list of EReference which are superset.
	 * 
	 * @param reference
	 *            The EReference subset from which is requested the superset.
	 * @return The list of EReference superset.
	 */
	private static Iterable<EReference> getSupersetReferences(EReference reference) {
		EAnnotation subsetsAnnotation = Iterables.find(reference.getEAnnotations(), UMLUtilForCompare
				.isSubsetsAnnotation(), null);
		if (subsetsAnnotation != null) {
			return Iterables.filter(subsetsAnnotation.getReferences(), EReference.class);
		}
		return Collections.emptyList();
	}

	/**
	 * This extends UMLUtil to get the name of the used annotations for subsets and unions.
	 * 
	 * @author <a href="mailto:cedric.notot@obeo.fr">Cedric Notot</a>
	 */
	private static class UMLUtilForCompare extends UMLUtil {
		public static Predicate<? super EAnnotation> isSubsetsAnnotation() {
			return new Predicate<EAnnotation>() {
				public boolean apply(EAnnotation input) {
					return input != null && input.getSource().equals(ANNOTATION__SUBSETS);
				}
			};
		}

		public static Predicate<? super EAnnotation> isUnionAnnotation() {
			return new Predicate<EAnnotation>() {
				public boolean apply(EAnnotation input) {
					return input != null && input.getSource().equals(ANNOTATION__UNION);
				}
			};
		}
	}

}
