/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _HARDWARE_GPS_NI_H
#define _HARDWARE_GPS_NI_H

#include <stdint.h>

#if __cplusplus
extern "C" {
#endif

/**
 * Name for NI interface
 */
#define GPS_NI_INTERFACE "gps-ni"

/**
 * String length constants
 */
#define GPS_NI_SHORT_STRING_MAXLEN      256
#define GPS_NI_LONG_STRING_MAXLEN       2048

/**
 * GpsNiType constants
 */
typedef uint32_t GpsNiType;
#define GPS_NI_TYPE_VOICE              1
#define GPS_NI_TYPE_UMTS_SUPL          2
#define GPS_NI_TYPE_UMTS_CTRL_PLANE    3

/**
 * GpsNiNotifyFlags constants
 */
typedef uint32_t GpsNiNotifyFlags;
/** NI requires notification */
#define GPS_NI_NEED_NOTIFY          0x0001
/** NI requires verification */
#define GPS_NI_NEED_VERIFY          0x0002
/** NI requires privacy override, no notification/minimal trace */
#define GPS_NI_PRIVACY_OVERRIDE     0x0004

/**
 * GPS NI responses, used to define the response in
 * NI structures
 */
typedef int GpsUserResponseType;
#define GPS_NI_RESPONSE_ACCEPT         1
#define GPS_NI_RESPONSE_DENY           2
#define GPS_NI_RESPONSE_NORESP         3

/**
 * NI data encoding scheme
 */
typedef int GpsNiEncodingType;
#define GPS_ENC_NONE                   0
#define GPS_ENC_SUPL_GSM_DEFAULT       1
#define GPS_ENC_SUPL_UTF8              2
#define GPS_ENC_SUPL_UCS2              3
#define GPS_ENC_UNKNOWN                -1

/** Represents an NI request */
typedef struct {
   /**
    * An ID generated by HAL to associate NI notifications and UI
    * responses
    */
   int             notification_id;

   /**
    * An NI type used to distinguish different categories of NI
    * events, such as GPS_NI_TYPE_VOICE, GPS_NI_TYPE_UMTS_SUPL, ...
    */
   GpsNiType       ni_type;

   /**
    * Notification/verification options, combinations of GpsNiNotifyFlags constants
    */
   GpsNiNotifyFlags notify_flags;

   /**
    * Timeout period to wait for user response.
    * Set to 0 for no time out limit.
    */
   int             timeout;

   /**
    * Default response when time out.
    */
   GpsUserResponseType default_response;

   /**
    * Requestor ID
    */
   char            requestor_id[GPS_NI_SHORT_STRING_MAXLEN];

   /**
    * Notification message. It can also be used to store client_id in some cases
    */
   char            text[GPS_NI_LONG_STRING_MAXLEN];

   /**
    * Client name decoding scheme
    */
   GpsNiEncodingType requestor_id_encoding;

   /**
    * Client name decoding scheme
    */
   GpsNiEncodingType text_encoding;

   /**
    * A pointer to extra data. Format:
    * key_1 = value_1
    * key_2 = value_2
    */
   char           extras[GPS_NI_LONG_STRING_MAXLEN];

} GpsNiNotification;

/** Callback with NI notification. */
typedef void (*gps_ni_notify_callback)(GpsNiNotification *notification);

/** GPS NI callback structure. */
typedef struct
{
   /**
    * Sends the notification request from HAL to GPSLocationProvider.
    */
   gps_ni_notify_callback notify_cb;
} GpsNiCallbacks;

/**
 * Extended interface for Network-initiated (NI) support.
 */
typedef struct
{
   /** Registers the callbacks for HAL to use. */
   void (*init) (GpsNiCallbacks *callbacks);

   /** Sends a response to HAL. */
   void (*respond) (int notif_id, GpsUserResponseType user_response);
} GpsNiInterface;

#if __cplusplus
}  // extern "C"
#endif

#endif  // _HARDWARE_GPS_NI_H
