/*
 * Copyright (C) 2011 The Android-x86 Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "lights"

#include <cutils/log.h>

#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>

#include <sys/ioctl.h>
#include <sys/types.h>

#include <hardware/lights.h>

/******************************************************************************/

#ifndef max
#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

static pthread_once_t g_init = PTHREAD_ONCE_INIT;
static pthread_mutex_t g_lock = PTHREAD_MUTEX_INITIALIZER;
static int g_haveTrackballLight = 0;
static struct light_state_t g_notification;
static struct light_state_t g_battery;
static int g_backlight = 255;
static int g_trackball = -1;
static int g_buttons = 0;
static int g_attention = 0;

char const* const TRACKBALL_FILE =
		"/sys/class/leds/jogball-backlight/brightness";

/**
 * device methods
 */

void init_globals(void) {
	// init the mutex
	pthread_mutex_init(&g_lock, NULL);

	// figure out if we have the trackball LED or not
	g_haveTrackballLight = (access(TRACKBALL_FILE, W_OK) == 0) ? 1 : 0;
}

static int write_int(char const* path, int value) {
	int fd;
	static int already_warned = 0;

	fd = open(path, O_RDWR);
	if (fd >= 0) {
		char buffer[20];
		int bytes = sprintf(buffer, "%d\n", value);
		int amt = write(fd, buffer, bytes);
		close(fd);
		return amt == -1 ? -errno : 0;
	} else {
		if (already_warned == 0) {
			LOGE("write_int failed to open %s\n", path);
			already_warned = 1;
		}
		return -errno;
	}
}

static int is_lit(struct light_state_t const* state) {
	return state->color & 0x00ffffff;
}

static int handle_trackball_light_locked(struct light_device_t* dev) {
	int mode = 0;

	if (g_attention > 127) {
		mode = 3;
	} else if (!g_backlight && g_attention <= 127 && g_attention > 0) {
		mode = 7;
	}

	// If the value isn't changing, don't set it, because this
	// can reset the timer on the breathing mode, which looks bad.
	if (g_trackball == mode) {
		return 0;
	}

	return write_int(TRACKBALL_FILE, mode);
}

static int rgb_to_brightness(struct light_state_t const* state) {
	int color = state->color & 0x00ffffff;
	return ((77 * ((color >> 16) & 0x00ff)) + (150 * ((color >> 8) & 0x00ff))
			+ (29 * (color & 0x00ff))) >> 8;
}

static int set_light_backlight(struct light_device_t* dev,
		struct light_state_t const* state) {
	int err = 0;
	int brightness = rgb_to_brightness(state); // brightness range: 0 - 255
	pthread_mutex_lock(&g_lock);
	g_backlight = brightness;
	brightness = brightness >> 4;
	brightness = min(brightness,10);// ideapad s10 3t brightness range: 0 - 10
	err = write_int("/sys/class/backlight/acpi_video0/brightness", brightness);
	if (err < 1) {
		LOGE("couldn't brightness errorcode was %d\n", err);
	} else {
		LOGI("set_light_backlight brightness=%d\n", brightness);
	}
	if (g_haveTrackballLight) {
		handle_trackball_light_locked(dev);
	}
	pthread_mutex_unlock(&g_lock);
	return err;
}

static int set_light_keyboard(struct light_device_t* dev,
		struct light_state_t const* state) {
	int err = 0;
	/*
	 int on = is_lit(state);
	 pthread_mutex_lock(&g_lock);
	 err = write_int("/sys/class/leds/keyboard-backlight/brightness", on?255:0);
	 pthread_mutex_unlock(&g_lock);
	 */
	return err;
}

static int set_light_buttons(struct light_device_t* dev,
		struct light_state_t const* state) {
	int err = 0;
	/*
	 int on = is_lit(state);
	 pthread_mutex_lock(&g_lock);
	 g_buttons = on;
	 err = write_int("/sys/class/leds/button-backlight/brightness", on?255:0);
	 pthread_mutex_unlock(&g_lock);
	 */
	return err;
}

static int set_speaker_light_locked(struct light_device_t* dev,
		struct light_state_t const* state) {
	/*
	 int len;
	 int alpha, red, green, blue;
	 int blink, freq, pwm;
	 int onMS, offMS;
	 unsigned int colorRGB;

	 switch (state->flashMode) {
	 case LIGHT_FLASH_TIMED:
	 onMS = state->flashOnMS;
	 offMS = state->flashOffMS;
	 break;
	 case LIGHT_FLASH_NONE:
	 default:
	 onMS = 0;
	 offMS = 0;
	 break;
	 }

	 colorRGB = state->color;

	 #if 0
	 LOGD("set_led_state colorRGB=%08X, onMS=%d, offMS=%d\n",
	 colorRGB, onMS, offMS);
	 #endif

	 red = (colorRGB >> 16) & 0xFF;
	 green = (colorRGB >> 8) & 0xFF;
	 blue = colorRGB & 0xFF;

	 write_int("/sys/class/leds/red/brightness", red);
	 write_int("/sys/class/leds/green/brightness", green);
	 write_int("/sys/class/leds/blue/brightness", blue);

	 if (onMS > 0 && offMS > 0) {
	 int totalMS = onMS  offMS;

	 // the LED appears to blink about once per second if freq is 20
	 // 1000ms / 20 = 50
	 freq = totalMS / 50;
	 // pwm specifies the ratio of ON versus OFF
	 // pwm = 0 -> always off
	 // pwm = 255 => always on
	 pwm = (onMS * 255) / totalMS;

	 // the low 4 bits are ignored, so round up if necessary
	 if (pwm > 0 && pwm < 16)
	 pwm = 16;

	 blink = 1;
	 } else {
	 blink = 0;
	 freq = 0;
	 pwm = 0;
	 }

	 if (blink) {
	 write_int("/sys/class/leds/red/device/grpfreq", freq);
	 write_int("/sys/class/leds/red/device/grppwm", pwm);
	 }
	 write_int("/sys/class/leds/red/device/blink", blink);
	 */

	return 0;
}

static void handle_speaker_battery_locked(struct light_device_t* dev) {
	if (is_lit(&g_battery)) {
		set_speaker_light_locked(dev, &g_battery);
	} else {
		set_speaker_light_locked(dev, &g_notification);
	}
}

static int set_light_battery(struct light_device_t* dev,
		struct light_state_t const* state) {
	/*
	 pthread_mutex_lock(&g_lock);
	 g_battery = *state;
	 if (g_haveTrackballLight) {
	 set_speaker_light_locked(dev, state);
	 }
	 handle_speaker_battery_locked(dev);
	 pthread_mutex_unlock(&g_lock);
	 */
	return 0;
}

static int set_light_notifications(struct light_device_t* dev,
		struct light_state_t const* state) {
	/*
	 pthread_mutex_lock(&g_lock);
	 g_notification = *state;
	 LOGV("set_light_notifications g_trackball=%d color=0x%08x",
	 g_trackball, state->color);
	 if (g_haveTrackballLight) {
	 handle_trackball_light_locked(dev);
	 }
	 handle_speaker_battery_locked(dev);
	 pthread_mutex_unlock(&g_lock);
	 */
	return 0;
}

static int set_light_attention(struct light_device_t* dev,
		struct light_state_t const* state) {
	/*
	 pthread_mutex_lock(&g_lock);
	 g_notification = *state;
	 LOGV("set_light_attention g_trackball=%d color=0x%08x",
	 g_trackball, state->color);
	 g_attention = rgb_to_brightness(state);
	 if (g_haveTrackballLight) {
	 handle_trackball_light_locked(dev);
	 }
	 pthread_mutex_unlock(&g_lock);
	 */
	return 0;
}

/** Close the lights device */
static int close_lights(struct light_device_t *dev) {
	if (dev) {
		free(dev);
	}
	return 0;
}

/******************************************************************************/

/**
 * module methods
 */

/** Open a new instance of a lights device using name */
static int open_lights(const struct hw_module_t* module, char const* name,
		struct hw_device_t** device) {
	int (*set_light)(struct light_device_t* dev,
			struct light_state_t const* state);

	if (0 == strcmp(LIGHT_ID_BACKLIGHT, name)) {
		set_light = set_light_backlight;
	} else if (0 == strcmp(LIGHT_ID_KEYBOARD, name)) {
		set_light = set_light_keyboard;
	} else if (0 == strcmp(LIGHT_ID_BUTTONS, name)) {
		set_light = set_light_buttons;
	} else if (0 == strcmp(LIGHT_ID_BATTERY, name)) {
		set_light = set_light_battery;
	} else if (0 == strcmp(LIGHT_ID_NOTIFICATIONS, name)) {
		set_light = set_light_notifications;
	} else if (0 == strcmp(LIGHT_ID_ATTENTION, name)) {
		set_light = set_light_attention;
	} else {
		return -EINVAL;
	}

	pthread_once(&g_init, init_globals);

	struct light_device_t *dev = malloc(sizeof(struct light_device_t));
	memset(dev, 0, sizeof(*dev));

	dev->common.tag = HARDWARE_DEVICE_TAG;
	dev->common.version = 0;
	dev->common.module = (struct hw_module_t*) module;
	dev->common.close = (int(*)(struct hw_device_t*)) close_lights;
	dev->set_light = set_light;

	*device = (struct hw_device_t*) dev;
	return 0;
}

static struct hw_module_methods_t lights_module_methods =
		{ .open = open_lights, };

/*
 * The lights Module
 */
const struct hw_module_t HAL_MODULE_INFO_SYM = {
	.tag = HARDWARE_MODULE_TAG,
	.version_major = 1,
	.version_minor = 0,
	.id = LIGHTS_HARDWARE_MODULE_ID,
	.name = "s103t lights Module",
	.author = "wuhaigang@gmail.com, kelly2.blue@gmail.com, oliver@ratio-informatik.de",
	.methods = &lights_module_methods,
};

