/*
 * Copyright (c) 2000 Blue Mug, Inc.  All Rights Reserved.
 */

#include <cs89712/ioregs.h>
#include <target/herrno.h>
#include <target/io.h>
#include <target/scan.h>
#include "tags.h"
#include "gunzip.h"
#include "linux.h"
#include "memmap.h"
#include "memzero.h"

/* for setting the root device */
#define MINORBITS       8
#define MKDEV(ma,mi)    (((ma) << MINORBITS) | (mi))

#define RAMDISK_MAJOR   1	/* major for "RAM disk" */
#define RAMDISK0_MINOR	0	/* b 1 0 == /dev/ram0 */

#define ROM_FLASH_MAJOR	31	/* major for "ROM/flash memory card" */
#define FLASH0_MINOR	16	/* b 31 16 == /dev/flash0 */
#define FLASH1_MINOR	17	/* b 31 17 == /dev/flash1 */

#define COMMAND_LINE_SIZE 1024

static int linux_cmdfunc(int argc, char *argv[])
{
	struct tag *tag = (struct tag *) LINUX_PARAM_ADDRESS;

	/* zero param block */
	memzero (tag, LINUX_PARAM_SIZE);

	/* set up core tag */
	tag->hdr.tag = ATAG_CORE;
	tag->hdr.size = tag_size(tag_core);
	tag->u.core.flags = 0;
	tag->u.core.pagesize = 0x1000;
	tag->u.core.rootdev = MKDEV(RAMDISK_MAJOR, RAMDISK0_MINOR);

	/* 16 MB of SDRAM at 0xc0000000 */
	tag = tag_next(tag);
	tag->hdr.tag = ATAG_MEM;
	tag->hdr.size = tag_size(tag_mem32);
	tag->u.mem.size = DRAM1_SIZE >> 12;
	tag->u.mem.start = DRAM1_START;

	/* an initial ramdisk image in flash at 0x00700000 */
	tag = tag_next(tag);
	tag->hdr.tag = ATAG_INITRD;
	tag->hdr.size = tag_size(tag_initrd);
	tag->u.initrd.start = INITRD_LOAD_ADDRESS;
	tag->u.initrd.size  = INITRD_SIZE;

	/* the command line arguments */
	if (argc > 1) {
		tag = tag_next(tag);
		tag->hdr.tag = ATAG_CMDLINE;
		tag->hdr.size = (COMMAND_LINE_SIZE + 3 +
			 sizeof(struct tag_header)) >> 2;

		{
			const unsigned char *src;
			unsigned char *dst;
			dst = tag->u.cmdline.cmdline;
			memzero (dst, COMMAND_LINE_SIZE);
			while (--argc > 0) {
				src = *++argv;
				hprintf ("Doing %s\n", src);
				while (*src)
					*dst++ = *src++;
				*dst++ = ' ';
			}
			*--dst = '\0';
		}
	}

	tag = tag_next(tag);
	tag->hdr.tag = 0;
	tag->hdr.size = 0;

	/* branch to kernel image */
	__asm__ volatile (
	"	mov	r4, #0xC0000000\n"	/* start of DRAM */
	"	add	r4, r4, #0x00028000\n"	/* kernel offset */
	"	mov	r0, #0\n"		/* kernel sanity check */
	"	mov	r1, #123\n"		/* CDB89712 arch. number */
	"	mov	r2, #0\n"
	"	mov	r3, #0\n"
	"	mov	pc, r4"			/* go there! */
	);

	/* never get here */
	return 0;
}

static int unzip_cmdfunc(int argc, char *argv[])
{
	addr_t kernel;
	addr_t ramdisk;

	if (argc < 2 || argc > 3)
		return -H_EUSAGE;

	argv++;
	if (scan(*argv, &kernel))
		return -H_EADDR;
	
	argv++;
	if (scan(*argv, &ramdisk))
		return -H_EADDR;
	
	/* turn on LED flasher (25% duty cycle) */
	IO_LEDFLSH = 0x4c; 
	
	/* decompress kernel image to DRAM */
	gunzip_object (" kernel", kernel, LINUX_LOAD_ADDRESS);
	gunzip_object ("ramdisk", ramdisk, INITRD_LOAD_ADDRESS);

	/* turn off LED flasher */
	IO_LEDFLSH = 0;

	return 0;
}

static int boot_cmdfunc(int argc, char *argv[])
{
	if (argc < 1)
		return -H_EUSAGE;
	
	IO_LEDFLSH = 0x4c; 
	gunzip_object (" kernel", 0x100000, LINUX_LOAD_ADDRESS);
	gunzip_object ("ramdisk", 0x200000, INITRD_LOAD_ADDRESS);
	IO_LEDFLSH = 0;

	linux_cmdfunc (argc, argv);

	return 0;
}

const command_t linux_command =
	{ "linux", "<linux options>", "start Linux", &linux_cmdfunc };
const command_t unzip_command =
	{ "unzip", "<addr> <addr>", "unzip Linux kernel and ramdisk from addrs", &unzip_cmdfunc };
const command_t boot_command =
	{ "boot", "", "boot default Linux kernel and ramdisk", &boot_cmdfunc };

