/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.stat.dist.continuous;

import net.morilib.awk.stat.dist.AbstractContinuousDistribution;

/**
 * コーシー分布です。
 *
 * @author MORIGUCHI, Yuichiro 2013/04/14
 */
public class CauchyDistribution
extends AbstractContinuousDistribution {

	private double x0, gamma;

	/**
	 * 確率分布を生成します。
	 * 
	 * @param x0    位置
	 * @param gamma 尺度
	 */
	public CauchyDistribution(double x0, double gamma) {
		if(gamma <= 0)  throw new IllegalArgumentException();
		this.x0    = x0;
		this.gamma = gamma;
	}

	/**
	 * 位置x_0を得ます。
	 * 
	 * @return x_0
	 */
	public double getX0() {
		return x0;
	}

	/**
	 * 尺度γを得ます。
	 * 
	 * @return γ
	 */
	public double getGamma() {
		return gamma;
	}

	public double invCdf(double x) {
		if(x < 0.0 || x > 1.0)  return Double.NaN;
		return gamma * Math.tan(Math.PI * (x - 0.5)) + x0;
	}

	public boolean isInSupport(double n) {
		return true;
	}

	public double supportInfimum() {
		return Double.NEGATIVE_INFINITY;
	}

	public double supportSupremum() {
		return Double.POSITIVE_INFINITY;
	}

	public double cdf(double x) {
		return Math.atan((x - x0) / gamma) / Math.PI + 0.5;
	}

	public double expectedValue() {
		return Double.NaN;
	}

	public double kurtosis() {
		return Double.NaN;
	}

	public double mode() {
		return x0;
	}

	public double skewness() {
		return Double.NaN;
	}

	public double variance() {
		return Double.NaN;
	}

	public double f(double x) {
		double k;

		k = (x - x0) / gamma;
		return 1.0 / (Math.PI * gamma * (1.0 + k * k));
	}

}
