/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.stat.dist.discrete;

import java.math.BigInteger;

import net.morilib.awk.misc.Hashes;
import net.morilib.awk.stat.dist.AbstractDiscreteDistribution;
import net.morilib.awk.stat.special.Beta;
import net.morilib.awk.stat.special.Gamma;

/**
 * 負の二項分布です。
 * 
 *
 * @author MORIGUCHI, Yuichiro 2012/02/21
 */
public class NegativeBinomialDistribution
extends AbstractDiscreteDistribution {

	//
	private BigInteger size;
	private double succeed;

	/**
	 * 確率分布を生成します。
	 * 
	 * @param size    回数
	 * @param succeed 成功確率
	 */
	public NegativeBinomialDistribution(BigInteger size,
			double succeed) {
		if(size == null) {
			throw new NullPointerException();
		} else if(size.signum() <= 0) {
			throw new IllegalArgumentException();
		} else if(succeed < 0 || succeed > 1) {
			throw new IllegalArgumentException();
		}
		this.size    = size;
		this.succeed = succeed;
	}

	/**
	 * 幾何分布を生成します。
	 * 
	 * @param succeed 成功確率
	 * @return 幾何分布
	 */
	public static NegativeBinomialDistribution geometric(
			double succeed) {
		return new NegativeBinomialDistribution(BigInteger.ONE,
				succeed);
	}

	/**
	 * 回数を得ます。
	 * 
	 * @return 回数
	 */
	public BigInteger getSize() {
		return size;
	}

	/**
	 * 成功確率を得ます。
	 * 
	 * @return 成功確率
	 */
	public double getSucceed() {
		return succeed;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.DiscreteDistribution#f(int)
	 */
	public double f(int x) {
		double n = size.doubleValue(), p = 1.0 - succeed;
		double lb, lp, k = (double)x;

		if(!isInSupport(x))  return 0.0;
		lb  = Gamma.lnGamma(k + n);
		lb -= Gamma.lnGamma(k + 1) + Gamma.lnGamma(n);
		lp  = k * Math.log(p) + n * Math.log(1 - p);
		return Math.exp(lb + lp);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.AbstractDiscreteDistribution#cdf(double)
	 */
	@Override
	public double cdf(double k) {
		double n = size.doubleValue(), p = 1.0 - succeed, r, a, b, l;

		if(k < 0) {
			return 0.0;
		} else {
			l = Math.floor(k);
			r = Beta.I(p,         l + 1, n);
			a = Beta.I(p - 0.001, l + 1, n);
			b = Beta.I(p + 0.001, l + 1, n);
			if(Double.isNaN(r) ||
					(a < r && r > b) || (a > r && r < b)) {
				// bug fix
				return 1 - (a + b) / 2;
			} else {
				return 1 - r;
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.DiscreteDistribution#isInSupport(int)
	 */
	public boolean isInSupport(int n) {
		return n >= 0;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.Distribution#expectedValue()
	 */
	public double expectedValue() {
		return succeed * size.doubleValue() / succeed;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.Distribution#variance()
	 */
	public double variance() {
		double fa = succeed;

		return succeed * size.doubleValue() / fa / fa;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.Distribution#mode()
	 */
	public double mode() {
		double n = size.doubleValue(), p = 1.0 - succeed;

		return Math.floor(p * (n - 1) / (1.0 - p));
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.Distribution#skewness()
	 */
	public double skewness() {
		double n = size.doubleValue(), p = 1.0 - succeed;

		return (1 + p) / Math.sqrt(p * n);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.Distribution#kurtosis()
	 */
	public double kurtosis() {
		double n = size.doubleValue(), f = succeed, p = 1.0 - f;

		return 6.0 / n + f * f / p / n;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.DiscreteDistribution#supportMinimum()
	 */
	public int supportMinimum() {
		return 0;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.stat.dist.DiscreteDistribution#supportMaximum()
	 */
	public int supportMaximum() {
		return Integer.MAX_VALUE - 1;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		int r = Hashes.INIT;

		r = Hashes.A * (size.hashCode() + r);
		r = Hashes.A * ((int)Double.doubleToLongBits(succeed) + r);
		return r;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object o) {
		if(o instanceof NegativeBinomialDistribution) {
			NegativeBinomialDistribution d;

			d = (NegativeBinomialDistribution)o;
			return size.equals(d.size) && succeed == d.succeed;
		}
		return false;
	}

}
