/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.matrix;

import java.util.Arrays;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/09/19
 */
public class ArrayDoubleMatrix extends AbstractArrayDoubleMatrix
implements java.io.Serializable {

	//
	private double[] array;
	private transient DoubleMatrix[] decomposed = null;

	/**
	 * 
	 * @param rows
	 * @param columns
	 */
	public ArrayDoubleMatrix(int rows, int columns) {
		this(rows, columns, 0.0);
	}

	/**
	 * 
	 * @param rows
	 * @param columns
	 * @param fill
	 */
	public ArrayDoubleMatrix(int rows, int columns, double fill) {
		this.array = new double[rows * columns];
		this.rows  = rows;
		this.columns = columns;
		Arrays.fill(this.array, fill);
	}

	/**
	 * 
	 * @param m
	 */
	public ArrayDoubleMatrix(ArrayDoubleMatrix m) {
		array   = new double[m.rows * m.columns];
		rows    = m.rows;
		columns = m.columns;
		System.arraycopy(m.array, 0, array, 0, array.length);
	}

	/**
	 * 
	 * @param m
	 */
	public ArrayDoubleMatrix(DoubleMatrix m) {
		array   = new double[m.rowSize() * m.columnSize()];
		rows    = m.rowSize();
		columns = m.columnSize();
		for(int i = 0; i < m.rowSize(); i++) {
			for(int j = 0; j < m.columnSize(); j++) {
				set(i, j, m.get(i, j));
			}
		}
	}

	/**
	 * 
	 * @param size
	 * @return
	 */
	public static ArrayDoubleMatrix newElementMatrix(int size) {
		ArrayDoubleMatrix r = new ArrayDoubleMatrix(size, size);

		for(int i = 0; i < size; i++) {
			r.set(i, i, 1.0);
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#array(int)
	 */
	@Override
	protected double array(int index) {
		return array[index];
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#arrayset(int, net.morilib.lisp.Datum)
	 */
	@Override
	protected void arrayset(int index, double x) {
		array[index] = x;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#prototype()
	 */
	@Override
	protected ArrayDoubleMatrix prototype() {
		return new ArrayDoubleMatrix(rows, columns);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#prototype(int, int)
	 */
	@Override
	protected ArrayDoubleMatrix prototype(int rows, int columns) {
		return new ArrayDoubleMatrix(rows, columns);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#clone()
	 */
	@Override
	public ArrayDoubleMatrix clone() {
		return new ArrayDoubleMatrix(this);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.AbstractLispMatrix#arraycopy(int, java.lang.Object, int, int)
	 */
	@Override
	protected void arraycopy(int srcPos, Object dest, int destPos,
			int len) {
		System.arraycopy(array, srcPos, dest, destPos, len);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.ILispMatrix#determinant()
	 */
	public double determinant() throws AwkMatrixException {
		if(columns != rows) {
			throw new AwkMatrixException("square matrix required");
		} else {
			if(decomposed == null) {
				decomposed = DoubleMatrices.decomposeLU(this);
			}
			// det(P) * det(U)
			return (decomposed[0].determinant() *
					decomposed[2].determinant());
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.matrix.ILispMatrix#invert()
	 */
	public DoubleMatrix inv() throws AwkMatrixException {
		DoubleMatrix x, y;

		if(columns != rows) {
			throw new AwkMatrixException("square matrix required");
		} else {
			if(decomposed == null) {
				decomposed = DoubleMatrices.decomposeLU(this);
			}
			x = decomposed[2].inv();
			y = decomposed[1].inv();
			x = x.mul(y);
			y = decomposed[0].inv();
			x = x.mul(y);
			return x;
		}
	}

}
