package net.morilib.unix.misc;

import java.text.ParseException;
import java.text.SimpleDateFormat;

import junit.framework.TestCase;

public class ParseDateTest extends TestCase {

	static final SimpleDateFormat FMT =
		new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	static void eql(String s, String t) {
		try {
			assertEquals(FMT.parse(t).getTime(), ParseDate.from(
					new java.util.Date(), s));
		} catch (ParseException e) {
			throw new RuntimeException(e);
		}
	}

	static void eql(java.util.Date z, String s, String t) {
		try {
			assertEquals(FMT.parse(t).getTime(), ParseDate.from(z, s));
		} catch (ParseException e) {
			throw new RuntimeException(e);
		}
	}

	public void testFrom001() {
		eql("2013/01/02", "2013/01/02 00:00:00");
		eql("02/01/2013", "2013/02/01 00:00:00");
		eql("02.01.2013", "2013/01/02 00:00:00");
		eql("2013-01-02", "2013/01/02 00:00:00");
		eql("2013.01.02", "2013/01/02 00:00:00");
		eql("2013年01月02日", "2013/01/02 00:00:00");
		eql("2013년01월02일", "2013/01/02 00:00:00");
		eql("20130102", "2013/01/02 00:00:00");
	}

	public void testFrom002() {
		eql("2013/01/02     12:34", "2013/01/02 12:34:00");
		eql("02/01/2013     12:34", "2013/02/01 12:34:00");
		eql("02.01.2013 12:34", "2013/01/02 12:34:00");
		eql("2013-01-02 12:34", "2013/01/02 12:34:00");
		eql("2013.01.02 12:34", "2013/01/02 12:34:00");
		eql("2013年01月02日 12:34", "2013/01/02 12:34:00");
		eql("20130102 12:34", "2013/01/02 12:34:00");
	}

	public void testFrom003() {
		eql("2013/01/02     12:34:56", "2013/01/02 12:34:56");
		eql("02/01/2013     12:34:56", "2013/02/01 12:34:56");
		eql("02.01.2013 12:34:56", "2013/01/02 12:34:56");
		eql("2013-01-02 12:34:56", "2013/01/02 12:34:56");
		eql("2013.01.02 12:34:56", "2013/01/02 12:34:56");
		eql("2013年01月02日 12:34:56", "2013/01/02 12:34:56");
		eql("20130102 12:34:56", "2013/01/02 12:34:56");
	}

	public void testEnglish001() {
		eql("Jan 02 2013", "2013/01/02 00:00:00");
		eql("Feb 02 2013", "2013/02/02 00:00:00");
		eql("Mar 25 2013", "2013/03/25 00:00:00");
		eql("Apr 03 2013", "2013/04/03 00:00:00");
		eql("May 05 2013", "2013/05/05 00:00:00");
		eql("Jun 23 2013", "2013/06/23 00:00:00");
		eql("Jul 19 2013", "2013/07/19 00:00:00");
		eql("Aug 27 2013", "2013/08/27 00:00:00");
		eql("Sep 02 2013", "2013/09/02 00:00:00");
		eql("Oct 02 2013", "2013/10/02 00:00:00");
		eql("Nov 02 2013", "2013/11/02 00:00:00");
		eql("Dec 24 2012", "2012/12/24 00:00:00");
	}

	public void testEnglish002() {
		eql("Jan  2, 2013", "2013/01/02 00:00:00");
		eql("Feb  2, 2013", "2013/02/02 00:00:00");
		eql("Mar 25, 2013", "2013/03/25 00:00:00");
		eql("Apr  3, 2013", "2013/04/03 00:00:00");
		eql("May  5, 2013", "2013/05/05 00:00:00");
		eql("Jun 23, 2013", "2013/06/23 00:00:00");
		eql("Jul 19, 2013", "2013/07/19 00:00:00");
		eql("Aug 27, 2013", "2013/08/27 00:00:00");
		eql("Sep  2, 2013", "2013/09/02 00:00:00");
		eql("Oct  2, 2013", "2013/10/02 00:00:00");
		eql("Nov  2,   2013", "2013/11/02 00:00:00");
		eql("Dec 24,2012", "2012/12/24 00:00:00");
	}

	public void testEnglish003() {
		eql("Jan  2, 2013 12:34", "2013/01/02 12:34:00");
	}

	public void testEnglish004() {
		eql("Jan 02", "2013/01/02 00:00:00");
		eql("Feb 02", "2013/02/02 00:00:00");
		eql("Mar 25", "2013/03/25 00:00:00");
		eql("Apr 03", "2013/04/03 00:00:00");
		eql("May 05", "2013/05/05 00:00:00");
		eql("Jun 23", "2013/06/23 00:00:00");
		eql("Jul 19", "2013/07/19 00:00:00");
		eql("Aug 27", "2013/08/27 00:00:00");
		eql("Sep 02", "2013/09/02 00:00:00");
		eql("Oct 02", "2013/10/02 00:00:00");
		eql("Nov 02", "2013/11/02 00:00:00");
		eql("Dec 24", "2013/12/24 00:00:00");
	}

	public void testFrench001() {
		eql("2 janv. 2013", "2013/01/02 00:00:00");
		eql("2 fevr. 2013", "2013/02/02 00:00:00");
		eql("25 mars  2013", "2013/03/25 00:00:00");
		eql("3 avr.  2013", "2013/04/03 00:00:00");
		eql("5 mai   2013", "2013/05/05 00:00:00");
		eql("23 juin  2013", "2013/06/23 00:00:00");
		eql("19 juil. 2013", "2013/07/19 00:00:00");
		eql("27 aout  2013", "2013/08/27 00:00:00");
		eql("2 sept. 2013", "2013/09/02 00:00:00");
		eql("2 oct.  2013", "2013/10/02 00:00:00");
		eql("2 nov.  2013", "2013/11/02 00:00:00");
		eql("24 dec.  2012", "2012/12/24 00:00:00");
	}

	public void testFrench002() {
		eql("02 janv. 2013", "2013/01/02 00:00:00");
		eql("02 févr. 2013", "2013/02/02 00:00:00");
		eql("25 mars  2013", "2013/03/25 00:00:00");
		eql("03 avr.  2013", "2013/04/03 00:00:00");
		eql("05 mai   2013", "2013/05/05 00:00:00");
		eql("23 juin  2013", "2013/06/23 00:00:00");
		eql("19 juil. 2013", "2013/07/19 00:00:00");
		eql("27 août  2013", "2013/08/27 00:00:00");
		eql("02 sept. 2013", "2013/09/02 00:00:00");
		eql("02 oct.  2013", "2013/10/02 00:00:00");
		eql("02 nov.  2013", "2013/11/02 00:00:00");
		eql("24 déc.  2012", "2012/12/24 00:00:00");
	}

	public void testFrench003() {
		eql("2 janv. 2013 12:34", "2013/01/02 12:34:00");
	}

	public void testFrench004() {
		eql("02 janv.", "2013/01/02 00:00:00");
		eql("02 févr.", "2013/02/02 00:00:00");
		eql("25 mars", "2013/03/25 00:00:00");
		eql("03 avr.", "2013/04/03 00:00:00");
		eql("05 mai", "2013/05/05 00:00:00");
		eql("23 juin", "2013/06/23 00:00:00");
		eql("19 juil.", "2013/07/19 00:00:00");
		eql("27 août", "2013/08/27 00:00:00");
		eql("02 sept.", "2013/09/02 00:00:00");
		eql("02 oct.", "2013/10/02 00:00:00");
		eql("02 nov.", "2013/11/02 00:00:00");
		eql("24 déc.", "2013/12/24 00:00:00");
	}

	public void testFrench005() {
		eql("02 janvier", "2013/01/02 00:00:00");
		eql("02 février", "2013/02/02 00:00:00");
		eql("25 mars", "2013/03/25 00:00:00");
		eql("03 avril", "2013/04/03 00:00:00");
		eql("05 mai", "2013/05/05 00:00:00");
		eql("23 juin", "2013/06/23 00:00:00");
		eql("19 juilet", "2013/07/19 00:00:00");
		eql("27 août", "2013/08/27 00:00:00");
		eql("02 septembre", "2013/09/02 00:00:00");
		eql("02 octobre", "2013/10/02 00:00:00");
		eql("02 novembre", "2013/11/02 00:00:00");
		eql("24 décembre", "2013/12/24 00:00:00");
	}

	public void testItaly001() {
		eql("2-gen-2013", "2013/01/02 00:00:00");
		eql("2-feb-2013", "2013/02/02 00:00:00");
		eql("25-mar-2013", "2013/03/25 00:00:00");
		eql("3-apr-2013", "2013/04/03 00:00:00");
		eql("5-mag-2013", "2013/05/05 00:00:00");
		eql("23-giu-2013", "2013/06/23 00:00:00");
		eql("19-lug-2013", "2013/07/19 00:00:00");
		eql("27-ago-2013", "2013/08/27 00:00:00");
		eql("2-set-2013", "2013/09/02 00:00:00");
		eql("2-ott-2013", "2013/10/02 00:00:00");
		eql("2-nov-2013", "2013/11/02 00:00:00");
		eql("24-dic-2012", "2012/12/24 00:00:00");
	}

	public void testItaly003() {
		eql("2-gen-2013 12:34", "2013/01/02 12:34:00");
	}

	public void testItaly004() {
		eql("2  gen", "2013/01/02 00:00:00");
		eql("2  feb", "2013/02/02 00:00:00");
		eql("25 mar", "2013/03/25 00:00:00");
		eql("3  apr", "2013/04/03 00:00:00");
		eql("5  mag", "2013/05/05 00:00:00");
		eql("23 giu", "2013/06/23 00:00:00");
		eql("19 lug", "2013/07/19 00:00:00");
		eql("27 ago", "2013/08/27 00:00:00");
		eql("2  set", "2013/09/02 00:00:00");
		eql("2  ott", "2013/10/02 00:00:00");
		eql("2  nov", "2013/11/02 00:00:00");
		eql("24 dic", "2013/12/24 00:00:00");
	}

	public void testSpanish001() {
		eql("2-ene-2013", "2013/01/02 00:00:00");
		eql("2-feb-2013", "2013/02/02 00:00:00");
		eql("25-mar-2013", "2013/03/25 00:00:00");
		eql("3-abr-2013", "2013/04/03 00:00:00");
		eql("5-may-2013", "2013/05/05 00:00:00");
		eql("23-jun-2013", "2013/06/23 00:00:00");
		eql("19-jul-2013", "2013/07/19 00:00:00");
		eql("27-ago-2013", "2013/08/27 00:00:00");
		eql("2-sep-2013", "2013/09/02 00:00:00");
		eql("2-oct-2013", "2013/10/02 00:00:00");
		eql("2-nov-2013", "2013/11/02 00:00:00");
		eql("24-dic-2012", "2012/12/24 00:00:00");
	}

	public void testSpanish003() {
		eql("2-ene-2013 12:34", "2013/01/02 12:34:00");
	}

	public void testSpanish004() {
		eql("2  de ene", "2013/01/02 00:00:00");
		eql("2  de feb", "2013/02/02 00:00:00");
		eql("25 de mar", "2013/03/25 00:00:00");
		eql("3  de abr", "2013/04/03 00:00:00");
		eql("5  de may", "2013/05/05 00:00:00");
		eql("23 de jun", "2013/06/23 00:00:00");
		eql("19 de jul", "2013/07/19 00:00:00");
		eql("27 de ago", "2013/08/27 00:00:00");
		eql("2  de sep", "2013/09/02 00:00:00");
		eql("2  de oct", "2013/10/02 00:00:00");
		eql("2  de nov", "2013/11/02 00:00:00");
		eql("24 de dic", "2013/12/24 00:00:00");
	}

	public void testPortuguese001() {
		eql("2/Jan/2013", "2013/01/02 00:00:00");
		eql("2/Fev/2013", "2013/02/02 00:00:00");
		eql("25/Mar/2013", "2013/03/25 00:00:00");
		eql("3/Abr/2013", "2013/04/03 00:00:00");
		eql("5/Mai/2013", "2013/05/05 00:00:00");
		eql("23/Jun/2013", "2013/06/23 00:00:00");
		eql("19/Jul/2013", "2013/07/19 00:00:00");
		eql("27/Ago/2013", "2013/08/27 00:00:00");
		eql("2/Set/2013", "2013/09/02 00:00:00");
		eql("2/Out/2013", "2013/10/02 00:00:00");
		eql("2/Nov/2013", "2013/11/02 00:00:00");
		eql("24/Dez/2012", "2012/12/24 00:00:00");
	}

	public void testPortuguese003() {
		eql("2/Jan/2013 12:34", "2013/01/02 12:34:00");
	}

	public void testPortuguese004() {
		eql("2  de jan", "2013/01/02 00:00:00");
		eql("2  de fev", "2013/02/02 00:00:00");
		eql("25 de mar", "2013/03/25 00:00:00");
		eql("3  de abr", "2013/04/03 00:00:00");
		eql("5  de mai", "2013/05/05 00:00:00");
		eql("23 de jun", "2013/06/23 00:00:00");
		eql("19 de jul", "2013/07/19 00:00:00");
		eql("27 de ago", "2013/08/27 00:00:00");
		eql("2  de set", "2013/09/02 00:00:00");
		eql("2  de out", "2013/10/02 00:00:00");
		eql("2  de nov", "2013/11/02 00:00:00");
		eql("24 de dez", "2013/12/24 00:00:00");
	}

	public void testDutch001() {
		eql("2-jan-2013", "2013/01/02 00:00:00");
		eql("2-feb-2013", "2013/02/02 00:00:00");
		eql("25-mrt-2013", "2013/03/25 00:00:00");
		eql("3-apr-2013", "2013/04/03 00:00:00");
		eql("5-mei-2013", "2013/05/05 00:00:00");
		eql("23-jun-2013", "2013/06/23 00:00:00");
		eql("19-jul-2013", "2013/07/19 00:00:00");
		eql("27-aug-2013", "2013/08/27 00:00:00");
		eql("2-sep-2013", "2013/09/02 00:00:00");
		eql("2-okt-2013", "2013/10/02 00:00:00");
		eql("2-nov-2013", "2013/11/02 00:00:00");
		eql("24-dec-2012", "2012/12/24 00:00:00");
	}

	public void testDutch003() {
		eql("2-jan-2013 12:34", "2013/01/02 12:34:00");
	}

	public void testDutch004() {
		eql("2  jan", "2013/01/02 00:00:00");
		eql("2  feb", "2013/02/02 00:00:00");
		eql("25 mrt", "2013/03/25 00:00:00");
		eql("3  apr", "2013/04/03 00:00:00");
		eql("5  mei", "2013/05/05 00:00:00");
		eql("23 jun", "2013/06/23 00:00:00");
		eql("19 jul", "2013/07/19 00:00:00");
		eql("27 aug", "2013/08/27 00:00:00");
		eql("2  sep", "2013/09/02 00:00:00");
		eql("2  okt", "2013/10/02 00:00:00");
		eql("2  nov", "2013/11/02 00:00:00");
		eql("24 dec", "2013/12/24 00:00:00");
	}

	public void testGermany004() {
		eql("2  Jan", "2013/01/02 00:00:00");
		eql("2  Feb", "2013/02/02 00:00:00");
		eql("25 Mär", "2013/03/25 00:00:00");
		eql("3  Apr", "2013/04/03 00:00:00");
		eql("5  Mai", "2013/05/05 00:00:00");
		eql("23 Jun", "2013/06/23 00:00:00");
		eql("19 Jul", "2013/07/19 00:00:00");
		eql("27 Aug", "2013/08/27 00:00:00");
		eql("2  Sep", "2013/09/02 00:00:00");
		eql("2  Okt", "2013/10/02 00:00:00");
		eql("2  Nov", "2013/11/02 00:00:00");
		eql("24 Dez", "2013/12/24 00:00:00");
	}

	public void testJapanese001() {
		eql("明治10年01月02日", "1877/01/02 00:00:00");
		eql("大正10年01月02日", "1921/01/02 00:00:00");
		eql("昭和10年01月02日", "1935/01/02 00:00:00");
		eql("平成10年01月02日", "1998/01/02 00:00:00");
	}

	public void testJapanese002() {
		eql("明治10-01-02", "1877/01/02 00:00:00");
		eql("大正10-01-02", "1921/01/02 00:00:00");
		eql("昭和10-01-02", "1935/01/02 00:00:00");
		eql("平成10-01-02", "1998/01/02 00:00:00");
		eql("明治100102", "1877/01/02 00:00:00");
		eql("大正100102", "1921/01/02 00:00:00");
		eql("昭和100102", "1935/01/02 00:00:00");
		eql("平成100102", "1998/01/02 00:00:00");
		eql("明10-01-02", "1877/01/02 00:00:00");
		eql("大10-01-02", "1921/01/02 00:00:00");
		eql("昭10-01-02", "1935/01/02 00:00:00");
		eql("平10-01-02", "1998/01/02 00:00:00");
		eql("M10-01-02", "1877/01/02 00:00:00");
		eql("T10-01-02", "1921/01/02 00:00:00");
		eql("S10-01-02", "1935/01/02 00:00:00");
		eql("H10-01-02", "1998/01/02 00:00:00");
	}

	public void testJapanese003() {
		eql("明治10年01月02日 12:34", "1877/01/02 12:34:00");
		eql("明治10-01-02 12:34", "1877/01/02 12:34:00");
	}

	public void testCJK004() {
		eql("1月2日", "2013/01/02 00:00:00");
		eql("1월2일", "2013/01/02 00:00:00");
	}

	public void testDiff001() {
		try {
			java.util.Date t = FMT.parse("2011/01/20 00:00:00");

			eql(t, "1  day", "2011/01/21 00:00:00");
			eql(t, "12 days", "2011/02/01 00:00:00");
			eql(t, "1  month", "2011/02/20 00:00:00");
			eql(t, "12 months", "2012/01/20 00:00:00");
			eql(t, "1  year", "2012/01/20 00:00:00");
			eql(t, "2  years", "2013/01/20 00:00:00");
			eql(t, "1  month 2 days", "2011/02/22 00:00:00");
			eql(t, "1  year  3 months 2 days", "2012/04/22 00:00:00");
		} catch (ParseException e) {
			throw new RuntimeException();
		}
	}

	public void testDiff002() {
		try {
			java.util.Date t = FMT.parse("2011/01/20 00:00:00");

			eql(t, "1  day ago", "2011/01/19 00:00:00");
			eql(t, "12 days ago", "2011/01/08 00:00:00");
			eql(t, "1  month ago", "2010/12/20 00:00:00");
			eql(t, "12 months ago", "2010/01/20 00:00:00");
			eql(t, "1  year ago", "2010/01/20 00:00:00");
			eql(t, "2  years ago", "2009/01/20 00:00:00");
			eql(t, "1  month 2 days ago", "2010/12/18 00:00:00");
			eql(t, "1  year  3 months 2 days ago", "2009/10/18 00:00:00");
		} catch (ParseException e) {
			throw new RuntimeException();
		}
	}

}
