	// AXL3 - An XLL Library
// Wrapper class library for Microsft Excel's xll add-in which written by C++.
//
// Copyright(c) 2009-2009, yukimi_sake@users.sourceforge.jp. All rights reserved.

// A class to keep Workbook file name.
// Works like an Excel VBA's Workbook object.
#pragma once

namespace xl{
	class xlBook{
	private:
		xlVar m_bookname;
		xlVar getFullName(const xlVar& sheetname){
			return xlVar("[")+m_bookname + xlVar("]")+sheetname;
		}
		xlVar getFullName(const char* sheetname){
			return xlVar("[")+m_bookname + xlVar("]")+sheetname;
		}

	public:
	// static methods
		static xlBook NewBook(){
			int r = API.Call(xlcNew, 0, 0);
			return xlBook(XLM.GetActiveBookName());
		}

		static xlBook ThisBook(){
			return xlBook(XLM.GetActiveBookName());
		}

		static xlBook OpenBook(const xlVar& filename){
			int r = API.Call(xlcOpen, 0 , "P", &filename);
			xlBook tmp(filename);
			tmp.m_bookname = XLM.GetActiveBookName();
			return tmp;
		}

		static xlBook OpenBook(const char* filename){
			return OpenBook(xlVar(filename));
		}

	// constructors
		xlBook();
		xlBook(const xlVar& name){
			m_bookname = name;
		}

		xlBook(const char* name){
			m_bookname = xlVar(name);
		}

	// properties
		PrReader(xlVar&, Path);
		PrReader(xlVar&, Name);

	// access methods
		xlVar getPath(){
			return XLM.GetDocumentPath(m_bookname);
		}
		xlVar& getName(){
			return m_bookname;
		}

	// public methods
		void Activate(){ 
			API.Call(xlcActivate, 0, "P", &m_bookname);
		}

		void Close(){
			API.Call(xlcClose, 0, 0);
		}

		void Close(bool save_logical){
			API.Call(xlcClose, 0, "B", save_logical);
		}

		void Close(bool save_logical, bool route_logical){
			API.Call(xlcClose, 0, "BB", save_logical, route_logical);
		}

		void Save(){
			API.Call(xlcSave, 0, 0);
		}

		void SaveAs(const xlVar& filename){ 
			API.Call(xlcSaveWorkbook,0, "P", &filename);
			m_bookname = XLM.GetActiveBookName();
		}

		void SaveAs(const char* filename){
			SaveAs(xlVar(filename));
		}

		void SaveAs(
				const xlVar& filename, 
				const int type_num, 
				const xlVar& prot_pwd,
				const bool backup,
				const xlVar& write_res_pwd, 
				const bool read_only_rec
			)
		{
			API.Call(xlcSaveWorkbook,0, "PIPBPB", &filename, type_num, &prot_pwd, backup, &write_res_pwd, read_only_rec);
			m_bookname = XLM.GetActiveBookName();
		}

		/*
			Type_num  and File format
			1  or omitted Normal
			2  SLK
			3  Text
			6  CSV
			17 Template
			36 Formatted text (space delimited)
			
			See XLM Macro help for more infomation.
		*/

		void SaveAs(
				const char* filename, 
				const int type_num, 
				const char* prot_pwd,
				const bool backup,
				const char* write_res_pwd, 
				const bool read_only_rec
			)
		{
			API.Call(xlcSaveWorkbook, 0, "CICBCB", filename, type_num, prot_pwd, backup, write_res_pwd, read_only_rec);
			m_bookname = XLM.GetActiveBookName();
		}
		
		xlSheet Sheet(const xlVar& sheetname){
			return xlSheet(getFullName(sheetname));
		}

		xlSheet Sheet(const char* sheetname){
			return xlSheet(getFullName(sheetname));
		}

		xlSheet InsertSheet(const int pos=1){
			Activate();
			return xlSheet::Insert(pos);
		}

		xlSheet InsertSheet(const xlVar& sheetname, const int pos=1){
			Activate();
			return xlSheet::Insert(sheetname, pos);
		}

		xlSheet InsertSheet(const char* sheetname, const int pos=1){
			return InsertSheet(xlVar(sheetname), pos);
		}

		void CopySheets(const xlVar& sheetnames, xlBook& destbook, const int pos=1){ // copy multi sheets
			API.Call(xlcWorkbookCopy, 0, "PPI", &sheetnames, &(destbook.Name), pos);
		}

		xlBook CopySheets(const xlVar& sheetnames){
			API.Call(xlcWorkbookCopy, 0, "P", &sheetnames);
			return xlBook::ThisBook();
		}

		void MoveSheets(const xlVar& sheetnames, xlBook& destbook, const int pos=1){ // move multi sheets 
			API.Call(xlcWorkbookMove, 0, "PPI", &sheetnames, &(destbook.Name), pos);
		}

		xlBook MoveSheets(const xlVar& sheetnames){
			API.Call(xlcWorkbookMove, 0, "P", &sheetnames);
			return xlBook::ThisBook();
		}

		void SelectSheets(const xlVar sheetnames, const xlVar activesheet){ // select multi sheets 
			API.Call(xlcWorkbookSelect, 0, "PP", &sheetnames, activesheet);
		}

		void SelectSheets(const xlVar sheetnames){
			API.Call(xlcWorkbookSelect, 0, "P", &sheetnames);
		}

		xlVar NamesOfSelection(){
			xlVar pret;
			int r = API.Call(xlfGetWorkbook, &pret, "IP", 3, &m_bookname);
			return pret;
		}

		xlVar NamesOfAllSheets(){
			xlVar pret;
			API.Call(xlfGetWorkbook, &pret, "IP", 1, &m_bookname);
			return pret;
		}
	};
}