/**
 *
 * Class variables
 */

// include all Eos command's info.
// For seaching with O(n), the object key name is command name.
var db = require('./DB.js').instance;
var commandList = require(__dirname + '/../../user-specific-files/OptionControlFile/command_list.json');
var ocfReference = {};

commandList.forEach(function(c) {
    ocfReference[c.name] = require(__dirname + '/../../user-specific-files/OptionControlFile/commands/' + c.name);
});
var spawn = require('child_process').spawn;


function hasOneProperty(options) {
    return new Promise(function(resolve, reject) {
        if(options.length === 0) {
            var errorMsg = 'At least one option is required.';
            throw new Error(errorMsg);
        } else {
            resolve();
        }
    });
}

function matchOption(options, ocf) {
    return new Promise(function(resolve, reject) {
        var ok = {};
        var notIncludingRequiredOptions = [];
        options.forEach(function(o) { 
            ok[o.name] = o.arguments;
        });
        ocf.forEach(function(o) {
            if(o.optionProperties && !ok[o.option]) {
                notIncludingRequiredOptions.push(o.option);
            }
        });

        // check whether all required option exist 
        if(notIncludingRequiredOptions.length > 0) {
            var errorMsg = 'Option ' + notIncludingRequiredOptions.toString() + ' are required';
            reject(new Error(errorMsg));
        } else {
            resolve();
        }
    });
}

function validArgumentsNumber(options, ocfObj) {
    return new Promise(function(resolve, reject) {
        options.forEach(function(o) {
            // option number
            var expectNum = ocfObj[o.name].optionNumber;
            var actualNum = o.arguments.length;
            if(expectNum !== actualNum) {
                reject(new Error('Invalid arguments number'));
            }
        });
        resolve();
    });
};

function validArgumentsType(options, ocfObj, workspace) {
    return new Promise(function(resolve, reject) {
        options.forEach(function(o) {
            o.arguments.forEach(function(arg,i) {
                // argType
                var formType = ocfObj[o.name].arg[i].formType
                if(formType === 'select') { // This argument is filename
                    var exist = workspace.indexOf(arg) > -1;
                    if(!exist) {
                        reject(new Error(arg + ' doesn\'t exist.'));
                    }
                } else {
                    var expectType = formType === 'text' ? 'string' : 'number';
                    var actualType = typeof arg;
                    if(expectType !== actualType) {
                        reject(new Error('argType is invalid'));
                    }
                }
            });
        });
        resolve();
    });
}

function validOutfileName(options, ocfObj, workspace) {
    return new Promise(function(resolve, reject) {
        // output file Regexp
        var outRegExp = /out|append/;

        options.forEach(function(o) {
            o.arguments.forEach(function(arg,i) {
                // outFile name
                if(outRegExp.test(ocfObj[o.name].arg[i].argType)) {
                    if(workspace.indexOf(o.arguments[i]) > -1) {
                        reject(new Error('Invalid outfile name.'));
                    }
                }
            });
        });
        resolve();
    });
}

/**
 * validate 
 * コマンドとオプションのバリデーション
 * @param command
 * @param params
 * @returns {valid: boolean, message: string}
 */
function validate(command, options, workspaceId) {
    return new Promise(function(resolve, reject) {

        var ocf;
        if(ocfReference[command]) {
            ocf = ocfReference[command];
        } else {
            var errorMsg = 'Command name is invalid';
            reject(new Error(errorMsg));
        }

        var ocfObj = {};
        ocf.forEach(function(o) {
            ocfObj[o.option] = o;
        });

        var optionsObj = {};
        if(Array.isArray(options)) {
            options.forEach(function(o) {
                if(o.name && o.arguments) {
                    if(Array.isArray(o.arguments)) {
                        optionsObj[o.name] = o.arguments;
                    } else {
                        var errorMsg = 'Each "arguments" properties needs to be Array';
                        reject(new Error(errorMsg));
                    }
                } else {
                    var errorMsg = 'Options need to include Object which have properties "name" and "arguments"';
                    reject(new Error(errorMsg));
                }
            });
        } else {
            var errorMsg = 'Options need to be Array';
            reject(new Error(errorMsg));
        }

        getFiles(workspaceId)
        .then(function(workspace) {

            // validate function
            var promises = [];
            promises.push(hasOneProperty(options));
            promises.push(matchOption(options, ocf));
            promises.push(validArgumentsNumber(options, ocfObj));
            promises.push(validArgumentsType(options, ocfObj, workspace));
            promises.push(validOutfileName(options, ocfObj, workspace));

            // do validation
            return Promise.all(promises)
        })
        .catch(function(error) {
            reject(error);
        })
        .then(function(r) {
            resolve();
        })
    });
}

/**
 * toExecString
 *
 * @param command
 * @param options
 * @returns {string}
 */
function toExecString(command, options, workspaceId) {
    var ocf = ocfReference[command]; // Array
    var finalOptions = {};
    var execStr = command + ' ';
    var ocfObj = {};
    ocf.forEach(function(o) {
        ocfObj[o.option] = o;
    });


    if(workspaceId === "1f83f620-c1ed-11e5-9657-7942989daa00") { // root
        var root = __dirname + '/../../user-specific-files/workspace/';
    }

    // set default parameters
    ocf.forEach(function(o) {
        o.arg.forEach(function(arg) {
            if(!(arg.initialValue === "") && arg.initialValue) {
                if(!(finalOptions[o.option])) {
                    finalOptions[o.option] = [];
                    finalOptions[o.option].push(arg.initialValue);
                } else {
                    finalOptions[o.option].push(arg.initialValue);
                }
            }
        });
    });

    // set user setting parameters
    options.forEach(function(o) {
        var s = [];
        var outRegExp = /out|append/;
        o.arguments.forEach(function(arg, i) {
            if(ocfObj[o.name].arg[i].formType === 'select' || outRegExp.test(ocfObj[o.name].arg[i].argType)) {
                s.push(root+arg);
            } else {
                s.push(arg);
            }
        });
        finalOptions[o.name] = s;
    });

    // set execution string
    Object.keys(finalOptions).forEach(function(key) {
        execStr += key + ' ';
        finalOptions[key].forEach(function(arg) {
            execStr += arg + ' ';
        });
    });

    // remove last blank
    execStr = execStr.slice(0,execStr.length-1);

    return execStr;
}

/**
 * toExecArray
 *
 * @param {fileId} 
 * @returns {string}
 */
function toExecArray(command, options, workspaceId) {
    return new Promise(function(resolve, reject) {
        var ocf = ocfReference[command]; // Array
        var finalOptions = {};
        var ocfObj = {};
        ocf.forEach(function(o) {
            ocfObj[o.option] = o;
        });

        // set default parameters
        ocf.forEach(function(o) {
            o.arg.forEach(function(arg) {
                if(!(arg.initialValue === "") && arg.initialValue) {
                    if(!(finalOptions[o.option])) {
                        finalOptions[o.option] = [];
                        finalOptions[o.option].push(arg.initialValue);
                    } else {
                        finalOptions[o.option].push(arg.initialValue);
                    }
                }
            });
        });

        getUUIDs(workspaceId)
        .then(function(uuids) {
            // set user setting parameters
            options.forEach(function(o) {
                var s = [];
                var outRegExp = /out|append/;
                o.arguments.forEach(function(arg, i) {
                    if(ocfObj[o.name].arg[i].formType === 'select') {
                        s.push(uuids[arg]);
                    } else {
                        s.push(arg);
                    }
                });
                finalOptions[o.name] = s;
            });
            var array = Object.keys(finalOptions).reduce(function(a,b) {
                a.push(b);
                finalOptions[b].forEach(function(v) {
                    a.push(v);
                });
                return a;
            },[]);
            resolve(array);
        });
    });
}

/**
 * execute
 *
 * @param command
 * @param params
 * @returns {object}
 */
function execute(command, optionsArray) {
    return new Promise(function(resolve, reject) {
        var workspace; 
        if(process.env.NODE_ENV === 'debug') {
            workspace = __dirname + '/../../user-specific-files/workspace.debug';
        } else {
            workspace = _dirname + '/../../user-specific-files/workspace';
        }
        
        var config = {
            cwd: workspace
        };
        var runner = spawn(command,optionsArray,config);
        
        runner.on('close', function() {
            resolve();
        });
    });
}

/**
 * getFiles
 * @param fileId
 * @returns {promise} resolve(Array)
 */
function getFiles(fileId) {
    return new Promise(function(resolve, reject) {
        if(process.env.NODE_ENV === 'test') {
            resolve(['file1.txt', 'file2.txt']);
        } else {
            db.getFiles(fileId)
            .then(function(r) {
                var workspace = r.map(function(f) { return f.name });
                resolve(workspace);
            });
        }
    });
}

/**
 * getUUID
 * @param fileId 
 * @returns {object} key: filename, value: uuid
 */
function getUUIDs(fileId) {
    return new Promise(function(resolve) {
        db.getFiles(fileId)
        .then(function(r) {
            var uuids = {};
            r.forEach(function(v) {
                uuids[v.dataValues.name] = v.dataValues.fileId;
            });
            resolve(uuids);
        });
    });
}

/**
 * Eosコマンドをエミュレートするクラス 
 * @varructor
 * @returns {object}
 * function execute(command, params) {
 */
var eos = {
    validate: validate,
    toExecString: toExecString,
    execute: execute,
    getFiles: getFiles,
    getUUIDs: getUUIDs,
    toExecArray: toExecArray
}

module.exports = { instance: eos };
