﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nft.framework.drawing;
using System.Windows.Forms;
using WinFormsGraphicsDevice;
using Microsoft.Xna.Framework.Graphics;

namespace nft.xna
{
    #region arias
    using SysColor = System.Drawing.Color;
    using XnaColor = Microsoft.Xna.Framework.Color;
    using SysRect = System.Drawing.Rectangle;
    using XnaRect = Microsoft.Xna.Framework.Rectangle;
    using Point = System.Drawing.Point;
    using DefaultEffect = StockEffects.AlphaTestEffect;
    #endregion

    public class XnaSurface : ISurface
    {
        static internal XnaSurface CreateForControl(Control c) {
            Form f = c.TopLevelControl as Form;
            if (f == null) {
                f = Application.OpenForms[0];
            }
            return new XnaSurface(f.Handle, c.ClientSize.Width, c.ClientSize.Height, c);
        }

        static int AdjustToOddPixels(int i) {
            return ((i + 1) | 1) - 1;
        }

        public XnaSurface(int width, int height) : this(IntPtr.Zero, width, height, null){
        }

        protected XnaSurface(IntPtr hwnd, int width, int height, Control related) {
            this.control = related;
            this.width = width;
            this.height = height;
            srvGraphicsDev = GraphicsDeviceService.AddRef(hwnd, width, height);

        }

        GraphicsDeviceService srvGraphicsDev;
        protected Control control;
        protected RenderTarget2D target = null;
        protected IEnumerable<I3DObject> drawObjects = null;
        protected PointF3D viewpos = new PointF3D();
        protected long frameTicks = 0;
        protected bool clear = true;
        protected XnaColor fillcolor = XnaColor.CornflowerBlue;
        protected int width;
        protected int height;

        #region ISurface implementation

        public IEnumerable<I3DObject> Objects {
            get {
                return drawObjects;
            }
            set {
                drawObjects = value;
            }
        }

        public void Draw() {
            if (control != null) {
                width = control.ClientSize.Width;
                height = control.ClientSize.Height;
            }
            BeginDraw();
            DrawMain();
            EndDraw();
        }

        public SysColor FillColor {
            get {
                return XnaUtil.ToSysColor(fillcolor);
            }
            set {
                fillcolor = XnaUtil.ToXnaColor(value);
            }
        }

        public long GameTicks {
            get { return this.frameTicks; }
            set { frameTicks = value; }
        }

        public bool ClearEveryDraw {
            get { return clear; }
            set { clear = value; }
        }

        public IEffectFilter Effect {
            get {
                throw new NotImplementedException();
            }
            set {
                throw new NotImplementedException();
            }
        }

        public PointF3D CameraPosition {
            get {
                return viewpos;
            }
            set {
                viewpos = value;
            }
        }

        public PointF3D ToWorldPosition(Point screenpos) {
            throw new NotImplementedException();
        }
        #endregion

        #region Drawing core
        protected void BeginDraw() {
            GraphicsDevice.SetRenderTarget(RenderTarget);
            // Check device-lost and prepare proper drawing region.
            // Adjust to odd number size to suppress pixel distortion.
            srvGraphicsDev.PrepareDrawing(AdjustToOddPixels(Width),AdjustToOddPixels(Height));
        }

        protected virtual void DrawMain() {
            DefaultEffect ef = Scene.Prepare(this);
            if (clear)
                GraphicsDevice.Clear(fillcolor);
            if (drawObjects == null) return;
            foreach (I3DObject o in drawObjects) {
                ((IDrawable)o).RenderSelf(this, ef);
            }
        }

        /// <summary>
        /// Commit drawing and render image onto proper region of the control.
        /// </summary>
        void EndDraw() {
            if (control != null) {
                try {
                    XnaRect sourceRectangle = new XnaRect(0, 0, Width, Height);
                    GraphicsDevice.Present(sourceRectangle, null, control.Handle);
                } catch {
                    // 'Present' may throw exception when device lost.
                    // Device lost will be detected on next BeginDraw, so do nothing here.
                }
            } else {
                GraphicsDevice.SetRenderTarget(null);
            }
        }

        #endregion

        #region Properties
        /// <summary>
        /// このコントロールへの描画に使用できる GraphicsDevice を取得します。
        /// </summary>
        public GraphicsDevice GraphicsDevice {
            get { return srvGraphicsDev.GraphicsDevice; }
        }

        /// <summary>
        /// オフスクリーン描画する場合のRenderTarget。コントロールに描画する場合はnullを返す。
        /// returns null when draw on screen
        /// </summary>
        public RenderTarget2D RenderTarget {
            get { return target; }
        }

        public int Width {
            get { return width; }
        }

        public int Height {
            get { return height; }
        }
        #endregion

        #region IDisposable 定型
        public virtual void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(Boolean disposing)
        {
            if (disposing && srvGraphicsDev != null) {
                srvGraphicsDev.Release(disposing);
                srvGraphicsDev = null;
            }
        }

        public bool IsDisposed { get { return srvGraphicsDev == null; } }

        ~XnaSurface()
        {
            Dispose(false);
        }
        #endregion
    }
}
