﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Runtime.Serialization;
using nft.framework.drawing;

using Geocon = nft.core.geometry.GeometricConstants;

namespace nft.core.geometry {
    public class CliffPolygon : ITerrainPolygon, ISerializable {
        static Dictionary<ushort, CliffPolygon> stock = new Dictionary<ushort, CliffPolygon>(256);
        // To enhance brightness, use another sun light vector which has only holizontal elements.
        static protected Vect3D SunLight2D;

        static CliffPolygon() {
#if DEBUG
            // TODO: 太陽光ベクトルをコントラストくっきりするように暫定調整
            SunLight2D = new Vect3D(1.0, 0.0, 0.0/*Geocon.SunLight*/);
#else
            SunLight2D = new Vect3D(Geocon.SunLight);
#endif
            SunLight2D.Z = 0;
            SunLight2D.Normalize();
            initPatterns(Geocon.TerrainHeightStep, Geocon.TerrainHeightMax);
        }

        #region initialize polygon pattern table
        private static void initPatterns(int step, int max) {
            for (int p2 = step; p2 <= max; p2 += step) {
                registerTriPattern(0, p2, -1);
                registerTriPattern(0, -1, p2);
                registerTriPattern(-1, 0, p2);
                registerTriPattern(p2, 0, -1);
                registerTriPattern(p2, -1, 0);
                registerTriPattern(-1, p2, 0);
            }
            for (int p1 = step; p1 <= max; p1 += step) {
                for (int p2 = p1; p2 <= max; p2 += step) {
                    registerRectPattern(p1, p2, -1);
                    registerRectPattern(p1, -1, p2);
                    registerRectPattern(-1, p1, p2);
                    registerRectPattern(p2, p1, -1);
                    registerRectPattern(p2, -1, p1);
                    registerRectPattern(-1, p2, p1);
                }
            }
        }

        /// <summary>
        /// One of the argument should be negative as it means unused point.
        /// Another one should be zero, and the last one is positive, 
        /// so that they form a triangle with the holizontal line.
        /// </summary>
        /// <param name="near"></param>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <param name="far"></param>
        private static void registerTriPattern(int near, int left, int right) {
            ushort id = PrivateMakePolygonID(near, left, right);
            int hp = Geocon.CellWidthPixel >> 2;
            if (!stock.ContainsKey(id)) {
                Location[] pt = new Location[3];
                int idx = 0;
                if (left >= 0) {
                    pt[idx++] = new Location(0, Geocon.CellWidthPixel, left * hp);
                    if (left > 0)
                        pt[idx++] = new Location(0, Geocon.CellWidthPixel, 0);
                }
                if (near >= 0) {
                    pt[idx++] = new Location(0, 0, near * hp);
                    if (near > 0)
                        pt[idx++] = new Location(0, 0, 0);
                }
                if (right >= 0) {
                    pt[idx++] = new Location(Geocon.CellWidthPixel, 0, right * hp);
                    if (right > 0)
                        pt[idx++] = new Location(Geocon.CellWidthPixel, 0, 0);
                }
                stock.Add(id, new CliffPolygon(id, pt));
            }
        }

        /// <summary>
        /// One of the argument should be negative so that which means unused point.
        /// </summary>
        /// <param name="near"></param>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <param name="far"></param>
        private static void registerRectPattern(int near, int left, int right) {
            ushort id = PrivateMakePolygonID(near, left, right);
            int hp = Geocon.CellWidthPixel >> 2;
            if (!stock.ContainsKey(id)) {
                Location[] pt = new Location[4];
                int idx = 0;
                if (left >= 0) {
                    pt[idx++] = new Location(0, Geocon.CellWidthPixel, left * hp);
                    pt[idx++] = new Location(0, Geocon.CellWidthPixel, 0);
                }
                if (near >= 0) {
                    pt[idx++] = new Location(0, 0, near * hp);
                    pt[idx++] = new Location(0, 0, 0);
                }
                if (right >= 0) {
                    pt[idx++] = new Location(Geocon.CellWidthPixel, 0, right * hp);
                    pt[idx++] = new Location(Geocon.CellWidthPixel, 0, 0);
                }
                // swap first and second location to make cyclic order for drawing polygon.
                Location l = pt[0];
                pt[0] = pt[1];
                pt[1] = l;
                stock.Add(id, new CliffPolygon(id, pt));
            }
        }
        #endregion

        public readonly ushort id;
        protected Point[] verticis;
        protected Rectangle bounds;
        /// <summary>
        /// Determined by surface normal angle between the sun light. 255 is maximum.
        /// Visible surface has positive value, effected at least by environment light.
        /// Zero value means inverted or perpendicular surface against camera angle.
        /// </summary>
        public readonly int Brightness;

        protected CliffPolygon(ushort id, Location[] pts) {
            this.id = id;
            this.bounds = new Rectangle();
            this.verticis = Location.ConvertToQuarterViewPos(pts, ref bounds);
            Vect3D normal = Vect3D.CalcNormalVector(pts);
            double inprd = normal.InnerProduct(SunLight2D);
            if (inprd > 0) {
                // inverted plane
                normal.Multiple(-1.0);
                inprd = -inprd;
            }
            double l = Geocon.EnvironmentLight;
            l += (l - 1.0) * inprd;
            Brightness = (int)(0xff * l);
        }

        #region ITerrainPolygon メンバ

        public ushort ID {
            get { return id; }
        }

        /// <summary>
        /// Returns 2D bounds rectangle which contains all verticis points.
        /// </summary>
        public Rectangle GetBounds(Scaler sc) {
            Rectangle ret = sc.Scale(bounds);
            return ret;
        }

        /// <summary>
        /// Calc verticis according to the ViewFactor. A bit heavy process.
        /// </summary>
        /// <param name="vf"></param>
        /// <returns></returns>
        public Point[] GetVerticis(Scaler sc) {
            Point[] ret = sc.Scale(verticis);
            return ret;
        }
        #endregion

        /// <summary>
        /// Get total count of stock polygon patterns.
        /// </summary>
        public static int StockCount {
            get { return stock.Count; }
        }

        public static Dictionary<ushort, CliffPolygon>.KeyCollection GetAllID() {
            return stock.Keys;
        }

        /// <summary>
        /// Try to get the polygon associated with an ID
        /// returns null if the ID is not registerd.
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public static CliffPolygon TryGetPolygon(ushort id) {
            CliffPolygon v;
            if (stock.TryGetValue(id, out v)) {
                return v;
            } else {
                return null;
            }
        }

        /// <summary>
        /// get the polygon associated with an ID.
        /// throw exception if the ID is not registerd.
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public static CliffPolygon GetPolygon(ushort id) {
            return stock[(ushort)id];
        }

        public static CliffPolygon GetPolygon(int near, int left, int right) {
            return stock[MakePolygonID(near, left, right)];
        }

        public static ushort MakePolygonID(int near, int left, int right) {
            Debug.Assert(near < 15 && left < 15 && right < 15);
            ushort id = PrivateMakePolygonID(near, left, right);
            if (stock.ContainsKey(id)) {
                return id;
            } else {
                throw new IndexOutOfRangeException("No corresponding id.");
            }
        }

        private static ushort PrivateMakePolygonID(int front, int left, int right) {
            return (ushort)((front & 0x000f) + ((right & 0x000f) << 4) + 0x0f00 + ((left & 0x000f) << 12));
        }

        // serialize this object by reference
        public virtual void GetObjectData(SerializationInfo info, StreamingContext context) {
            info.SetType(typeof(ReferenceImpl));
            info.AddValue("id", id);
        }


        [Serializable]
        internal sealed class ReferenceImpl : IObjectReference {
            private ushort id = 0xffff;
            public object GetRealObject(StreamingContext context) {
                return stock[id];
            }
        }

    }
}
