using System;
using System.IO;
using System.Collections;
using System.Reflection;
using nft.util;
using System.Windows.Forms;

namespace nft.framework
{
	/// <summary>
	/// Environment ̊Tv̐łB
	/// </summary>
	public class Directories
	{
		
		private Directories()
		{
		}

		static private string baseDir;
        static private string resDir; // static application resources
		static private string pluginDir;
		static private string dataDir;
		static private string settingDir;
		static private string defaultSaveDir;
		static private string workDir1;

		static public void Initialize(Hashtable args)
		{
			// set application program existing path

			string current = AppDomain.CurrentDomain.SetupInformation.ApplicationBase;
			baseDir = getPath(args,"-B","-BASEDIR", current);
            string appname = GetAppFolderName();
            string appDataDir = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
            appDataDir = Path.Combine(appDataDir, appname);
            string localAppDataDir = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
            localAppDataDir = Path.Combine(localAppDataDir, appname);
            resDir = Path.Combine(baseDir, @"res\");
			GetOrCreate(resDir,false);
            dataDir = Path.Combine(appDataDir, @"data\");
			GetOrCreate(dataDir,false);
            defaultSaveDir = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
			GetOrCreate(defaultSaveDir,false);
            workDir1 = Path.Combine(localAppDataDir, @"work\");
			GetOrCreate(workDir1,false);
            settingDir = getPath(args, "-S", "-SETTINGDIR", Path.Combine(appDataDir, @"settings\"));
            GetOrCreate(settingDir, false);
            pluginDir = getPath(args, "-P", "-PLUGINDIR", Path.Combine(baseDir, @"plugins\"));
            GetOrCreate(pluginDir, false);
		}

        static private string GetAppFolderName() {
            string ver = Application.ProductVersion;
            int i = ver.IndexOf('.',ver.IndexOf('.')+1);
            string ret = Path.Combine(Application.ProductName, ver.Substring(0,i));
            return ret;
        }

		static private string getPath(Hashtable args, string key1, string key2, string defaultPath)
		{
			string path="";
			try
			{
				if( args.ContainsKey(key1) )
				{
					path = args[key1].ToString();
					return GetOrCreate(path,false).FullName;
				}
				else if( args.ContainsKey(key2) )
				{
					path = args[key2].ToString();
					return GetOrCreate(path,false).FullName;
				}
			}
			catch(Exception e)
			{
				string templ = Main.resources["directories.errormsg_on_dir_creation"].stringValue;
				UIUtil.ShowException(string.Format(templ,path),e,UIInformLevel.normal);
			}
			return defaultPath;
		}

        /// <summary>
        /// Utility function to calculate total size of files in the specific directory.
        /// </summary>
        /// <param name="dirInfo"></param>
        /// <returns></returns>
        public static long GetDirectorySize(string dirname) {
            DirectoryInfo info = new DirectoryInfo(dirname);
            return GetDirectorySize(info);
        }

        static private long GetDirectorySize(DirectoryInfo info) {
            long size = 0;
            foreach (FileInfo fi in info.GetFiles()) {
                size += fi.Length;
            }
            foreach (DirectoryInfo di in info.GetDirectories()) {
                size += GetDirectorySize(di);
            }
            return size;
        }

		/// <summary>
		/// returns specified DirectoryInfo, createing if not exist.
		/// (when prompt=true, and if user doesn't allow create directory, returns null.)
		/// </summary>
		/// <param name="dir">full path for the directory</param>
		/// <param name="prompt">show prompt before creating directory</param>
		/// <returns></returns>
		static public DirectoryInfo GetOrCreate(string dir, bool prompt)
		{
			if(!Directory.Exists(dir))
			{
				if(prompt)
				{
					string templ = Main.resources["directories.create_dir_prompt"].stringValue;
					if(!UIUtil.YesNoMessage(string.Format(templ,dir)))
						return null;
				}
				Directory.CreateDirectory(dir);
			}
			return new DirectoryInfo(dir);
		}

		static public string AppBaseDir
		{
			get{ return baseDir; }
		}
        /// <summary>
        /// Directory which contanins static application resources
        /// </summary>
		static public string SystemResourceDir
		{
			get{ return resDir; }
		}
        /// <summary>
        /// Directroy where contains plug-ins
        /// </summary>
		static public string PluginDir
		{
			get{ return pluginDir;	}
		}
        /// <summary>
        /// Drectory where application data will be stored.
        /// Mainly used for cache data of plug-ins.
        /// </summary>
		static public string DataDir
		{
			get{ return dataDir;	}
		}
        /// <summary>
        /// Default game saving directory.
        /// </summary>
		static public string DefaultSaveDir
		{
			get{ return defaultSaveDir;	}
			set{ defaultSaveDir = value; }
		}
        /// <summary>
        /// Directory where application settings are stored.
        /// </summary>
		static public string SettingDir
		{
			get{ return settingDir;	}
			set{ settingDir = value; }
		}
        /// <summary>
        /// Directory where game data will be cached.
        /// </summary>
		static public string WorkDir
		{
			get{ return workDir1;	}
			set{ workDir1 = value; }
		}

        /// <summary>
        /// retruns diretory path under the DataDir, which name based on the type 't'
        /// recommended for cache directory for longlife(game independent) resource data.
        /// </summary>
        /// <param name="t"></param>
        /// <returns></returns>
        static public string MakeSubDataDirPathFor(Type t) {
            string ret = Path.Combine(dataDir, t.FullName);
            return ret;
        }
        /// <summary>
        /// retruns diretory path under the  WorkDir, which name based on the type 't'
        /// recommended for cache directory for some game data.
        /// </summary>
        /// <param name="t"></param>
        /// <returns></returns>
        static public string MakeSubWorkDirPathFor(Type t) {
            string ret = Path.Combine(workDir1, t.FullName);
            return ret;
        }
	}
}
