/*
 * Copyright (C) 2008-2009 GLAD!! (ITO Yoshiichi)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.glad.calendar.holiday;

import java.util.TimeZone;

import jp.sourceforge.glad.calendar.ISOCalendar;

/**
 * 春分の日。
 * 
 * @author GLAD!!
 */
public class VernalEquinoxDay extends Holiday {

    private static final long serialVersionUID = 4381935089551411017L;

    /**
     * オブジェクトを構築します。
     * 
     * @param country   国コード (ISO 3166-1 alpha-2)
     * @param name      名前
     * @param startYear 適用開始年
     * @param endYear   適用終了年
     */
    public VernalEquinoxDay(
            String country, String name, Integer startYear, Integer endYear) {
        super(country, name, "03-??", startYear, endYear);
    }

    /**
     * {@inheritDoc}
     */
    public boolean isAvailableYear(int year) {
        return super.isAvailableYear(year)
                && (1900 <= year && year < 2100);
    }

    /**
     * {@inheritDoc}
     */
    ISOCalendar getISOCalendar(int year, TimeZone zone) {
        if (!isAvailableYear(year)) {
            throw new IllegalArgumentException("year=" + year);
        }
        return new ISOCalendar(year, month, getDay(year), zone);
    }

    /**
     * 日を算出します。
     * 
     * @see <a href="http://ja.wikipedia.org/wiki/%E6%98%A5%E5%88%86%E3%81%AE%E6%97%A5">
     *         春分の日 - Wikipedia</a>
     * @see <a href="http://www.asahi-net.or.jp/~CI5M-NMR/misc/equinox.html">
     *         Vernal Equinox & Autumnal Equinox in Japan</a>
     */
    int getDay(int year) {
        switch (year % 4) {
        case 0:
            if (2092 <= year) return 19;
            if (1960 <= year) return 20;
                              return 21;
        case 1:
            if (1993 <= year) return 20;
                              return 21;
        case 2:
            if (2026 <= year) return 20;
                              return 21;
        default:
            if (2059 <= year) return 20;
            if (1927 <= year) return 21;
                              return 22;
        }
    }

}
