/*
 * Copyright (C) 2008-2009 GLAD!! (ITO Yoshiichi)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.glad.util;

/**
 * 日本語関連のユーティリティ。
 * 
 * @author GLAD!!
 */
public class JapaneseUtils {

    /**
     * 文字列を JIS X 0221 の文字集合の文字列に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * 
     * @param s 変換前の文字列
     * @return 変換後の文字列
     */
    public static String toJISX0221CS(String s) {
        if (s == null || s.length() == 0) {
            return s;
        }
        char[] chars = s.toCharArray();
        for (int i = 0, len = chars.length; i < len; ++i) {
            chars[i] = toJISX0221CS(chars[i]);
        }
        return new String(chars);
    }

    /**
     * 文字を JIS X 0221 の文字集合の文字に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * 
     * @param c 変換前の文字
     * @return 変換後の文字
     */
    public static char toJISX0221CS(char c) {
        switch (c) {
        case 0xFF5E:
            // 区点 1-33 '～'
            // FULL WIDTH TILDE
            // (Windows-31J)
            // -> WAVE DASH
            return 0x301C;
        case 0x2225:
            // 区点 1-34 '∥'
            // PARALLEL TO
            // (Windows-31J, Oracle JA16SJIS)
            // -> DOUBLE VERTICAL LINE
            return 0x2016;
        case 0xFF0D:
            // 区点 1-61 '－'
            // FULL WIDTH HYPHEN-MINUS
            // (Windows-31J, Oracle JA16SJIS)
            // -> MINUS SIGN
            return 0x2212;
        case 0xFFE0:
            // 区点 1-81 '￠'
            // FULL WIDTH CENT SIGN
            // (Windows-31J, Oracle JA16SJIS)
            // -> CENT SIGN
            return 0x00A2;
        case 0xFFE1:
            // 区点 1-82 '￡'
            // FULL WIDTH POUND SIGN
            // (Windows-31J, Oracle JA16SJIS)
            // -> POUND SIGN
            return 0x00A3;
        case 0xFFE2:
            // 区点 2-44 '￢'
            // FULL WIDTH NOT SIGN
            // (Windows-31J, Oracle JA16SJIS)
            // -> NOT SIGN
            return 0x00AC;
        default:
            return c;
        }
    }

    /**
     * 文字列を Windows-31J の文字集合の文字列に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * 
     * @param s 変換前の文字列
     * @return 変換後の文字列
     */
    public static String toWindows31JCS(String s) {
        if (s == null || s.length() == 0) {
            return s;
        }
        char[] chars = s.toCharArray();
        for (int i = 0, len = chars.length; i < len; ++i) {
            chars[i] = toWindows31JCS(chars[i]);
        }
        return new String(chars);
    }

    /**
     * 文字を Windows-31J の文字集合の文字に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * 
     * @param c 変換前の文字
     * @return 変換後の文字
     */
    public static char toWindows31JCS(char c) {
        // * 印のものは置き換えなくても変換可能。
        switch (c) {
        case 0x301C:
            // 区点 1-33 '～'
            // WAVE DASH
            // (JIS X 0221, Oracle JA16SJIS)
            // -> FULL WIDTH TILDE
            return 0xFF5E;
        case 0x2016:
            // 区点 1-34 '∥'
            // DOUBLE VERTICAL LINE
            // (JIS X 0221)
            // -> PARALLEL SIGN
            return 0x2225;
        case 0x2212:
            // 区点 1-61 '－'
            // MINUS SIGN
            // (JIS X 0221)
            // -> FULL WIDTH HYPHEN-MINUS
            return 0xFF0D;
        case 0x00A2:
            // 区点 1-81 '￠'
            // CENT SIGN
            // (JIS X 0221)*
            // -> FULL WIDTH CENT SIGN
            return 0xFFE0;
        case 0x00A3:
            // 区点 1-82 '￡'
            // POUND SIGN
            // (JIS X 0221)*
            // -> FULL WIDTH POUND SIGN
            return 0xFFE1;
        case 0x00AC:
            // 区点 2-44 '￢'
            // NOT SIGN
            //(JIS X 0221)*
            // -> FULL WIDTH NOT SIGN
            return 0xFFE2;
        default:
            return c;
        }
    }

    /**
     * 文字列を Oracle JA31SJIS の文字集合の文字列に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * R9.0.1.4 以降で JA16SJISTILDE を指定した場合は、toWindows31JCS を使用する。
     * 
     * @param s 変換前の文字列
     * @return 変換後の文字列
     */
    public static String toOracleJA16CS(String s) {
        if (s == null || s.length() == 0) {
            return s;
        }
        char[] chars = s.toCharArray();
        for (int i = 0, len = chars.length; i < len; ++i) {
            chars[i] = toOracleJA16CS(chars[i]);
        }
        return new String(chars);
    }

    /**
     * 文字を Oracle JA16SJIS の文字集合の文字に変換する。
     * エンコーディングは UTF-16 のまま変更しない。
     * R9.0.1.4 以降で JA16SJISTILDE を指定した場合は、toWindows31JCS を使用する。
     * 
     * @param c 変換前の文字
     * @return 変換後の文字
     */
    public static char toOracleJA16CS(char c) {
        switch (c) {
        case 0xFF5E:
            // 区点 1-33 '～'
            // FULL WIDTH TILDE
            // (Windows-31J)
            // -> WAVE DASH
            return 0x301C;
        case 0x2016:
            // 区点 1-34 '∥'
            // DOUBLE VERTICAL LINE
            // (JIS X 0221)
            // -> PARALLEL SIGN
            return 0x2225;
        case 0x2212:
            // 区点 1-61 '－'
            // MINUS SIGN
            // (JIS X 0221)
            // -> FULL WIDTH HYPHEN-MINUS
            return 0xFF0D;
        case 0x00A2:
            // 区点 1-81 '￠'
            // CENT SIGN
            // (JIS X 0221)
            // -> FULL WIDTH CENT SIGN
            return 0xFFE0;
        case 0x00A3:
            // 区点 1-82 '￡'
            // POUND SIGN
            // (JIS X 0221)
            // -> FULL WIDTH POUND SIGN
            return 0xFFE1;
        case 0x00AC:
            // 区点 2-44 '￢'
            // NOT SIGN
            //(JIS X 0221)
            // -> FULL WIDTH NOT SIGN
            return 0xFFE2;
        default:
            return c;
        }
    }

}
