package jp.sourceforge.glad.calendar.holiday;

import jp.sourceforge.glad.calendar.CalendarConsts;
import jp.sourceforge.glad.calendar.ISOCalendar;
import junit.framework.TestCase;

/**
 * {@link Holidays} の単体テスト。
 * 
 * @author GLAD!!
 */
public class HolidaysTest extends TestCase {

    Holidays holidays;

    protected void setUp() throws Exception {
        holidays = Holidays.getInstance();
    }

    protected void tearDown() throws Exception {
        holidays = null;
    }

    // 2008年はすべての祝日についてテスト。それ以外は変更点のみ。

    /**
     * 。「国民の祝日に関する法律」制定。(1948-07-20)
     */
    public void testHoliday1948() {
        // 最初の「国民の祝日」。
        assertNotHoliday(1948, 9, 22, 23, 59, 59, 999);
        assertHoliday("秋分の日", 1948, 9, 23,  0,  0,  0,   0);
        assertHoliday("秋分の日", 1948, 9, 23, 23, 59, 59, 999);
        assertNotHoliday(1948, 9, 24, 0, 0, 0, 0);
        
        assertNotHoliday(1948, 11, 2, 23, 59, 59, 999);
        assertHoliday("文化の日", 1948, 11, 3,  0,  0,  0,   0);
        assertHoliday("文化の日", 1948, 11, 3, 23, 59, 59, 999);
        assertNotHoliday(1948, 11, 4, 0, 0, 0, 0);
        
        assertNotHoliday(1948, 11, 22, 23, 59, 59, 999);
        assertHoliday("勤労感謝の日", 1948, 11, 23,  0,  0,  0,   0);
        assertHoliday("勤労感謝の日", 1948, 11, 23, 23, 59, 59, 999);
        assertNotHoliday(1948, 11, 24, 0, 0, 0, 0);
    }

    public void testHoliday1949() {
        assertNotHoliday(1949, 12, 31, 23, 59, 59, 999);
        assertHoliday("元日", 1949, 1, 1,  0,  0,  0,   0);
        assertHoliday("元日", 1949, 1, 1, 23, 59, 59, 999);
        assertNotHoliday(1949, 1, 2, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 1, 14, 23, 59, 59, 999);
        assertHoliday("成人の日", 1949, 1, 15,  0,  0,  0,   0);
        assertHoliday("成人の日", 1949, 1, 15, 23, 59, 59, 999);
        assertNotHoliday(1949, 1, 16, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 3, 20, 23, 59, 59, 999);
        assertHoliday("春分の日", 1949, 3, 21,  0,  0,  0,   0);
        assertHoliday("春分の日", 1949, 3, 21, 23, 59, 59, 999);
        assertNotHoliday(1949, 3, 22, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 4, 28, 23, 59, 59, 999);
        assertHoliday("天皇誕生日", 1949, 4, 29,  0,  0,  0,   0);
        assertHoliday("天皇誕生日", 1949, 4, 29, 23, 59, 59, 999);
        assertNotHoliday(1949, 4, 30, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 1949, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 1949, 5, 3, 23, 59, 59, 999);
        assertNotHoliday(1949, 5, 4, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 5, 4, 23, 59, 59, 999);
        assertHoliday("こどもの日", 1949, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 1949, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(1949, 5, 6, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 9, 22, 23, 59, 59, 999);
        assertHoliday("秋分の日", 1949, 9, 23,  0,  0,  0,   0);
        assertHoliday("秋分の日", 1949, 9, 23, 23, 59, 59, 999);
        assertNotHoliday(1949, 9, 24, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 11, 2, 23, 59, 59, 999);
        assertHoliday("文化の日", 1949, 11, 3,  0,  0,  0,   0);
        assertHoliday("文化の日", 1949, 11, 3, 23, 59, 59, 999);
        assertNotHoliday(1949, 11, 4, 0, 0, 0, 0);
        
        assertNotHoliday(1949, 11, 22, 23, 59, 59, 999);
        assertHoliday("勤労感謝の日", 1949, 11, 23,  0,  0,  0,   0);
        assertHoliday("勤労感謝の日", 1949, 11, 23, 23, 59, 59, 999);
        assertNotHoliday(1949, 11, 24, 0, 0, 0, 0);
    }

    public void testHoliday1965() {
        assertNotHoliday(1965, 10, 10);
    }

    public void testHoliday1966() {
        assertNotHoliday(1966, 2, 11);
        
        // 最初の「体育の日」。
        assertNotHoliday(1966, 10, 9, 23, 59, 59, 999);
        assertHoliday("体育の日", 1966, 10, 10,  0,  0,  0,   0);
        assertHoliday("体育の日", 1966, 10, 10, 23, 59, 59, 999);
        assertNotHoliday(1966, 10, 11, 0, 0, 0, 0);
    }

    public void testHoliday1967() {
        // 最初の「建国記念の日」。
        assertNotHoliday(1967, 2, 10, 23, 59, 59, 999);
        assertHoliday("建国記念の日", 1967, 2, 11,  0,  0,  0,   0);
        assertHoliday("建国記念の日", 1967, 2, 11, 23, 59, 59, 999);
        assertNotHoliday(1967, 2, 12, 0, 0, 0, 0);
    }

    /**
     * 「振替休日」制定。(1973-04-12)
     */
    public void testHoliday1973() {
        assertNotHoliday(1973, 2, 10, 23, 59, 59, 999);
        assertHoliday("建国記念の日", 1973, 2, 11,  0,  0,  0,   0);
        assertHoliday("建国記念の日", 1973, 2, 11, 23, 59, 59, 999);
        assertNotHoliday(1973, 2, 12, 0, 0, 0, 0);
        
        // 1973-02-12 は「振替休日」ではない。
        assertEquals(CalendarConsts.MONDAY, 
                new ISOCalendar(1973, 2, 12).getDayOfWeek());
        
        assertNotHoliday(1973, 4, 28, 23, 59, 59, 999);
        assertHoliday("天皇誕生日", 1973, 4, 29,  0,  0,  0,   0);
        assertHoliday("天皇誕生日", 1973, 4, 29, 23, 59, 59, 999);
        
        // 最初の「振替休日」。
        assertHoliday("振替休日", 1973, 4, 30,  0,  0,  0,   0);
        assertHoliday("振替休日", 1973, 4, 30, 23, 59, 59, 999);
        assertNotHoliday(1973, 5, 1, 0, 0, 0, 0);
        
    }

    public void testHoliday1985() {
        assertNotHoliday(1985, 5, 4);
    }

    /**
     * 「国民の休日」制定。(1985-12-27)
     */
    public void testHoliday1986() {
        assertNotHoliday(1986, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 1986, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 1986, 5, 3, 23, 59, 59, 999);
        assertNotHoliday(1986, 5, 4, 0, 0, 0, 0);
        
        assertEquals(CalendarConsts.SUNDAY,
                new ISOCalendar(1986, 5, 4).getDayOfWeek());
        
        assertNotHoliday(1986, 5, 4, 23, 59, 59, 999);
        assertHoliday("こどもの日", 1986, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 1986, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(1986, 5, 6, 0, 0, 0, 0);
    }

    public void testHoliday1987() {
        assertNotHoliday(1987, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 1987, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 1987, 5, 3, 23, 59, 59, 999);
        
        assertHoliday("振替休日", 1987, 5, 4,  0,  0,  0,   0);
        assertHoliday("振替休日", 1987, 5, 4, 23, 59, 59, 999);
        
        assertHoliday("こどもの日", 1987, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 1987, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(1987, 5, 6, 0, 0, 0, 0);
    }

    /**
     * 「国民の休日」開始。
     */
    public void testHoliday1988() {
        // 最後の 04-29 の「天皇誕生日」。
        assertNotHoliday(1988, 4, 28, 23, 59, 59, 999);
        assertHoliday("天皇誕生日", 1988, 4, 29,  0,  0,  0,   0);
        assertHoliday("天皇誕生日", 1988, 4, 29, 23, 59, 59, 999);
        assertNotHoliday(1988, 4, 30, 0, 0, 0, 0);
        
        assertNotHoliday(1988, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 1988, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 1988, 5, 3, 23, 59, 59, 999);
        
        // 最初の5月の「国民の休日」。
        assertHoliday("国民の休日", 1988, 5, 4,  0,  0,  0,   0);
        assertHoliday("国民の休日", 1988, 5, 4, 23, 59, 59, 999);
        
        assertHoliday("こどもの日", 1988, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 1988, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(1988, 5, 6, 0, 0, 0, 0);
        
        assertNotHoliday(1988, 12, 23);
    }

    /**
     * 平成元年。
     */
    public void testHoliday1989() {
        // 最初の 04-29 の「みどりの日」。
        assertNotHoliday(1989, 4, 28, 23, 59, 59, 999);
        assertHoliday("みどりの日", 1989, 4, 29,  0,  0,  0,   0);
        assertHoliday("みどりの日", 1989, 4, 29, 23, 59, 59, 999);
        assertNotHoliday(1989, 4, 30, 0, 0, 0, 0);
        
        // 最初の 12-23 の「天皇誕生日」。
        assertNotHoliday(1989, 12, 22, 23, 59, 59, 999);
        assertHoliday("天皇誕生日", 1989, 12, 23,  0,  0,  0,   0);
        assertHoliday("天皇誕生日", 1989, 12, 23, 23, 59, 59, 999);
        assertNotHoliday(1989, 12, 24, 0, 0, 0, 0);
    }

    public void testHoliday1995() {
        assertNotHoliday(1995, 7, 20);
    }

    public void testHoliday1996() {
        // 最初の「海の日」。
        assertNotHoliday(1996, 7, 19, 23, 59, 59, 999);
        assertHoliday("海の日", 1996, 7, 20,  0,  0,  0,   0);
        assertHoliday("海の日", 1996, 7, 20, 23, 59, 59, 999);
        assertNotHoliday(1996, 7, 21, 0, 0, 0, 0);
    }

    public void testHoliday1999() {
        // 最後の 01-15 固定の「成人の日」。
        assertNotHoliday(1999, 1, 14, 23, 59, 59, 999);
        assertHoliday("成人の日", 1999, 1, 15,  0,  0,  0,   0);
        assertHoliday("成人の日", 1999, 1, 15, 23, 59, 59, 999);
        assertNotHoliday(1999, 1, 16, 0, 0, 0, 0);
        
        // 最後の 10-10 固定の「体育の日」。
        assertNotHoliday(1999, 10, 9, 23, 59, 59, 999);
        assertHoliday("体育の日", 1999, 10, 10,  0,  0,  0,   0);
        assertHoliday("体育の日", 1999, 10, 10, 23, 59, 59, 999);
        
        assertHoliday("振替休日", 1999, 10, 11,  0,  0,  0,   0);
        assertHoliday("振替休日", 1999, 10, 11, 23, 59, 59, 999);
        assertNotHoliday(1999, 10, 12, 0, 0, 0, 0);
    }

    /**
     * 「ハッピーマンデー」制定。
     */
    public void testHoliday2000() {
        // ハッピーマンデー「成人の日」。
        assertNotHoliday(2000, 1, 9, 23, 59, 59, 999);
        assertHoliday("成人の日", 2000, 1, 10,  0,  0,  0,   0);
        assertHoliday("成人の日", 2000, 1, 10, 23, 59, 59, 999);
        assertNotHoliday(2000, 1, 11, 0, 0, 0, 0);
        
        // ハッピーマンデー「体育の日」。
        assertNotHoliday(2000, 10, 8, 23, 59, 59, 999);
        assertHoliday("体育の日", 2000, 10, 9,  0,  0,  0,   0);
        assertHoliday("体育の日", 2000, 10, 9, 23, 59, 59, 999);
        assertNotHoliday(2000, 10, 10, 0, 0, 0, 0);
    }

    public void testHoliday2002() {
        // 最後の 07-20 固定の「海の日」。
        assertNotHoliday(2002, 7, 19, 23, 59, 59, 999);
        assertHoliday("海の日", 2002, 7, 20,  0,  0,  0,   0);
        assertHoliday("海の日", 2002, 7, 20, 23, 59, 59, 999);
        assertNotHoliday(2002, 7, 21, 0, 0, 0, 0);
        
        // 最後の 09-15 固定の「敬老の日」。
        assertNotHoliday(2002, 9, 14, 23, 59, 59, 999);
        assertHoliday("敬老の日", 2002, 9, 15,  0,  0,  0,   0);
        assertHoliday("敬老の日", 2002, 9, 15, 23, 59, 59, 999);
        
        assertHoliday("振替休日", 2002, 9, 16,  0,  0,  0,   0);
        assertHoliday("振替休日", 2002, 9, 16, 23, 59, 59, 999);
        assertNotHoliday(2002, 9, 17, 0, 0, 0, 0);
    }

    public void testHoliday2003() {
        // ハッピーマンデー「海の日」。
        assertNotHoliday(2003, 7, 20, 23, 59, 59, 999);
        assertHoliday("海の日", 2003, 7, 21,  0,  0,  0,   0);
        assertHoliday("海の日", 2003, 7, 21, 23, 59, 59, 999);
        assertNotHoliday(2003, 7, 22, 0, 0, 0, 0);
        
        // ハッピーマンデー「敬老の日」。
        assertNotHoliday(2003, 9, 14, 23, 59, 59, 999);
        assertHoliday("敬老の日", 2003, 9, 15,  0,  0,  0,   0);
        assertHoliday("敬老の日", 2003, 9, 15, 23, 59, 59, 999);
        assertNotHoliday(2003, 9, 16, 0, 0, 0, 0);
    }

    public void testHoliday2006() {
        // 最後の 04-29 の「みどりの日」。
        assertNotHoliday(2006, 4, 28, 23, 59, 59, 999);
        assertHoliday("みどりの日", 2006, 4, 29,  0,  0,  0,   0);
        assertHoliday("みどりの日", 2006, 4, 29, 23, 59, 59, 999);
        assertNotHoliday(2006, 4, 30, 0, 0, 0, 0);
        
        assertNotHoliday(2006, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 2006, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 2006, 5, 3, 23, 59, 59, 999);
        
        // 最後の5月の「国民の休日」。
        assertHoliday("国民の休日", 2006, 5, 4,  0,  0,  0,   0);
        assertHoliday("国民の休日", 2006, 5, 4, 23, 59, 59, 999);
        
        assertHoliday("こどもの日", 2006, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 2006, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(2006, 5, 6, 0, 0, 0, 0);
    }

    public void testHoliday2007() {
        // 最初の「昭和の日」。
        assertNotHoliday(2007, 4, 28, 23, 59, 59, 999);
        assertHoliday("昭和の日", 2007, 4, 29,  0,  0,  0,   0);
        assertHoliday("昭和の日", 2007, 4, 29, 23, 59, 59, 999);
        
        assertHoliday("振替休日", 2007, 4, 30,  0,  0,  0,   0);
        assertHoliday("振替休日", 2007, 4, 30, 23, 59, 59, 999);
        assertNotHoliday(2007, 5, 1, 0, 0, 0, 0);
        
        assertNotHoliday(2007, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 2007, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 2007, 5, 3, 23, 59, 59, 999);
        
        // 最初の 05-04 の「みどりの日」。
        assertHoliday("みどりの日", 2007, 5, 4,  0,  0,  0,   0);
        assertHoliday("みどりの日", 2007, 5, 4, 23, 59, 59, 999);
        
        assertHoliday("こどもの日", 2007, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 2007, 5, 5, 23, 59, 59, 999);
        assertNotHoliday(2007, 5, 6, 0, 0, 0, 0);
    }

    public void testHoliday2008() {
        assertNotHoliday(2007, 12, 31, 23, 59, 59, 999);
        assertHoliday("元日", 2008, 1, 1,  0,  0,  0,   0);
        assertHoliday("元日", 2008, 1, 1, 23, 59, 59, 999);
        assertNotHoliday(2008, 1, 2, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 1, 13, 23, 59, 59, 999);
        assertHoliday("成人の日", 2008, 1, 14,  0,  0,  0,   0);
        assertHoliday("成人の日", 2008, 1, 14, 23, 59, 59, 999);
        assertNotHoliday(2008, 1, 15, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 2, 10, 23, 59, 59, 999);
        assertHoliday("建国記念の日", 2008, 2, 11,  0,  0,  0,   0);
        assertHoliday("建国記念の日", 2008, 2, 11, 23, 59, 59, 999);
        assertNotHoliday(2008, 2, 12, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 3, 19, 23, 59, 59, 999);
        assertHoliday("春分の日", 2008, 3, 20,  0,  0,  0,   0);
        assertHoliday("春分の日", 2008, 3, 20, 23, 59, 59, 999);
        assertNotHoliday(2008, 3, 21, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 4, 28, 23, 59, 59, 999);
        assertHoliday("昭和の日", 2008, 4, 29,  0,  0,  0,   0);
        assertHoliday("昭和の日", 2008, 4, 29, 23, 59, 59, 999);
        assertNotHoliday(2008, 4, 30, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 5, 2, 23, 59, 59, 999);
        assertHoliday("憲法記念日", 2008, 5, 3,  0,  0,  0,   0);
        assertHoliday("憲法記念日", 2008, 5, 3, 23, 59, 59, 999);
        
        assertHoliday("みどりの日", 2008, 5, 4,  0,  0,  0,   0);
        assertHoliday("みどりの日", 2008, 5, 4, 23, 59, 59, 999);
        
        assertHoliday("こどもの日", 2008, 5, 5,  0,  0,  0,   0);
        assertHoliday("こどもの日", 2008, 5, 5, 23, 59, 59, 999);
        
        // 最初の火曜日の「振替休日」。
        assertHoliday("振替休日", 2008, 5, 6,  0,  0,  0,   0);
        assertHoliday("振替休日", 2008, 5, 6, 23, 59, 59, 999);
        assertNotHoliday(2008, 5, 7, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 7, 20, 23, 59, 59, 999);
        assertHoliday("海の日", 2008, 7, 21,  0,  0,  0,   0);
        assertHoliday("海の日", 2008, 7, 21, 23, 59, 59, 999);
        assertNotHoliday(2008, 7, 22, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 9, 14, 23, 59, 59, 999);
        assertHoliday("敬老の日", 2008, 9, 15,  0,  0,  0,   0);
        assertHoliday("敬老の日", 2008, 9, 15, 23, 59, 59, 999);
        assertNotHoliday(2008, 9, 16, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 9, 22, 23, 59, 59, 999);
        assertHoliday("秋分の日", 2008, 9, 23,  0,  0,  0,   0);
        assertHoliday("秋分の日", 2008, 9, 23, 23, 59, 59, 999);
        assertNotHoliday(2008, 9, 24, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 10, 12, 23, 59, 59, 999);
        assertHoliday("体育の日", 2008, 10, 13,  0,  0,  0,   0);
        assertHoliday("体育の日", 2008, 10, 13, 23, 59, 59, 999);
        assertNotHoliday(2008, 10, 14, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 11, 2, 23, 59, 59, 999);
        assertHoliday("文化の日", 2008, 11, 3,  0,  0,  0,   0);
        assertHoliday("文化の日", 2008, 11, 3, 23, 59, 59, 999);
        assertNotHoliday(2008, 11, 4, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 11, 22, 23, 59, 59, 999);
        assertHoliday("勤労感謝の日", 2008, 11, 23,  0,  0,  0,   0);
        assertHoliday("勤労感謝の日", 2008, 11, 23, 23, 59, 59, 999);
        
        assertHoliday("振替休日", 2008, 11, 24,  0,  0,  0,   0);
        assertHoliday("振替休日", 2008, 11, 24, 23, 59, 59, 999);
        assertNotHoliday(2008, 11, 25, 0, 0, 0, 0);
        
        assertNotHoliday(2008, 12, 22, 23, 59, 59, 999);
        assertHoliday("天皇誕生日", 2008, 12, 23,  0,  0,  0,   0);
        assertHoliday("天皇誕生日", 2008, 12, 23, 23, 59, 59, 999);
        assertNotHoliday(2008, 12, 24, 0, 0, 0, 0);
    }

    public void testHoliday2009() {
        assertNotHoliday(2009, 9, 20, 23, 59, 59, 999);
        assertHoliday("敬老の日", 2009, 9, 21,  0,  0,  0,   0);
        assertHoliday("敬老の日", 2009, 9, 21, 23, 59, 59, 999);
        
        // 最初の9月の「国民の休日」。
        assertHoliday("国民の休日", 2009, 9, 22,  0,  0,  0,   0);
        assertHoliday("国民の休日", 2009, 9, 22, 23, 59, 59, 999);
        
        assertHoliday("秋分の日", 2009, 9, 23,  0,  0,  0,   0);
        assertHoliday("秋分の日", 2009, 9, 23, 23, 59, 59, 999);
        assertNotHoliday(2009, 9, 24, 0, 0, 0, 0);
    }

    void assertHoliday(String name,
            int year, int month, int day,
            int hour, int minute, int second, int millis) {
        assertEquals(name, holidays.getHoliday(new ISOCalendar(
                year, month, day, hour, minute, second, millis)).getName());
    }

    void assertNotHoliday(
            int year, int month, int day) {
        assertFalse(holidays.isHoliday(new ISOCalendar(year, month, day)));
    }

    void assertNotHoliday(
            int year, int month, int day,
            int hour, int minute, int second, int millis) {
        assertFalse(holidays.isHoliday(new ISOCalendar(
                year, month, day, hour, minute, second, millis)));
    }

}
