package jp.sourceforge.glad.calendar.joda.format;

import java.util.Locale;

import jp.sourceforge.glad.calendar.joda.chrono.JapaneseChronology;
import junit.framework.TestCase;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormatter;

public class DateTimeFormatTest extends TestCase {

    static final DateTimeZone UTC = DateTimeZone.UTC;
    static final DateTimeZone JST = DateTimeZone.forID("Asia/Tokyo");

    public void testCalendarDate() {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(
                "yyyy-MM-dd'T'HH:mm:ss.SSSzz");
        DateTime dateTime = new DateTime(2008, 1, 23, 1, 12, 23, 123, JST);
        assertEquals("2008-01-23T01:12:23.123+09:00", fmt.print(dateTime));
        dateTime = dateTime.withZone(UTC);
        assertEquals("2008-01-22T16:12:23.123Z", fmt.print(dateTime));
    }

    public void testCalendarDate_JapaneseLocale() {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(
                "y年M月d日 (E) aK時m分s秒 (ZZZ)")
            .withLocale(Locale.JAPAN);
        DateTime dateTime = new DateTime(2008, 1, 23, 12, 23, 34, 123, JST);
        assertEquals("2008年1月23日 (水) 午後0時23分34秒 (Asia/Tokyo)",
                fmt.print(dateTime));
    }

    public void testOrdinalDate() {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(
                "yyyy-DDD'T'HH:mm:ss.SSSzz");
        DateTime dateTime = new DateTime(2008, 1, 23, 1, 12, 23, 123, JST);
        assertEquals("2008-023T01:12:23.123+09:00", fmt.print(dateTime));
        dateTime = dateTime.withZone(UTC);
        assertEquals("2008-022T16:12:23.123Z", fmt.print(dateTime));
    }

    public void testWeekDate() {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(
                "xxxx-'W'ww-e'T'HH:mm:ss.SSSzz");
        DateTime dateTime = new DateTime(2008, 1, 23, 1, 12, 23, 123, JST);
        assertEquals("2008-W04-3T01:12:23.123+09:00", fmt.print(dateTime));
        dateTime = dateTime.withZone(UTC);
        assertEquals("2008-W04-2T16:12:23.123Z", fmt.print(dateTime));
        
        // 年末年始がともに前年の週に含まれる場合。
        DateTime dt20051231 = new DateTime(2005, 12, 31, 23, 59, 59, 999, JST);
        assertEquals("2005-W52-6T23:59:59.999+09:00", fmt.print(dt20051231));
        DateTime dt20060101 = new DateTime(2006, 1, 1, 0, 0, 0, 0, JST);
        assertEquals("2005-W52-7T00:00:00.000+09:00", fmt.print(dt20060101));
        // 年末年始がそれぞれの年の週に含まれる場合。
        DateTime dt20061231 = new DateTime(2006, 12, 31, 23, 59, 59, 999, JST);
        assertEquals("2006-W52-7T23:59:59.999+09:00", fmt.print(dt20061231));
        DateTime dt20070101 = new DateTime(2007, 1, 1, 0, 0, 0, 0, JST);
        assertEquals("2007-W01-1T00:00:00.000+09:00", fmt.print(dt20070101));
        // 年末年始がともに翌年の週に含まれる場合。
        DateTime dt20071231 = new DateTime(2007, 12, 31, 23, 59, 59, 999, JST);
        assertEquals("2008-W01-1T23:59:59.999+09:00", fmt.print(dt20071231));
        DateTime dt20080101 = new DateTime(2008, 1, 1, 0, 0, 0, 0, JST);
        assertEquals("2008-W01-2T00:00:00.000+09:00", fmt.print(dt20080101));
    }

    public void testJISDate() {
        DateTimeFormatter fmt = DateTimeFormat
            .forPattern("GYY.MM.dd")
            .withChronology(JapaneseChronology.getInstance());
        DateTime dateTime = new DateTime(2008, 1, 23, 1, 12, 23, 123, JST);
        assertEquals("H20.01.23", fmt.print(dateTime));
        dateTime = dateTime.withZone(UTC);
        assertEquals("H20.01.23", fmt.print(dateTime));
    }

    public void testJISDateTime() {
        DateTimeFormatter fmt = DateTimeFormat
            .forPattern("GYY.MM.dd'T'HH:mm:ss.SSSzz")
            .withChronology(JapaneseChronology.getInstance());
        DateTime dateTime = new DateTime(2008, 1, 23, 1, 12, 23, 123, JST);
        assertEquals("H20.01.23T01:12:23.123+09:00", fmt.print(dateTime));
        fmt = fmt.withZone(UTC);
        assertEquals("H20.01.22T16:12:23.123Z", fmt.print(dateTime));
    }

    public void testJISDate_JapaneseLocale() {
        DateTimeFormatter fmt = DateTimeFormat.forPattern(
                "GGGGYYYYY年M月d日 (EEEE) aK時m分s秒 (ZZZ)")
            .withChronology(JapaneseChronology.getInstance())
            .withLocale(Locale.JAPAN);
        DateTime dateTime = new DateTime(2008, 1, 23, 12, 23, 34, 123, JST);
        assertEquals("平成20年1月23日 (水曜日) 午後0時23分34秒 (Asia/Tokyo)",
                fmt.print(dateTime));
        assertEquals("平成元年1月23日 (月曜日) 午後0時23分34秒 (Asia/Tokyo)",
                fmt.print(new DateTime(1989, 1, 23, 12, 23, 34, 123, JST)));
        assertEquals("昭和2年1月23日 (日曜日) 午後0時23分34秒 (Asia/Tokyo)",
                fmt.print(new DateTime(1927, 1, 23, 12, 23, 34, 123, JST)));
    }

}
