#
# test_LinearGraph.rb
#
# Copyright (C) 2010 GLAD!! (ITO Yoshiichi)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. See the License for the specific language
# governing permissions and limitations under the License.
#

require 'test/unit'
require 'LinearGraph'

class LinearGraphTest < Test::Unit::TestCase

  def test_L4
    assert_equal(3, 1 ^ 2)
    
    # --------
    assert_equal(2, 1 ^ 3)
    assert_equal(1, 2 ^ 3)
    
    # --------
    graph = LinearGraph.new([1, 2])
    assert_equal([1, 2, 3], graph.values)
    assert_equal([1, 2], graph.points)
    assert_equal([3], graph.lines)
    assert_equal(4, graph.level)
    assert_equal(graph, LinearGraph.new([1, 2]))
    assert_equal('[1, 2; 3]', graph.to_s)
  end

  def test_L8
    assert_equal(3, 1 ^ 2)
    
    assert_equal(5, 1 ^ 4)
    assert_equal(6, 2 ^ 4)
    assert_equal(7, 3 ^ 4)
    assert_equal(7, 2 ^ 5)
    assert_equal(7, 1 ^ 6)
    
    # --------
    assert_equal(4, 1 ^ 5)
    assert_equal(6, 1 ^ 7)
    assert_equal(4, 2 ^ 6)
    assert_equal(5, 2 ^ 7)
    
    assert_equal(6, 3 ^ 5)
    assert_equal(5, 3 ^ 6)
    assert_equal(3, 5 ^ 6)
    
    # --------
    graph = LinearGraph.new([1, 2, 4])
    assert_equal((1..7).to_a, graph.values)
    assert_equal([1, 2, 4], graph.points)
    assert_equal([3, 5, 6, 7], graph.lines)
    assert_equal(8, graph.level)
    assert_equal(graph, LinearGraph.new([1, 2, 4]))
    assert_not_equal(graph, LinearGraph.new([1, 2]))
    assert_equal('[1, 2, 4; 3, 5, 6, 7]', graph.to_s)
    
    assert_equal((1..7).to_a,
        LinearGraph.new([2, 4], LinearGraph.new([1])).values)
    assert_equal((1..7).to_a,
        LinearGraph.new([4], LinearGraph.new([1, 2])).values)
  end

  def test_L16
    assert_equal( 3, 1 ^  2)
    
    assert_equal( 5, 1 ^  4)
    assert_equal( 6, 2 ^  4)
    assert_equal( 7, 3 ^  4)
    assert_equal( 7, 2 ^  5)
    assert_equal( 7, 1 ^  6)
    
    assert_equal( 9, 1 ^  8)
    assert_equal(10, 2 ^  8)
    assert_equal(11, 3 ^  8)
    assert_equal(11, 2 ^  9)
    assert_equal(11, 1 ^ 10)
    
    assert_equal(12, 4 ^  8)
    assert_equal(13, 5 ^  8)
    assert_equal(13, 4 ^  9)
    assert_equal(13, 1 ^ 12)
    
    assert_equal(14, 6 ^  8)
    assert_equal(14, 4 ^ 10)
    assert_equal(14, 2 ^ 12)
    
    assert_equal(15, 7 ^  8)
    assert_equal(15, 4 ^ 11)
    assert_equal(15, 2 ^ 13)
    assert_equal(15, 1 ^ 14)
    
    # --------
    graph = LinearGraph.new([1, 2, 4, 8])
    assert_equal((1..15).to_a, graph.values)
    assert_equal([1, 2, 4, 8], graph.points)
    assert_equal([3, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15], graph.lines)
    assert_equal(16, graph.level)
    assert_equal(graph, LinearGraph.new([1, 2, 4, 8]))
    assert_not_equal(graph, LinearGraph.new([1, 2]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4]))
    assert_equal('[1, 2, 4, 8; 3, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15]',
        graph.to_s)
    
    assert_equal((1..15).to_a,
        LinearGraph.new([2, 4, 8], LinearGraph.new([1])).values)
    assert_equal((1..15).to_a,
        LinearGraph.new([4, 8], LinearGraph.new([1, 2])).values)
    assert_equal((1..15).to_a,
        LinearGraph.new([8], LinearGraph.new([1, 2, 4])).values)
  end

  def test_L16_4_5
    assert_equal( 3, 1 ^  2)
    assert_equal(12, 4 ^  8)
    assert_equal(15, 5 ^ 10)
    assert_equal(13, 6 ^ 11)
    assert_equal(14, 7 ^  9)
    
    # --------
    assert_equal([ 3], LinearGraph.new([1,  2]).lines)
    assert_equal([12], LinearGraph.new([4,  8]).lines)
    assert_equal([15], LinearGraph.new([5, 10]).lines)
    assert_equal([13], LinearGraph.new([6, 11]).lines)
    assert_equal([14], LinearGraph.new([7,  9]).lines)
  end

  def test_L32
    graph = LinearGraph.new([1, 2, 4, 8, 16])
    assert_equal((1..31).to_a, graph.values)
    assert_equal([1, 2, 4, 8, 16], graph.points)
    assert_equal((1..31).to_a - graph.points, graph.lines)
    assert_equal(32, graph.level)
    assert_equal(graph, LinearGraph.new([1, 2, 4, 8, 16]))
    assert_not_equal(graph, LinearGraph.new([1, 2]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8]))
    
    assert_equal((1..31).to_a,
        LinearGraph.new([2, 4, 8, 16], LinearGraph.new([1])).values)
    assert_equal((1..31).to_a,
        LinearGraph.new([4, 8, 16], LinearGraph.new([1, 2])).values)
    assert_equal((1..31).to_a,
        LinearGraph.new([8, 16], LinearGraph.new([1, 2, 4])).values)
    assert_equal((1..31).to_a,
        LinearGraph.new([16], LinearGraph.new([1, 2, 4, 8])).values)
  end

  def test_L32_8_1_4_8
    assert_equal(24,  8 ^ 16)
    assert_equal(26,  9 ^ 19)
    assert_equal(30, 10 ^ 20)
    assert_equal(28, 11 ^ 23)
    assert_equal(29, 12 ^ 17)
    assert_equal(31, 13 ^ 18)
    assert_equal(27, 14 ^ 21)
    assert_equal(25, 15 ^ 22)
    
    # --------
    assert_equal([24], LinearGraph.new([ 8, 16]).lines)
    assert_equal([26], LinearGraph.new([ 9, 19]).lines)
    assert_equal([30], LinearGraph.new([10, 20]).lines)
    assert_equal([28], LinearGraph.new([11, 23]).lines)
    assert_equal([29], LinearGraph.new([12, 17]).lines)
    assert_equal([31], LinearGraph.new([13, 18]).lines)
    assert_equal([27], LinearGraph.new([14, 21]).lines)
    assert_equal([25], LinearGraph.new([15, 22]).lines)
  end

  def test_L64
    graph = LinearGraph.new([1, 2, 4, 8, 16, 32])
    assert_equal((1..63).to_a, graph.values)
    assert_equal([1, 2, 4, 8, 16, 32], graph.points)
    assert_equal((1..63).to_a - graph.points, graph.lines)
    assert_equal(64, graph.level)
    assert_equal(graph, LinearGraph.new([1, 2, 4, 8, 16, 32]))
    assert_not_equal(graph, LinearGraph.new([1, 2]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8, 16]))
    
    assert_equal((1..63).to_a,
        LinearGraph.new([2, 4, 8, 16, 32], LinearGraph.new([1])).values)
    assert_equal((1..63).to_a,
        LinearGraph.new([4, 8, 16, 32], LinearGraph.new([1, 2])).values)
    assert_equal((1..63).to_a,
        LinearGraph.new([8, 16, 32], LinearGraph.new([1, 2, 4])).values)
    assert_equal((1..63).to_a,
        LinearGraph.new([16, 32], LinearGraph.new([1, 2, 4, 8])).values)
    assert_equal((1..63).to_a,
        LinearGraph.new([32], LinearGraph.new([1, 2, 4, 8, 16])).values)
  end

  def test_L64_8_9
    assert_equal([ 3,  5,  6,  7], LinearGraph.new([ 1,  2,  4]).lines)
    assert_equal([24, 40, 48, 56], LinearGraph.new([ 8, 16, 32]).lines)
    assert_equal([27, 45, 54, 63], LinearGraph.new([ 9, 18, 36]).lines)
    assert_equal([30, 41, 55, 61], LinearGraph.new([10, 20, 35]).lines)
    assert_equal([29, 44, 49, 58], LinearGraph.new([11, 22, 39]).lines)
    assert_equal([31, 42, 53, 57], LinearGraph.new([12, 19, 38]).lines)
    assert_equal([28, 47, 51, 62], LinearGraph.new([13, 17, 34]).lines)
    assert_equal([25, 43, 50, 60], LinearGraph.new([14, 23, 37]).lines)
    assert_equal([26, 46, 52, 59], LinearGraph.new([15, 21, 33]).lines)
  end

  def test_L128
    graph = LinearGraph.new([1, 2, 4, 8, 16, 32, 64])
    assert_equal((1..127).to_a, graph.values)
    assert_equal([1, 2, 4, 8, 16, 32, 64], graph.points)
    assert_equal((1..127).to_a - graph.points, graph.lines)
    assert_equal(graph, LinearGraph.new([1, 2, 4, 8, 16, 32, 64]))
    assert_equal(128, graph.level)
    assert_not_equal(graph, LinearGraph.new([1, 2]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8, 16]))
    assert_not_equal(graph, LinearGraph.new([1, 2, 4, 8, 16, 32]))
    
    assert_equal((1..127).to_a,
        LinearGraph.new([2, 4, 8, 16, 32, 64], LinearGraph.new([1])).values)
    assert_equal((1..127).to_a,
        LinearGraph.new([4, 8, 16, 32, 64], LinearGraph.new([1, 2])).values)
    assert_equal((1..127).to_a,
        LinearGraph.new([8, 16, 32, 64], LinearGraph.new([1, 2, 4])).values)
    assert_equal((1..127).to_a,
        LinearGraph.new([16, 32, 64], LinearGraph.new([1, 2, 4, 8])).values)
    assert_equal((1..127).to_a,
        LinearGraph.new([32, 64], LinearGraph.new([1, 2, 4, 8, 16])).values)
    assert_equal((1..127).to_a,
        LinearGraph.new([64], LinearGraph.new([1, 2, 4, 8, 16, 32])).values)
  end

end
